-- Add_Subtract_Integers.lua
-- Copyright (C) 2010, Texas Instruments Incorporated
-- All rights reserved
--
-- This Nspire app teaches concepts of addition and subtraction by moving plus
-- and minus counters into a counting arena
-- 
-- John Powers  2010-06-18



----------------------------------------------------------------------------- Grid
Grid = class()

Grid.X              = 7
Grid.Y              = 7

Grid.x              = 1/6   -- from left edge
Grid.y              = 0
Grid.w              = 2/3   -- width of window
Grid.h              = 1


function Grid:init()
    -- Create an empty grid
    local grid = {}
    for x = 1, self.X do
        local row = {}
        for y = 1, self.Y do
            row[y] = 0
        end
        grid[x] = row
    end
    self.grid = grid
end


-- Returns true if unit coordinates ux,uy are within the bounds of the grid
function Grid:contains(ux, uy)
    local gx, gy = self.x, self.y
    local gw, gh = self.w, self.h

    return ux >= gx and ux < (gx + gw) and
           uy >= gy and uy < (gy + gh)
end


-- Returns the grid coordinates of a hole in the grid nearest the given
-- unit coordinates.
-- Returns nil, nil if all grid locations are filled.
function Grid:findHole(ux, uy)
    local grid = self.grid
    local mdist = math.huge
    local mgx, mgy = nil, nil
    for gx = 1, self.X do
        for gy = 1, self.Y do
            -- Is the location empty?
            if grid[gx][gy] == 0 then
                -- How far away is location?
                local ugx, ugy = self:toUnitCoords(gx, gy)
                local dist = (ux - ugx)^2 + (uy - ugy)^2
                -- Closer than other locations?
                if dist < mdist then
                    -- Note nearer location
                    mgx = gx
                    mgy = gy
                    mdist = dist
                end
            end
        end
    end
    return mgx, mgy
end


-- Places a counter in the grid at grid coordinates gx,gy
function Grid:place(counter, gx, gy)
    self.grid[gx][gy]   = counter.value

    -- Let counter know which grid point it is on
    counter.gx          = gx
    counter.gy          = gy
end


-- Removes a counter from the grid.
function Grid:clearPlace(gx, gy)
    self.grid[gx][gy] = 0
end


function Grid:clear()
    for gx = 1, self.X do
        for gy = 1, self.Y do
            self.grid[gx][gy] = 0
        end
    end
end


-- Returns the grid location nearest unit coordinates x,y
function Grid:toGridCoords(ux, uy)
    local gx, gy = self.x, self.y
    local gw, gh = self.w, self.h
    local cgx, cgy

    if ux < gx then
        cgx = 1
    elseif ux >= (gx + gw) then
        cgx = self.X
    else
        cgx = math.floor((ux - gx) * self.X / gw) + 1
    end

    if uy < gy then
        cgy = 1
    elseif uy >= (gy + gh) then
        cgy = self.Y
    else
        cgy = math.floor((uy - gy) * self.Y / gh) + 1
    end

    return cgx, cgy
end



-- Returns coordinates of x,y grid location in the unit square.
function Grid:toUnitCoords(x, y)
    local gx, gy = self.x, self.y
    local gw, gh = self.w, self.h
    local ux = (x - 0.5) * gw / self.X + gx
    local uy = (y - 0.5) * gh / self.Y + gy
    return ux, uy
end


-- Returns the total value of counters on the grid.
function Grid:total()
    local total = 0
    for _, row in ipairs(self.grid) do
        for _, value in ipairs(row) do
            total = total + value
        end
    end
    return total
end


function Grid:paint(gc)
    local wx, wy = util.toWindowCoords(self.x, self.y)
    local ww, wh = util.toWindowCoords(self.w, self.h)

    gc:setColorRGB(128, 128, 128)
    gc:fillRect(wx, wy, ww, wh)
end






----------------------------------------------------------------------------- Counter
Counter = class()

Counter.ADD         = 1
Counter.SUBTRACT    = -1

Counter.color = {
    [Counter.ADD]       = {0, 0, 0},        -- black
    [Counter.SUBTRACT]  = {255, 255, 255},  -- white
}


function Counter:init(value, ux, uy)
    self.value          = value
    self.x              = ux
    self.y              = uy
    self.radius         = 0.04 -- fraction of window width
    self.homex          = ux
    self.homey          = uy
end


-- Sets the unit coordinates ux,uy of the counter
function Counter:setxy(ux, uy)
    self.x      = ux
    self.y      = uy
end


-- Returns true if the counter contains the x,y unit coordinates
function Counter:contains(ux, uy)
    local dist = math.sqrt((ux - self.x)^2 + (uy - self.y)^2)
    return dist <= self.radius
end


-- Is counter on a grid location?
function Counter:onGrid()
    return self.gx ~= nil
end


-- Returns grid coordinates of counter
function Counter:gridCoords()
    return self.gx, self.gy
end


-- Clears counter of grid location
function Counter:clearGridCoords()
    self.gx = nil
    self.gy = nil
end


-- Handle picking a counter up on mouseDown
function Counter:pickup(wx, wy)
    -- Is counter on the grid?
    if self:onGrid() then
        grid:clearPlace(self:gridCoords())
        self:clearGridCoords()
    end
end


-- Handle dropping a counter on mouseUp
function Counter:drop(wx, wy)
    local ux, uy    = util.toUnitCoords(wx, wy)

    -- Where did the counter drop?
    if grid:contains(ux, uy) then
        -- Counter dropped in the grid
        local gx, gy = grid:findHole(ux, uy)
        if gx ~= nil then
            grid:place(self, gx, gy)
            animation.zipto(self, grid:toUnitCoords(gx, gy))
        else
            -- No room in grid, send counter back to well
            animation.zipto(self, self.homex, self.homey)
        end
    else
       -- Move counter back to well
       animation.zipto(self, self.homex, self.homey)
    end
end


function Counter:paint(gc, is_selected)
    local wx, wy = util.toWindowCoords(self.x, self.y)

    local fillcolor = Counter.color[self.value]

    gc:setColorRGB(unpack(fillcolor))

    -- Fill counter
    local r = self.radius * platform.window:width()
    local d = 2*r
    gc:fillArc(wx - r, wy - r, d, d, 0, 360)

    -- Outline counter
    if is_selected then
        gc:setPen("medium", "smooth")
    else
        gc:setPen("thin", "smooth")
    end
    gc:setColorRGB(0, 0, 0)
    gc:drawArc(wx - r, wy - r, d, d, 0, 360)
end






----------------------------------------------------------------------------- Zero Pair

ZeroPair = class()

function ZeroPair:init(ux, uy)
    self:populate(ux, uy)
    local rad   = self.add.radius
    self.x      = ux - rad
    self.y      = uy - rad
    self.w      = 2*rad
    self.h      = 4*rad
    self.homex  = ux
    self.homey  = uy
end


function ZeroPair:returnHome()
    self.x      = self.homex - self.radius
    self.y      = self.homey - self.radius
end


-- Populate zero pair with counters
function ZeroPair:populate(ux, uy)
    local add   = Counter(Counter.ADD, ux, uy)
    self.add    = add

    local rad   = add.radius
    self.radius = rad
    local sub   = Counter(Counter.SUBTRACT, ux, uy+2*rad)
    self.sub    = sub
    sub.homex   = sub.x
    sub.homey   = sub.y
end


-- Sets the unit coordinates ux,uy of the zero pair
function ZeroPair:setxy(ux, uy)
    self.x      = ux
    self.y      = uy
    self.add:setxy(ux + self.radius, uy + self.radius)
    self.sub:setxy(ux + self.radius, uy + 3*self.radius)
end


function ZeroPair:contains(ux, uy)
    return ux >= self.x and ux < (self.x + self.w) and
           uy >= self.y and uy < (self.y + self.h)
end


function ZeroPair:clearGridCoords()
    self.add:clearGridCoords()
    self.sub:clearGridCoords()
end


-- Handle picking up a zero pair on mouseDown
function ZeroPair:pickup(wx, wy)
end


-- Handle dropping a zero pair on mouseDown
function ZeroPair:drop(wx, wy)
    -- Let the well have my counters
    well:addCounter(self.add)
    well:addCounter(self.sub)

    -- Drop my counters
    self.add:drop(wx, wy)
    self.sub:drop(wx, wy)

    -- Return zero pair frame back to well
    self:returnHome()

    -- Populate myself with new counters
    self:populate(self.homex, self.homey)
end


function ZeroPair:paint(gc)
    self.add:paint(gc)
    self.sub:paint(gc)
end






----------------------------------------------------------------------------- Counter well

Well = class()

Well.X          = 11/12       -- unit square coordinate system
Well.ADDY       = 0.15
Well.SUBY       = 0.45
Well.ZERO_PAIRY = 0.75


Well.SIGNY_OFFSET = 0.1
Well.SIGNW  = 0.04
Well.SIGNH  = 0.01

function Well:init()
    local x         = Well.X
    local y         = Well.ADDY

    local counters  = {}      -- keep track of all counters

    -- Well of adders
    for i = 1, Grid.X * Grid.Y do
        table.insert(counters, Counter(Counter.ADD, x, y))
    end

    -- Well of subtracters
    y = Well.SUBY
    for i = 1, Grid.X * Grid.Y do
        table.insert(counters, Counter(Counter.SUBTRACT, x, y))
    end

    -- Well of zero pairs
    local zeropairs = {}
    y = Well.ZERO_PAIRY
    for i = 1, 2 do
        table.insert(zeropairs, ZeroPair(x, y))
    end
    self.zptitle = Text("Zero Pairs", x, Well.ZERO_PAIRY + 0.2, "r")

    self.counters = counters
    self.zeropairs = zeropairs
end


-- Adds a counter to the well
function Well:addCounter(counter)
    if counter.value > 0 then
        counter.homey = Well.ADDY
    else
        counter.homey = Well.SUBY
    end
    table.insert(self.counters, counter)
end


-- Finds an object at the given window coordinates. The object could be a
-- zero pair or a counter.
-- Returns nil if no object is at those coordinates.
function Well:findObject(wx, wy)
    local ux, uy = util.toUnitCoords(wx, wy)

    for _, zp in ipairs(self.zeropairs) do
        if zp:contains(ux, uy) then
            return zp
        end
    end
        
    for _, counter in ipairs(self.counters) do
        if counter:contains(ux, uy) then
            return counter
        end
    end
    return nil
end


-- Rearrange counter Z-order so it is drawn over all other counters.
function Well:bringToFront(counter)
    for i, c in ipairs(self.counters) do
        if c == counter then
            table.remove(self.counters, i)
            table.insert(self.counters, c)
            return
        end
    end
end


function Well:paint(gc)
    local wx, wy = util.toWindowCoords(5/6, 0)
    local ww, wh = util.toWindowCoords(1/6, 1)
    gc:setColorRGB(0xD0, 0xD0, 0xD0)    -- light gray
    gc:fillRect(wx, wy, ww, wh)

    -- Plus sign
    wx, wy = util.toWindowCoords(Well.X - Well.SIGNW/2, Well.ADDY + Well.SIGNY_OFFSET - Well.SIGNH/2)
    ww, wh = util.toWindowCoords(Well.SIGNW, Well.SIGNH)
    gc:setColorRGB(0, 0, 0)
    gc:fillRect(wx, wy, ww, wh)
    gc:fillRect(wx+(ww-wh)/2, wy-(ww-wh)/2, wh, ww)

    -- Minus sign
    wx, wy = util.toWindowCoords(Well.X - Well.SIGNW/2, Well.SUBY + Well.SIGNY_OFFSET - Well.SIGNH/2)
    ww, wh = util.toWindowCoords(Well.SIGNW, Well.SIGNH)
    gc:fillRect(wx, wy, ww, wh)

    for _, counter in ipairs(self.counters) do
        counter:paint(gc)
    end

    for _, zp in ipairs(self.zeropairs) do
        zp:paint(gc)
    end

    self.zptitle:paint(gc)
end





----------------------------------------------------------------------------- Button

Button = class()

-- Creates a button centered at unit coordinates ux,uy
function Button:init(text, ux, uy, exec)
    self.text   = text
    self.ux     = ux
    self.uy     = uy

    local gc    = platform.gc()
    gc:begin()
    gc:setFont("sansserif", "r", 9)
    self.bw     = gc:getStringWidth(text) + 4   -- two pixel x-pad
    self.bh     = gc:getStringHeight(text) + 4  -- two pixel y-pad
    gc:finish()

    -- Execution function when button is pushed
    self.exec   = exec
end


function Button:pressed()
    self.exec()
end


-- Returns window coordinates and width,height of button
function Button:bounds()
    local bx, by    = util.toWindowCoords(self.ux, self.uy)
    bx              = bx - self.bw/2
    by              = by - self.bh/2
    return bx, by, self.bw, self.bh
end

    
-- Returns true if button bounds window coordinates wx,wy
function Button:contains(wx, wy)
    local bx, by, bw, bh = self:bounds()

    return wx >= bx and wx < (bx + bw) and
           wy >= by and wy < (by + bh)
end


function Button:paint(gc)
    local bx, by, bw, bh = self:bounds()

    -- White rectangle
    gc:setColorRGB(255, 255, 255)
    gc:fillRect(bx, by, bw, bh)

    -- Black border
    gc:setColorRGB(0, 0, 0)
    gc:drawRect(bx, by, bw, bh)

    -- Text in button
    gc:setFont("sansserif", "r", 9)
    gc:drawString(self.text, bx+2, by+2, "top")
end






----------------------------------------------------------------------------- Text

Text = class()

-- Create a text field centered at unit coordinates ux,uy
function Text:init(text, ux, uy, style)
    self.text   = text
    self.ux     = ux
    self.uy     = uy
    self.style  = style
end


-- Calculate dimensions of text field in window coordinates
function Text:calcDimensions(gc)
    if #self.text > 0 then
        local style = self.style
        local size  = 9
        if style == "b" or style == "bi" then
            size = 11
        end
        gc:setFont("sansserif", style, size)
        self.ww     = gc:getStringWidth(self.text)
        self.wh     = gc:getStringHeight(self.text)

        local tx, ty = util.toWindowCoords(self.ux, self.uy)
        self.wx     = tx - self.ww/2
        self.wy     = ty - self.wh/2
    end
end


-- Updates the text of the field
function Text:setText(text)
    self.text   = text
end


function Text:paint(gc)
    self:calcDimensions(gc)
    gc:setColorRGB(0, 0, 0)
    gc:drawString(self.text, self.wx, self.wy, "top")
end
        





----------------------------------------------------------------------------- Tally

Tally = class()


-- Dimensions of tally panel (unit coordinates)

Tally.x = 0
Tally.y = 0
Tally.w = 1/6
Tally.h = 1

Tally.midx = Tally.x + Tally.w/2

Tally.buttony   = 0.9
Tally.titley    = 0.25
Tally.totaly    = 0.35


function Tally:init()
    self.button     = Button("RESET", self.midx, self.buttony, Tally.reset)
    self.title      = Text("Total", self.midx, self.titley, "r")
    self.total      = Text("", self.midx, self.totaly, "b")
end


function Tally:checkButtonPressed(wx, wy)
    if self.button:contains(wx, wy) then
        self.button:pressed()
    end
end


function Tally:reset()
    for _, counter in ipairs(well.counters) do
        counter:clearGridCoords()
        animation.zipto(counter, counter.homex, counter.homey)
    end
    for _, zp in ipairs(well.zeropairs) do
        local a, s = zp.add, zp.sub
        zp:clearGridCoords()
        animation.zipto(a, a.homex, a.homey)
        animation.zipto(s, s.homex, s.homey)
    end
    grid:clear()
end


function Tally:paint(gc)
    self.button:paint(gc)
    self.title:paint(gc)

    self.total:setText(tostring(grid:total()))
    self.total:paint(gc)
end





----------------------------------------------------------------------------- Animation

animation = {}

animation.list = {}


-- Causes an object to zip to unit coordinates ux,uy
function animation.zipto(obj, ux, uy)
    -- Add object to list of animated objects
    table.insert(animation.list, function() return animation.zipstep(obj, ux, uy) end)

    -- Do we need to start the animation timer?
    if #animation.list == 1 then
        timer.start(0.1)
    end
end


-- Moves an object along the line from its location to its destination
function animation.zipstep(obj, ux, uy)
    local ox, oy = obj.x, obj.y
    local dx = ux - ox
    local dy = uy - oy
    local dist = math.sqrt(dx*dx + dy*dy)
    local done = false

    local step
    if dist > 0.1 then
        step = 0.1
    elseif dist > 0.01 then
        step = 0.01
    else
        step = dist
        done = true
    end

    local xstep, ystep
    if dx == 0 then
        xstep = 0
        if dy >= 0 then
            ystep = step
        else
            ystep = -step
        end
    else
        local aa = dy / dx
        local denom = 1 / math.sqrt(1 + aa*aa)
        xstep = step * denom            -- step * cos(atan(dy/dx))
        ystep = step * aa * denom       -- step * sin(atan(dy/dx))
        if dx < 0 then
            xstep = -xstep
            ystep = -ystep
        end
    end

    obj.x = ox + xstep
    obj.y = oy + ystep
    return done                         -- done if obj arrived at destination
end


function animation.step()
    -- Step each object in animation list
    local newlist = {}
    for _, afunc in ipairs(animation.list) do
        if not afunc() then
            table.insert(newlist, afunc)
        end
    end

    -- All animations complete?
    if #newlist == 0 then
        timer.stop()
    end

    -- Keep new list of animated objects
    animation.list = newlist

    platform.window:invalidate()
end








----------------------------------------------------------------------------- Utilities

util = {}

-- Converts unit square coordinates to window coordinates
function util.toWindowCoords(ux, uy)
    local window = platform.window
    return ux * window:width(), uy * window:height()
end


-- Converts window coordinates to unit square coordinates
function util.toUnitCoords(wx, wy)
    local window = platform.window
    return wx / window:width(), wy / window:height()
end


-- Returns a function that tracks mouse movements. Obj is updated to
-- follow the mouse.
function util.mouseTracker(obj, wx, wy)
    local oldx, oldy = util.toUnitCoords(wx, wy)
    return function(wx, wy)
        local ux, uy = util.toUnitCoords(wx, wy)
        obj:setxy(obj.x + ux - oldx, obj.y + uy - oldy)
        oldx = ux
        oldy = uy
        platform.window:invalidate()
    end
end






----------------------------------------------------------------------------- Event handlers


function on.mouseDown(wx, wy)
    -- Look for a counter to grab
    local object = well:findObject(wx, wy)

    -- Counter follows mouse movements
    if object ~= nil then
        well:bringToFront(object)

        -- Pick up object
        object:pickup(wx, wy)

        -- Move object around with mouse pointer
        on.mouseMove = util.mouseTracker(object, wx, wy)

        -- Drop object
        on.mouseUp   = function(wx, wy)
            on.mouseMove = nil
            on.mouseUp   = nil
            object:drop(wx, wy)
        end
    else
        tally:checkButtonPressed(wx, wy)
    end
        

end


function on.timer()
    animation.step()
end


function on.paint(gc)
    grid:paint(gc)
    well:paint(gc)
    tally:paint(gc)
end






----------------------------------------------------------------------------- Initialization

grid = Grid()               -- create the grid
well = Well()               -- create the counter well
tally = Tally()             -- create tally panel
