-- Copyright (C) 2010, Texas Instruments Incorporated
-- All rights reserved
--
-- 3D rendering
-- 3daccel.lua
-- 
-- Draw an object in 3D. Use the 3-axis Vernier sensor to orient the object view.
-- 
-- John Powers  2010-08-11
-- 
-- John Powers  2010-10-14
-- Use images of spheres to draw the atoms of the molecule


vector = {

    unit = function(v)
        local x, y, z = unpack(v)
        local dist = math.sqrt(x*x + y*y + z*z)
        if dist == 0 then
            return {0, 0, 0}
        end
        return {x/dist, y/dist, z/dist}
    end,

    dotProduct = function(a, b)
        local sum = 0
        for i = 1, #a do
            sum = sum + a[i] * b[i]
        end
        return sum
    end,

    crossProduct = function(a, b)
        return {a[2]*b[3] - a[3]*b[2], a[3]*b[1] - a[1]*b[3], a[1]*b[2] - a[2]*b[1]}
    end,


    -- Returns a rotation matrix given the 3-axis accelerometer vector.
    computeRotationMatrix = function(ax, ay, az)
        local uav = vector.unit({ax, ay, az})
    
        -- Cosine of deflection angle from -z axis
        local cd = vector.dotProduct({0, 0, -1}, uav)
        
        -- Sine of deflection angle from -z axis
        local sd = math.sqrt(1 - cd*cd)
    
        -- Normal vector of rotation
        local nvr = vector.crossProduct({0, 0, -1}, uav)
        
        -- Components of unit normal vector of rotation
        local ux, uy, uz = unpack(vector.unit(nvr))
    
        local ux2 = ux*ux
        local uy2 = uy*uy
        local uz2 = uz*uz
    
        return {
            {ux2 + (1 - ux2)*cd,       ux*uy*(1 - cd) - uz*sd,   ux*uz*(1 - cd) + uy*sd},
            {ux*uy*(1 - cd) + uz*sd,   uy2 + (1 - uy2)*cd,       uy*uz*(1 - cd) - ux*sd},
            {ux*uz*(1 - cd) - uy*sd,   uy*uz*(1 - cd) + ux*sd,   uz2 + (1 - uz2)*cd    }
        }
    end,


    -- Returns a vector transformed by a rotation matrix
    transform = function(rm, v)
        local x, y, z = unpack(v)
        return {rm[1][1]*x + rm[1][2]*y + rm[1][3]*z,
                rm[2][1]*x + rm[2][2]*y + rm[2][3]*z,
                rm[3][1]*x + rm[3][2]*y + rm[3][3]*z
               }
    end,

}





--------------------------------------------------------------------------- VisualObject
-- A VisualObject is an object that can be drawn on the screen.
-- This is the base class for other objects in 3D space.
VisualObject = class()


function VisualObject:init(options)
    self:setOptions(options)
end


function VisualObject:setOptions(options)
    for key, value in pairs(options) do
        if type(key) == "string" then
            self[key] = value
        end
    end
end


function VisualObject:resize(w, h)
    -- Implemented in subclasses
end
    





--------------------------------------------------------------------------- Sphere

Sphere = class(VisualObject)


function Sphere:centerOfGravity()
    return {self.x, self.y, self.z}
end


-- Sets the coordinates of the Sphere normalized to the center of gravity
-- of the Constellation.
function Sphere:offsetByCenterOfGravity(cog)
    local cx, cy, cz = unpack(cog)
    self.x = self.x - cx
    self.y = self.y - cy
    self.z = self.z - cz
end


function Sphere:transform(rotationMatrix)
    self.tx, self.ty, self.tz = unpack(vector.transform(rotationMatrix, {self.x, self.y, self.z}))
end


function Sphere:resize()
    local d = Layout.windowScale(2*self.radius)
    self.scaledImage = Sphere.images[self.image]:copy(d, d)
end


function Sphere:paint(gc)
    local r = self.radius
    local wx, wy = Layout.windowCoords(self.tx - r, self.ty + r)
    gc:drawImage(self.scaledImage, wx, wy)
    --[[
    local d = Layout.windowScale(r + r)
    gc:setColorRGB(unpack(self.color))
    gc:fillArc(wx, wy, d, d, 0, 360)
    ]]
end






--------------------------------------------------------------------------- Line

Line = class(VisualObject)


function Line:centerOfGravity()
    return {(self.x1 + self.x2)/2, (self.y1 + self.y2)/2, (self.z1 + self.z2)/2}
end


function Line:offsetByCenterOfGravity(cog)
    local cx, cy, cz = unpack(cog)
    self.x1 = self.x1 - cx
    self.x2 = self.x2 - cx
    self.y1 = self.y1 - cy
    self.y2 = self.y2 - cy
    self.z1 = self.z1 - cz
    self.z2 = self.z2 - cz
end


function Line:transform(rotationMatrix)
    self.tx1, self.ty1, self.tz1 = unpack(vector.transform(rotationMatrix, {self.x1, self.y1, self.z1}))
    self.tx2, self.ty2, self.tz2 = unpack(vector.transform(rotationMatrix, {self.x2, self.y2, self.z2}))
    -- Compute average z depth for Z ordering
    self.tz = (self.tz1 + self.tz2)/2
end


function Line:paint(gc)
    local wx1, wy1 = Layout.windowCoords(self.tx1, self.ty1)
    local wx2, wy2 = Layout.windowCoords(self.tx2, self.ty2)
    gc:setColorRGB(unpack(self.color))
    gc:setPen("thick", "smooth")
    gc:drawLine(wx1, wy1, wx2, wy2)
end






--------------------------------------------------------------------------- Constellation
-- A Constellation is a group of objects arranged in 3D space.
Constellation = class(VisualObject)


function Constellation:init(members)
    VisualObject.init(self, members)
    self.rotationMatrix = {{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}

    self.objectList = {}

    for _, obj in ipairs(members) do
        local objType = obj[1]
        table.insert(self.objectList, objType(obj))
    end

    self:offsetByCenterOfGravity()

end


function Constellation:transform()
    local rm = self.rotationMatrix

    for _, obj in ipairs(self.objectList) do
        obj:transform(rm)
    end
end


function Constellation:resize(w, h)
    for _, obj in ipairs(self.objectList) do
        obj:resize(w, h)
    end
end


function Constellation:paint(gc)
    self:paintTitle(gc)
    for _, obj in ipairs(self:sortZOrder()) do
        obj:paint(gc)
    end
end


function Constellation:paintTitle(gc)
    gc:setFont("sansserif", "b", 18)
    gc:setColorRGB(unpack(Color.black))
    local titleWidth = gc:getStringWidth(self.name)
    local x = Layout.center - titleWidth/2
    gc:drawString(self.name, x, 10, "top")
end


-- Returns the center of gravity of a Constellation as a table of {x, y, z}
function Constellation:centerOfGravity()
    if self.cogx == nil then
        local minx = math.huge
        local maxx = -math.huge
        local miny = math.huge
        local maxy = -math.huge
        local minz = math.huge
        local maxz = -math.huge

        for _, obj in ipairs(self.objectList) do
            local cx, cy, cz = unpack(obj:centerOfGravity())
            minx = math.min(cx, minx)
            maxx = math.max(cx, maxx)
            miny = math.min(cy, miny)
            maxy = math.max(cy, maxy)
            minz = math.min(cz, minz)
            maxz = math.max(cz, maxz)
        end

        self.cogx = (minx + maxx)/2
        self.cogy = (miny + maxy)/2
        self.cogz = (minz + maxz)/2
    end
    return {self.cogx, self.cogy, self.cogz}
end


-- Adjusts the coordinates of objects in the Constellation so they are arranged
-- around the center of gravity.
function Constellation:offsetByCenterOfGravity()
    local cog = self:centerOfGravity()
    for _, obj in ipairs(self.objectList) do
        obj:offsetByCenterOfGravity(cog)
    end
end


-- Returns a list of objects ordered by their z coordinates, back to front.
function Constellation:sortZOrder()
    local objs = {}
    for _, obj in ipairs(self.objectList) do
        table.insert(objs, obj)
    end

    -- The z coordinate is in position 3 of each object
    table.sort(objs, function(a, b) return a.tz < b.tz end)
    return objs
end


function Constellation:setRotationMatrix(x, y, z)
    self.rotationMatrix = vector.computeRotationMatrix(x, y, z)
    self:transform()
end



--------------------------------------------------------------------------- Color
Color = {
    black = {   0,    0,    0},
    red   = {0xff,    0,    0},
    green = {   0, 0xff,    0},
    blue  = {   0,    0, 0xff},
    white = {0xff, 0xff, 0xff},
}


--------------------------------------------------------------------------- Layout

Layout = {}

function Layout.windowCoords(x, y)
    local s = Layout.scaleFactor
    local c = Layout.center
    return x*s + c, c - y*s
end


function Layout.windowScale(x)
    return x*Layout.scaleFactor
end


function Layout.resize(w, h)
    local length = math.min(w, h)
    Layout.scaleFactor = length/10
    Layout.center = length/2
end


--[[
Water = Constellation { name="Water",
    {Sphere,  name="H",  color=Color.blue,  radius=1,   x=-1.25, y=4.8, z=0},
    {Line,               color=Color.black,             x1=-0.25, y1=0.97, z1=0, x2=-1, y2=3.87, z2=0},
    {Sphere,  name="O",  color=Color.red,   radius=1,   x=0,  y=0, z=0},
    {Line,               color=Color.black,             x1=1, y1=0, z1=0, x2=4, y2=0, z2=0},
    {Sphere,  name="H",  color=Color.blue,  radius=1,   x=5, y=0, z=0},
}
]]

Ethane = Constellation { name="Ethane",
    {Sphere,  name="C",  image="green",  radius=1,   x=-2, y=0, z=0},
    {Line,               color=Color.black,                                     x1=-2.5, y1=0, z1=0.87, x2=-3.75, y2=0, z2=3.03},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=-4, y=0, z=3.46},
    {Line,               color=Color.black,                                     x1=-2.5, y1=-0.87, z1=-0.5, x2=-3.75, y2=-3.03, z2=-1.75},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=-4, y=3.46, z=-2},
    {Line,               color=Color.black,                                     x1=-2.5, y1=0.87, z1=-0.5, x2=-3.75, y2=3.03,  z2=-1.75},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=-4, y=-3.46, z=-2},
    {Sphere,  name="C",  image="green",  radius=1,   x=2, y=0, z=0},
    {Line,               color=Color.black,                                     x1=2.5, y1=0, z1=0.866, x2=3.75, y2=0, z2=3.03},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=4, y=0, z=3.46},
    {Line,               color=Color.black,                                     x1=2.5, y1=-0.87, z1=-0.5, x2=3.75, y2=-3.03, z2=-1.75},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=4, y=3.46, z=-2},
    {Line,               color=Color.black,                                     x1=2.5, y1=0.86, z1=-0.5, x2=3.75, y2=3.03,  z2=-1.75},
    {Sphere,  name="H",  image="blue",   radius=0.5, x=4, y=-3.46, z=-2},

    {Line,               color=Color.black,              x1=-1, y1=0, z1=0, x2=1, y2=0, z2=0},  -- connect carbons
}


C = Ethane


--------------------------------------------------------------------------- Event handlers

function on.paint(gc)
    local x = var.recall(AccelX) or 0
    local y = var.recall(AccelY) or 0
    local z = var.recall(AccelZ) or -9.8
    C:setRotationMatrix(x, -y, z)

    C:paint(gc)
end


function on.resize(w, h)
    Layout.resize(w, h)
    C:resize(w, h)
end

AccelX = "meter.acceleration"
AccelY = "meter.acceleration2"
AccelZ = "meter.acceleration3"

function on.create()
    timer.start(1/4)
end

function on.timer()
    platform.window:invalidate()
end


---------------------------------------------------------------------------- Images of spheres
-- These are at the bottom of the file so the program is easier to read

Sphere.images = {

    blue = image.new("k\000\000\000l\000\000\000\000\000\000\000\214\000\000\000\016\000\001\000\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\223{\255{\255\127\255{\255{\255{\255w\255\247\223\247\255{\255\127\255\127\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255{\255w\255\127\255\127\255\127\255{\255w\255w\255{\255\247\255{\255w\255{\255\127\255\127\221\127\221\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255{\255{\255{\255{\255{\255\127\255\127\255{\255w\156\2359\223\247\214\214\214\247\214\247\214\214\210\214\210\247\2149\223{\231\222\243\255{\255{\255\127\255\127\255{\255\127\255{\255{\255{\255{\255{\255{\255{\255{\254\127\255\127\254\127\253\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254{\254{\253{\253{\253\127\254\127\254\127\222{\255\127\255\127\255{\255\247\127\227\156\194\182\165S\149\172\132\172\136\204\136\204\136\172\136\171\132\171\132\171\132\204\136\204\136\172\136\204\136\204\136\205\136\205\140\205\140\178\161W\182\127\223\255\243\255w\255\247\255w\255\127\255\127\255\127\255\127\255\127\255\127\255w\255\247\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254{\254{\254{\253{\254\127\254\127\255\127\255\127Z\2314\198\012\157\138\136\171\132\015\141\149\1619\178\221\206\031\215\159\227\255\239\255\243\255\243\255\243\255\243\255\243\255\243\255\239\223\235\127\223\030\211\154\1948\186\178\161\013\141\169\128\168\132\010\149\017\186Z\231\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\254\127\255\127\255\127\255\127\255{\255{\255{\223\127\223\127\255\127\255\127\255w>\223\243\173\202\136\235\140\143\161\183\194\158\231\255\247\255\251\255\251\255\255\255\255\255\255\255\255\222\255\222\255\222\255\254\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\223\251\255\251\255\251]\2274\186N\157\235\140\169\132\210\169\028\215\255\247\255{\255{\255\127\255\127\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\254\127\255\127\255\127\255\127\255{\255{\255{\223\127\255\127\223{V\202\234\148\202\140\020\182\127\223\255\239\255\243\255\243\255\247\255\247\255\247\253\247\253\251\222\255\222\255\254\255\255\255\254\255\254\255\222\255\222\255\222\255\222\255\222\255\222\255\222\255\222\255\222\255\222\255\255\251\255\251\255\247\255\247\255\247\255\243\255\235=\215\209\165\201\136\201\140T\194\223\247\255\127\254\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255\127\222{\255\127\255\127\254\127\255\127\255\127\222\127\255\127\255\127\255{\255{\223\243\255{\223\235\213\173\171\136\178\173~\231\255\255\255\255\255\255\221\255\221\255\253\251\253\251\254\247\255\247\255\243\255\243\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\247\255\243\255\243\255\247\223\247\223\251\223\251\222\255\222\255\255\255\222\251\248\222m\169\134\132\209\169\191\227\255\243\255\127\255\127\223{\255\127\255\127\223{\255\127\255\127\255\127\223{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255\127\254\127\255\127\255\127\254\127\222\127\255\127\255\127\255{\255w\255{\255{\142\169\205\140\154\194\255\243\255\251\223\243\191\247\223\251\222\251\221\255\221\255\221\251\253\251\254\247\254\243\255\243\255\243\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\223\247\255\243\255\243\223\243\223\247\223\251\222\251\222\255\222\255\156\247\255\255\255\255\255\251\255\243\019\178\169\132o\157\190\243\255\127\223{\255\127\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\255\127\255\127\255\127\255{\222w\255\127\255\127\255{\255{\255{\255{\208\177\134\136\248\214\252\255\251\251\252\247\253\247\254\247\255\243\223\239\223\239\223\239\223\243\191\247\191\247\191\251\191\255\159\255\159\255\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\191\247\187\255\187\255\188\255\189\251\222\247\223\247\223\243\223\243\255\247~\231\191\235\191\239\191\239\255\251\255\2513\194\134\140\240\181\255{\255w\255{\191\243\255\127\255{\222w\255\127\255\127\255\127\255\127\255\127\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\255\127\255\127\255\127\189\243\255\127\255\127\223\243\215\214\166\140\150\202\255\251\255\247\252\251\219\247\219\243\220\243\222\243\222\239\223\239\255\243\223\239\191\243\191\243\191\247\191\251\159\251\159\255\159\255\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\190\247\187\255\187\255\188\251\189\251\190\247\191\243\223\243\223\239\223\239\255\247\255\243\223\243\223\243\191\243\190\243\255\255\255\251\174\173\166\140\183\210\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255{\255{\223\243\255{\255{\255\247m\165\175\173\223\243\255\251\191\243\158\235\191\239\223\243\223\243\190\239\190\243\190\247\189\247\156\243\189\247\189\247\189\247\189\247\190\243\190\243\190\243\190\239\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\190\239\190\243\190\243\190\243\189\247\189\247\189\247\189\247\156\243\222\251\190\247\157\239\159\239\223\239\191\239\191\235\223\243\255\243\158\235\010\153\009\149\255{\255{\223w\255{\223{\255\127\255\127\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\255\127\255\127\255\127\255{\255{\255w\026\215\167\140;\223\255\243\191\239\158\235\190\239\223\243\191\235\159\235\191\239\223\247\223\247\157\243\157\247\222\251\188\247\188\247\189\247\189\243\189\243\190\243\190\239\190\239\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\222\235\190\239\190\243\190\243\189\243\189\243\189\247\157\247\157\247\189\247|\243\158\243\223\247\223\243\191\239\191\235\191\239\158\231\255\243\191\239\255\251u\198\167\140\183\206\255{\255\127\255{\222w\255\127\255\127\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\222\127\255{\255{\255\247\255w\241\173,\153\255\247\159\231\223\239\158\235\190\239\255\247|\235\157\243\190\247\190\247\157\243|\235}\235\157\239\157\235\158\235\158\235\190\235\190\235\190\235\190\235\189\235\189\235\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\157\243\158\239\158\239\158\239\158\239\158\235\158\235\158\239\158\239\223\243\158\239\158\239\158\243}\239\157\243\158\243}\239\159\231\191\239\159\235\191\235\255\247\092\227\199\144\175\173\223\247\255\127\255{\255\127\255\127\190w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\255\127\255\127\255\127\255{\168\1364\186\255\247~\227\191\235\158\231\191\235\158\235\092\227\190\243\157\243|\239|\239\157\243\190\243\158\239\158\239\158\239\158\235\158\235\158\235\158\235\157\235\157\235\157\235\157\235\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\239\157\235\158\235\158\235\158\235\158\235\158\235\158\235\158\235}\235\158\239\158\239}\239\157\239\157\243}\239\157\243\191\235~\231\159\235\159\235\158\231\223\239\255\247*\157\009\153\223\247\255\127\223{\255\127\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255\127\189{\254\127\254\127\255\247\255{\223\243\255{\223\243\168\140\183\206\191\239\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\157\235\156\239\156\239\156\239\157\235\157\231\159\231\191\235\223\239\176\165\234\144\191\235\255\247\255{\255{\255\127\255\127\255\127\255\127\222\127\255\127\255\127\254\127\255\127\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\223w\255\127\255\127\255\127\255\127\254\127\220\255\255{\255{\255{\159\235\200\140\249\214\191\239\158\235}\231}\231}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\231}\231}\231}\231}\231}\231}\231}\231}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235}\235{\235{\239|\235|\231}\227}\227\158\227\159\231\255\239\242\177\167\136\092\227\255{\255\127\255\127\222{\223{\255\127\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222\127\255\127\189\247\222{\255\127\255\127\255\247\223\243\159\235\200\140\183\206\191\239\092\227\092\227}\231}\231}\231}\231|\231|\231|\231|\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231}\231|\231|\235|\235}\231}\227}\227}\227~\227}\227\191\235\209\173\167\140\026\219\255{\255{\255\127\255{\255{\254{\222\127\255\127\254\127\221{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222w\255{\223\239\134\132U\194\223\243<\223\158\235\092\223\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227[\231\092\231|\231}\231}\227}\227~\227~\227\159\231\092\223\191\235\176\173\200\144\249\214\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\221{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\188\255\254\127\254\127\255\127\255{\255\243\255\247\159\235\201\140\209\177\191\239<\219<\219]\227}\231\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\227\092\223\092\223\092\227\092\227\092\227\092\227\092\227]\227\092\223~\231]\223~\227,\153\011\149^\223\255\247\255\247\255{\255\127\255\127\222\127\255\127\255\127\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\221\127\254\127\221\127\188\255\255\127\190\243\255\247\255\243\011\149,\153~\231]\223<\223\092\223\026\215<\223<\223<\223\092\223\092\223;\223;\223;\223;\223[\223[\223[\223[\223[\223[\223[\223[\223;\223;\223;\223;\223;\223;\223;\223;\223[\223[\223[\223[\223[\223[\223[\223[\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223;\223\092\223[\223;\223;\223;\223;\223[\223\092\223\092\223\027\215]\223<\219]\223\234\144\168\136\223\235\255{\255{\222w\255\127\254\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255\127\255\127\253\127\221\127\221\255\255\127\190\243\255w\255\239-\141\168\136]\223<\219\027\215<\219]\223]\223<\219<\219<\219<\219<\219;\219;\219;\219;\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219<\219\092\219<\219<\219<\219;\219;\223;\223;\223\092\223]\227\027\215~\223\028\2116\182p\153\235\136\255\239\255{\255\247\255\127\255\127\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255{\255\127\255\127\253\127\253\127\254\127\189\127\255\127\255\247\145\153\013\1374\186<\219\027\215<\219\027\215<\219\250\210<\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219\027\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\219;\215;\215;\215\027\219;\219;\219\026\219\026\219;\219\027\215\027\215<\215=\215^\219O\149.\145m\157\255\243\255{\223w\255\127\222\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\254\127\255\127\223w\255w\152\194\014\133/\141=\219<\215\027\215\027\211\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215;\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\028\211\028\211\028\211\027\215\027\215\026\215\026\215\026\215\026\215<\219\027\215=\215\029\211^\211\252\198\013\137-\145\242\169\255\243\255w\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\255\127]\223\202\136P\141X\178<\215\028\215\027\211\027\211\027\211\027\211\027\215\027\215\027\211\027\211\027\215\027\215\027\215\027\215\027\215\027\215\027\211\027\211\027\211\027\211\027\211\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\215\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\215\026\215\026\215\026\215\250\210\027\219\251\210\028\211\028\207=\211\029\203\178\161.\145M\149\250\206\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255\247\255\243N\149.\141/\141?\207\027\211\027\211\251\206\251\206\251\210\027\211\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\027\211\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\251\210\027\207\027\207\027\207\251\210\026\211\250\210\250\210\250\210\250\206\027\215\250\206\251\206\251\206\028\207=\211\153\186/\133q\145\012\137\255\239\255w\223\247\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255w4\182\235\136O\1457\174\030\199\250\206\250\206\250\206\250\206\251\206\251\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\210\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\210\250\210\250\210\249\206\251\210\218\202\251\206\251\206\251\202\029\207\029\207q\145/\133.\145\243\173\255w\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\255\127\255w\223\235\191\223\014\133P\141P\145y\186\029\207\218\202\218\202\250\202\250\202\250\202\250\202\218\206\218\206\218\202\218\206\218\206\218\206\218\206\218\206\218\206\218\206\250\202\250\202\250\202\250\202\250\202\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\206\250\202\250\202\250\202\250\202\250\202\250\206\250\206\250\206\250\206\250\206\250\206\218\206\218\206\218\206\250\206\217\202\218\206\251\206\218\202\250\202\028\211\024\154t\137Q\141\013\137\028\211\255w\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\254\127\253\127\253\127\254\127\255{\255\243\255\243P\145\146\149P\145P\145\253\202\219\198\217\202\217\202\218\202\218\202\218\202\217\202\218\202\218\202\217\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\218\202\218\202\217\202\218\202\250\202\217\202\185\198\219\202\222\1781\129\146\145O\145N\153\255w\190\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\253\127\252\255\252\255\254\127\255{\255\243\253\194\239\128s\141Q\137\245\161\252\206\152\194\185\198\217\198\217\198\217\198\185\198\185\198\185\198\217\198\185\198\185\198\185\202\185\202\185\202\185\202\217\202\217\202\217\198\218\198\218\198\218\198\218\198\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\202\218\198\218\198\218\198\218\198\217\198\216\202\216\202\216\202\185\202\217\198\217\198\217\198\217\198\217\198\185\198\217\198\185\202\185\202\185\202\184\202\184\202\031\183T\129\147\141P\141-\145T\186\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\253\127\253\255\253\255\254\127\255{\255\2390\137s\141s\1411\137\022\170\219\202\185\198\185\198\185\198\185\198\185\198\184\194\184\194\184\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\185\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\217\202\218\198\218\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\217\198\185\198\185\198\185\198\185\198\184\198\184\198\184\198\185\198\185\198\185\198\185\198\185\198\185\198\185\194\217\198\185\198\152\198\185\202\184\202\184\202\031\183\149\137\147\141P\141N\145,\153\255w\255\127\255\127\255\127\255\127\255\127\255\127\223\127\254\127\253\127\254\127\255w\255\243\152\182r\137t\137t\137R\137y\178\185\194\185\194\186\194\186\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\218\194\218\194\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\218\198\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\190\185\190\185\190\185\190\185\190\185\190\185\194\185\194\185\194\185\194\186\194\219\194\244\161r\145r\137s\137\181\1457\166\255\239\255\247\255{\253\127\253\127\254\127\222\127\254\127\220\251\254\127\254\243\255\243o\1451\133\149\141t\137R\137\154\186\185\194\152\190\153\190\153\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\218\194\218\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\218\194\218\194\218\194\218\194\218\194\218\194\218\194\218\194\218\194\218\194\218\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\153\190\153\186X\174P\137\147\141\181\145R\133/\133\255\239\223\239\255\127\253\127\253\127\254\127\254\127\254\127\255\127\255\127\255\243\250\202.\133r\137\181\145t\1371\133\186\186\152\190\152\190\153\186\153\186\153\186\153\190\153\190\153\190\152\190\152\190\153\190\153\190\153\190\153\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\153\190\153\190\152\190\153\190\152\190\152\190\152\190\153\190\153\190\153\190\153\190\153\190\153\190\153\186W\174P\137\148\145s\141s\1370\137\153\190\255\247\255{\254\127\221\127\255\127\255\127\255\127\255\127\255{\255\247M\149\146\149\180\145t\137\148\137R\137\154\182w\182\184\190x\186x\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\190\152\190\152\190\152\190\153\190\152\190\152\190\152\190\152\190\152\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\194\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\152\190\152\190\152\190\152\190\152\190\152\190\152\190\152\190\152\190\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\186\190\245\161r\141\147\145s\137\148\141\213\153\012\137\255\247\222\243\254\127\254\127\254\127\255\127\255\127\222\247\255{~\223-\141r\145\148\141S\133\149\141\147\145\154\182U\178\151\186x\186x\186x\186x\186x\186x\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\190\152\190\152\190\152\190\153\190\153\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\153\190\153\190\152\190\152\190\152\190\152\190\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186x\186x\186x\186x\186x\186x\186\153\182\022\166r\141\148\141\181\149s\137r\145.\141\027\211\255\243\255\127\255\127\254\127\255\127\255\127\255{\255{\241\173p\153Q\141\149\145t\137\148\141r\141x\174V\182\151\186x\182x\182x\182x\182x\182x\186x\186x\186x\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\190\152\190\152\190\152\190\153\190\153\190\153\190\153\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\185\190\153\190\153\190\153\190\153\190\152\190\152\190\152\190\152\190\152\190\152\190\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186x\186x\186x\186x\182x\182x\182x\182x\182x\182x\182x\182x\1827\170Q\137\148\145\148\145s\141Q\141\145\153o\157\255{\254{\254\127\255\127\254{\255\127\255{\255\243\013\141r\149\148\141\182\145\149\141t\137Q\1377\170v\178w\182x\178x\178x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182\152\182\152\182\152\182\152\182\152\182\152\186\152\186\152\186\152\186\152\186\152\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\185\186\185\186\185\186\185\186\185\186\185\186\185\186\185\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\152\186\152\186\152\186\152\186\152\186\152\186\152\182\152\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\178x\178W\178W\178W\178y\178\245\161r\137\148\141s\137\181\145\148\145q\145\012\141\223\239\255{\255{\255\127\255\127\255{\255{\251\210O\149s\149\148\145\148\141\148\137\149\141\147\1456\166v\178U\178W\178W\178W\178x\178x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182x\182\152\182\152\182\152\182\152\182\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\152\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\153\186\152\186\152\186\152\186\152\186\152\186\152\186\152\182\152\182\152\182\152\182x\182x\182x\182x\182x\182w\182x\182x\182x\182w\182W\178W\178W\178W\178W\178W\178W\178W\178X\174\180\153\180\145s\141\181\145s\137\147\145q\149N\149\184\202\255{\255{\255\127\254{\255w\255\247\146\161\146\153s\145\148\141\149\145\148\141\180\141\146\141\211\153w\174V\174X\178X\178X\178X\178W\178x\178x\178x\178x\178x\178x\178x\178x\178x\182x\182x\182x\182x\182x\182x\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\153\182\153\182\153\182\152\182\152\182\152\182\152\182\152\182\152\182\153\186\153\186\153\186\153\186\152\182\152\182\152\182\152\182\152\182\152\182\153\182\153\182\153\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182x\182x\182x\182x\178x\178x\178x\178w\178w\178w\178W\178x\178W\174W\174W\174W\178W\174W\1747\174x\174\179\149r\141\181\149\149\145\148\141\180\145\147\145p\149\177\161\255\243\255{\255\127\255\127\255w\255\243P\153Q\149\148\149\149\141\149\145\148\141\180\145\146\145\211\153V\174V\174W\174W\174W\174W\178W\178W\178W\178W\178w\178x\178x\178x\178x\178x\178x\178x\178x\178x\182x\182x\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\153\182\153\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\153\182\153\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182\152\182x\182x\182x\178x\178x\178x\178x\178x\178w\178w\178W\178W\178W\178W\174W\174W\174W\174W\174W\1747\1747\170\147\149r\141\181\145\149\141\148\141\181\145\147\145p\149n\153\191\235\255{\255\127\255{\255\243^\215P\145R\141\181\149\181\141\181\141\180\141\180\145\147\145\179\1496\166X\174W\174W\174W\174W\174W\174X\174W\174W\174x\174x\174w\174x\178x\178x\178x\178x\178x\178x\178x\178x\178\152\178\152\178\152\178\152\182\152\182\152\182\152\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\152\182\152\182\152\182\152\178\152\178\152\178x\178x\178x\178x\178x\178x\178x\178x\178x\178w\174w\174W\174W\174W\174W\174W\174W\170W\170W\170W\1707\170\021\162\147\145\147\141\181\145\149\141\149\141\181\145\148\145\147\149.\141\028\211\255\247\255{\255{\255wx\190r\153s\145\181\145\181\145\181\141\180\141\212\145\179\145\179\149\021\162X\174W\170W\170W\174W\174W\174W\174W\174W\174W\174W\174w\174w\174w\174x\178x\178x\178x\178x\178x\178x\178x\178\152\178\152\178\152\178\152\178\152\178\152\182\152\182\153\182\153\182\153\182\153\182\152\182\152\182\152\182\152\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\153\182\152\182\152\182\152\182\152\182\153\182\153\182\153\182\153\182\152\182\152\182\152\178\152\178x\178x\178x\178x\178x\178x\178x\178x\178x\178x\178x\178w\174w\174W\174W\174W\174W\170W\174W\1747\1707\1707\1707\1706\170\211\153\147\145\147\145\181\145\149\141\149\145\181\145\148\145\179\149O\1415\182\255\247\255{\255w\255\247\178\161\180\153\149\149\148\141\181\145\180\141\179\145\212\145\180\145\179\149\213\1538\170W\170W\170W\170W\170W\174X\174W\174X\174W\174X\174W\174W\174x\174x\174x\174x\174x\174x\174x\178x\178x\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\153\182\153\182\152\178\152\178\152\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178x\178x\178x\178x\174x\174x\174x\174x\174x\174x\174x\174W\174W\174W\174W\174W\1747\170W\170W\1707\1707\1707\1707\170\022\166\179\145\179\145\180\145\181\145\149\145\181\145\181\145\149\145\180\145\145\149\144\153\255\243\255{\255{\255\247.\145\148\153\149\149\181\145\213\145\180\141\212\145\212\145\212\149\180\149\180\149\023\1667\1707\1707\170W\170W\170W\174W\174W\174W\174W\174W\174W\174W\174x\174x\174x\174x\174x\174x\174x\174x\178x\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\152\178x\178x\178x\178x\174x\174x\174x\174x\174x\174x\174W\174W\174W\174W\174W\174W\1747\170W\170W\1707\1706\1707\170\022\166\245\161\178\145\179\149\180\145\181\145\181\145\181\145\181\145\181\145\147\145\179\153N\145\255\235\255{\255{\159\231/\145\180\153\148\145\213\145\180\145\212\145\212\149\179\145\212\149\213\153\148\145\247\1577\1667\1667\1707\170W\170W\170W\170W\170W\170W\174W\174X\174x\174x\174x\174x\174x\174x\174x\174x\174x\174x\174\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178x\174x\174x\174x\174x\174x\174x\174w\174w\174X\174W\174W\170W\170W\170W\1707\1707\1707\1706\1666\1667\166\021\162\211\153\179\145\180\149\180\145\181\145\182\145\181\145\149\141\182\145\180\145\179\149p\145]\215\255{\255w\251\210q\153\181\153\148\145\214\149\180\145\212\145\244\149\179\145\212\149\213\153\148\145\214\1576\1666\1667\1667\1667\1707\170W\170W\170W\170W\174W\174X\174x\174x\174x\174x\174x\174x\174x\174x\174x\174x\174x\174\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\178\153\178\152\178\152\178\152\178\152\178\152\178x\174x\174x\174x\174x\174x\174w\174W\174X\174W\174W\170W\170W\170W\1707\1667\1706\166\022\1666\1666\166\245\157\178\149\212\149\212\149\180\145\181\145\182\149\181\145\149\141\182\145\213\149\146\145\145\149\217\198\255w\255wx\190\145\149\213\153\180\141\213\145\213\149\213\149\212\149\212\153\212\153\213\153\245\149\245\149\245\157\022\1627\1668\1667\1667\166W\166X\170X\170X\170X\170X\170x\170x\170x\170x\170x\174x\174x\174x\174x\174x\174x\174x\174\152\174\152\178\152\178\152\178\152\178\152\178\153\178\153\178\153\174\153\174\153\174\153\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\153\174\153\174\153\174\153\174\153\178\153\178\152\178\152\178\152\178\152\178\152\178\152\174x\174x\174x\174x\174x\174x\174x\174x\174x\174X\174X\170X\170X\170X\170X\170W\1707\1667\1667\1627\1627\162\022\162\213\153\180\149\212\149\213\149\181\149\181\149\212\145\212\145\180\153\180\153\212\145\212\141\149\141:\170\255\127\255w6\182\146\149\213\149\181\145\213\149\213\149\213\149\212\153\212\153\212\153\213\153\246\153\246\153\245\157\022\1627\1668\1667\1667\166W\166X\170X\170X\170X\170X\170X\170x\170x\170x\170x\174x\174x\174x\174x\174x\174x\174x\174\152\174\152\178\152\178\152\178\152\178\153\178\153\178\153\178\153\174\153\174\153\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\153\174\153\174\153\174\153\178\153\178\153\178\152\178\152\178\152\178\152\178\152\174x\174x\174x\174x\174x\174x\174x\174x\174x\174X\174X\170W\170X\170X\170X\170W\1707\1667\1667\1667\162\023\162\246\157\213\153\212\149\213\149\213\149\181\149\181\149\212\145\212\145\180\153\180\153\212\141\212\141\149\141\248\157\255\127\255w\211\169\178\153\213\149\181\145\214\149\213\149\213\149\212\153\213\153\245\153\245\153\246\153\246\153\245\157\022\158\022\1627\166X\166X\166W\166W\166W\170X\170W\170X\170w\170x\170x\170x\170x\170x\170x\170x\174x\174\152\174\152\174\152\174\152\174\152\174\152\174\152\174\152\174\153\178\153\178\153\178\185\174\185\174\185\174\185\174\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\174\185\174\185\174\185\174\153\178\153\178\153\178\152\174\152\174\152\174\152\174\152\174\152\174\152\174\152\174x\174x\174x\170x\170x\170x\174x\170X\170W\170X\170W\170W\166W\1667\1667\1668\1667\162\022\158\245\153\213\153\245\153\212\149\213\149\181\149\181\149\212\145\212\145\180\153\180\153\212\141\212\145\149\141\181\149\254\127\255wp\157\179\153\212\149\214\149\214\149\213\149\213\153\213\153\213\153\245\153\245\157\246\153\246\153\245\157\246\157\022\1627\1627\166X\166W\166W\166W\170W\170W\170X\170w\170x\170x\170x\170x\170x\170x\174x\174x\174\152\174\152\174\152\174\153\174\153\174\152\174\152\174\153\178\153\178\153\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\153\178\153\178\153\178\152\174\152\174\153\174\153\174\152\174\152\174\152\174x\174x\174x\174x\170x\170x\174x\170X\170W\170W\170W\170W\166W\1667\1667\1667\162\022\162\245\153\213\153\245\153\245\153\213\149\213\149\214\149\213\149\212\145\212\145\181\153\181\153\212\141\245\145\181\145t\141\254\127\255\247N\149\211\153\180\149\214\149\213\149\213\149\213\153\213\153\245\153\245\153\245\157\246\153\246\153\022\158\022\158\022\158\022\1627\166W\166W\166W\166W\166W\170W\170X\170w\170x\170x\170x\170x\170x\170\152\170\152\170\152\174\152\174\152\174\152\174\152\174\152\174\152\174\185\174\185\174\185\174\185\174\185\174\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\174\185\174\185\174\185\174\185\174\152\174\152\174\152\174\152\174\152\174\152\174\152\174\152\170\152\170x\170x\170x\170x\170X\170W\170X\166W\166W\166W\166W\166W\1627\162\022\158\245\153\245\153\245\153\245\153\213\153\213\153\214\153\214\149\212\149\212\145\181\153\181\153\212\145\245\149\182\145t\141\254\127\255\247N\149\211\157\212\149\246\149\213\149\214\149\213\153\213\153\245\153\245\157\245\157\022\154\022\158\022\162\022\158\022\158\022\1626\1627\166W\166X\166W\170W\170W\170x\170x\170x\170x\170x\170x\170\152\170\152\170\152\170\152\174\152\174\152\174\153\174\152\174\152\174\185\174\185\174\185\174\185\174\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\174\185\174\185\174\185\174\152\174\152\174\153\174\152\174\152\174\152\174\152\170\152\170\152\170x\170x\174x\170x\170X\170X\166W\166W\166W\166W\1667\1626\158\022\158\022\158\022\158\245\153\245\153\245\153\214\153\214\153\214\153\212\149\212\149\181\153\181\153\212\145\212\145\214\145t\141\220\251\255wo\153\211\157\212\149\246\153\213\149\246\153\245\153\245\153\245\157\245\157\246\157\022\158\023\158\022\158\022\158\022\1626\1626\1627\162W\166W\166W\166W\166w\170x\170x\170x\170x\170x\170\152\170\152\170\152\170\152\170\152\174\152\174\152\174\152\174\152\174\185\174\184\174\185\174\185\174\185\174\185\174\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\174\185\174\185\174\185\174\184\174\185\174\152\174\152\174\152\174\152\174\152\174\152\170\152\170\152\170\152\170\152\170x\170x\170w\170x\166w\166W\166W\166W\1627\1626\158\022\158\022\158\022\158\021\154\245\153\245\153\214\153\214\153\214\153\212\149\212\149\181\153\181\153\212\145\212\145\214\145\149\145\155\247\255w\144\157\211\157\213\149\246\153\213\149\246\153\245\153\245\157\245\157\245\157\022\158\023\158\023\158\022\158\022\1627\1627\1627\1627\162W\166W\166W\166x\170w\170x\170x\170x\170x\170\152\170\152\170\152\170\152\170\152\170\152\174\152\174\152\174\152\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\178\185\174\185\174\185\174\185\174\185\174\185\174\185\174\152\174\152\174\152\174\152\174\152\170\152\170\152\170\152\170\152\170x\170x\170x\170x\166x\166W\166W\1666\1626\1627\1626\158\022\158\022\158\022\154\021\154\245\153\246\153\214\153\214\153\213\149\212\149\181\153\181\153\245\149\212\145\214\145\181\149z\243\255\243\144\161\178\153\213\153\246\153\246\153\246\153\246\153\245\157\245\157\245\157\022\158\023\158\023\1586\1627\1627\1627\1627\162W\162W\166W\166w\166w\166w\166x\170x\170x\170x\170\152\170\152\170\152\170\152\170\152\170\152\174\184\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\217\178\185\178\217\178\217\178\217\178\217\178\217\178\217\178\217\178\218\178\218\178\217\178\217\178\217\178\217\178\217\178\217\178\217\178\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\184\174\152\174\152\170\152\170\152\170\152\170\152\170\152\170x\170x\166w\166x\166W\166W\166W\162W\166W\1627\1626\1626\158\022\158\022\158\021\154\245\153\246\157\214\153\214\153\245\149\245\149\213\157\213\157\245\149\245\149\214\149\181\149\155\247\255\247\144\157\211\157\245\153\246\153\246\153\246\153\246\153\245\157\022\158\022\158\022\158\023\158\023\1587\1627\1627\162W\162W\166W\166W\166W\166w\166x\166w\170x\170x\170x\170\152\170\152\170\152\170\152\170\153\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\178\217\178\217\178\217\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\217\178\217\178\217\178\185\178\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\184\174\153\174\152\170\152\170\152\170\152\170\152\170x\170x\166x\170x\166W\166W\166W\166W\1627\1626\1626\158\022\158\022\158\021\158\246\153\246\157\246\157\214\153\245\149\245\149\213\157\213\157\213\145\245\149\214\149\149\145\155\247\255\247o\157\244\161\245\153\246\153\246\153\246\153\246\157\021\158\021\158\021\158\022\162\023\1587\1627\162W\162W\162W\166W\166W\166w\166x\166w\166x\170w\170x\170\152\170\152\170\152\170\152\170\184\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\178\217\178\217\178\217\178\217\178\217\178\217\178\217\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\217\178\217\178\217\178\217\178\217\178\217\178\185\178\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\185\174\184\174\184\174\152\170\152\170\152\170\152\170x\170x\170x\166W\166W\166W\1667\1626\1627\1626\158\022\158\022\158\246\157\246\157\246\157\214\153\245\153\245\153\213\157\213\157\212\149\245\149\214\149t\145\188\247\255wo\157\244\161\245\153\246\153\246\153\246\153\246\157\021\158\022\162\022\162\022\1627\1627\162W\162W\162W\166W\166W\166W\166x\166x\170x\170x\170x\170\152\170\152\170\152\174\152\174\152\174\185\174\185\174\185\174\185\174\185\174\185\174\185\178\217\178\217\178\218\178\218\178\218\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\218\178\218\178\218\178\218\178\218\178\218\178\217\178\217\178\218\178\217\178\217\178\217\178\185\178\185\174\185\174\185\174\185\174\185\174\185\174\185\174\153\174\152\170\152\170\152\170\152\170x\170x\170x\166W\166W\166W\1627\1627\1626\162\022\158\022\158\022\158\246\157\247\157\246\157\245\153\245\153\213\157\213\157\245\149\245\149\215\153t\145\221\127\255w\144\157\243\157\245\153\246\153\247\153\022\154\022\158\022\162\022\162\022\162\023\1627\1628\162W\162W\162W\166W\166W\166x\166x\166x\170\152\170\152\170\152\170\152\170\152\174\152\174\152\174\185\174\185\174\185\174\185\174\185\174\185\178\217\178\217\178\217\178\218\178\218\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\218\178\218\178\218\178\218\178\218\178\218\178\217\178\217\178\185\178\185\174\185\174\185\174\185\174\185\174\185\174\153\174\152\174\152\170\152\170\152\170x\170x\170X\166W\166W\1667\1667\1666\162\022\162\022\162\022\158\022\158\247\157\246\157\245\153\245\153\213\161\213\161\245\149\245\153\214\153\149\149\253\127\255w\210\169\211\157\245\153\246\153\023\154\022\158\022\158\022\162\022\1626\1627\1628\1628\162W\162W\166W\166W\166x\166x\166x\170x\170\152\170\152\170\152\174\152\174\152\174\185\174\185\174\185\174\185\174\185\174\185\174\185\178\217\178\218\178\218\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\251\182\251\182\251\182\251\182\251\182\251\182\251\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\218\178\218\178\218\178\218\178\217\178\218\178\217\178\185\178\185\174\185\174\185\174\185\174\185\174\153\174\152\174\153\174\152\170x\170x\170x\170X\166W\166W\1667\1667\1626\162\022\162\022\162\022\158\247\157\247\157\245\153\245\153\213\161\213\161\245\149\022\154\214\149\182\153\254\127\255w5\182\211\157\246\157\246\153\023\154\022\158\022\158\022\162\022\1626\1627\166X\162X\166W\162X\166x\166x\166x\166x\166x\170\152\170\153\170\153\170\153\174\185\174\185\174\185\174\185\174\185\174\185\178\185\178\218\178\218\178\218\178\218\178\218\182\218\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\251\182\251\182\251\182\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\251\182\251\182\251\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\218\182\218\178\218\178\218\178\218\178\218\178\185\178\185\178\185\178\185\178\185\174\185\174\153\174\153\174\153\174x\170x\170x\170X\170W\166W\1667\1667\1666\162\022\162\022\162\023\162\247\161\247\157\021\154\245\153\213\161\213\161\245\149\022\154\214\149\248\161\254\127\255ww\190\211\157\246\157\246\153\023\154\022\158\022\158\022\1626\1626\1627\166X\162X\166X\166x\166x\166x\166x\166x\170\152\170\153\170\153\170\153\170\153\174\185\174\185\174\185\174\185\174\185\174\218\178\218\178\218\178\218\178\218\182\218\182\218\182\218\182\250\182\250\182\250\182\251\182\251\182\251\186\251\186\251\186\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\027\183\250\186\250\182\251\182\250\182\250\182\250\182\250\182\250\182\218\182\218\182\218\182\218\178\218\178\218\178\218\178\218\178\186\178\185\178\185\174\185\174\153\174\153\174\153\174x\170x\170x\170W\170W\1667\1667\1666\1626\162\022\162\023\162\247\161\247\157\021\154\245\153\213\161\213\161\245\149\022\158\214\149\025\166\254\127\255{\250\206\211\161\246\157\023\158\023\158\023\158\023\158\022\1626\1626\1667\166W\166W\166W\166W\166x\170x\170x\170x\170\152\170\153\174\152\174\153\174\185\174\185\174\185\174\185\174\217\178\217\178\217\178\217\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\026\183\026\183\026\183\027\183\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\026\187\026\187\027\183\027\183\027\183\027\183\250\182\250\182\250\182\250\182\250\182\250\182\250\182\218\178\218\178\218\178\218\178\218\178\217\178\217\178\185\174\185\174\185\174\185\174\152\174\152\170x\170x\170x\170X\166X\166W\1667\1667\162\022\162\022\162\022\162\022\158\246\157\246\157\247\153\247\153\247\157\180\153\211\161\185\194\255{\255{]\219\178\157\246\161\023\158\023\158\023\158\023\158\022\1626\1626\1667\166W\166W\166W\166x\166x\170x\170x\170\152\170\153\174\153\174\153\174\185\174\185\174\185\174\185\178\217\178\217\178\218\178\218\182\218\182\218\178\250\182\250\182\250\182\250\182\250\182\250\182\250\182\250\182\026\183\027\183\027\183\027\183\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\027\183\027\183\027\183\027\183\250\182\251\182\251\182\250\182\250\182\251\182\250\182\218\182\218\178\218\182\218\178\217\178\217\178\185\178\185\174\185\174\185\174\152\174\153\174\152\170x\170x\170X\166X\166W\1667\1667\162\022\162\022\162\022\162\022\158\022\158\246\157\247\153\247\153\246\157\213\157\179\157\028\211\255{\255{\223\239\177\157\245\161\246\157\023\162\023\158\023\162\022\1626\1666\166W\166W\166W\166x\166x\170x\170\152\170\152\170\153\174\153\174\185\174\185\174\185\174\185\178\217\178\217\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\251\182\251\182\026\183\026\183\027\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187\027\187\027\187\027\187\027\187\027\187\027\187\027\183\027\183\251\182\251\182\251\182\250\182\250\182\250\182\218\182\218\178\218\178\217\178\217\178\185\174\185\174\185\174\152\174\153\174\152\174x\170x\170x\170X\170W\1667\1667\166\022\162\022\162\022\162\022\162\022\158\247\157\247\157\247\157\246\157\245\161\145\157\159\231\255{\255{\255\247\243\165\245\161\246\157\023\162\023\162\023\1627\1626\1667\166W\166W\166W\166x\170x\170x\170\153\170\152\174\153\174\185\174\185\174\185\174\185\178\217\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\251\182\027\187\027\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\187<\187<\187<\187<\187;\187;\187;\187;\187;\187;\187;\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\251\182\251\182\251\182\251\182\250\182\250\182\250\182\218\178\218\178\217\178\217\178\185\174\185\174\185\174\153\174\153\174\153\170x\170x\170X\170X\166W\1667\1666\166\022\162\022\162\022\162\022\158\023\158\247\157\247\157\246\157\022\166\177\157\255\243\255{\255{\255\247w\186\211\161\022\162\022\162\023\162\023\1627\1627\1667\166W\166W\170X\170x\170x\170\152\170\153\170\153\174\153\174\185\174\185\174\185\174\185\178\217\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\251\182\027\187\027\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\187;\187;\187;\187;\187;\191;\191;\191<\191;\191;\191;\187;\187;\187;\187;\187;\187;\187;\187;\187\027\187\027\187\027\187\027\187\027\187\027\183\251\182\251\182\251\182\250\182\250\182\250\182\250\182\218\178\217\178\217\178\217\178\185\174\185\174\185\174\153\174\153\174x\170x\170x\170X\170W\1667\1666\1666\1666\162\023\162\023\158\023\158\022\158\246\161\245\161\244\165\243\173\255\247\255\127\255{\255\247<\211\211\161\022\162\022\162\023\1627\1627\1667\166W\166W\170X\170X\170x\170x\170\153\170\153\174\153\174\153\174\185\174\185\174\185\178\185\178\218\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\251\182\027\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187<\187<\187;\191;\191<\191<\191<\191<\191<\191\092\191\092\191\092\191\092\191<\191<\191;\191;\191;\191<\187<\187<\187;\187;\187;\187\027\187\027\187\027\187\027\187\027\187\251\182\251\182\251\182\250\182\250\182\250\182\250\178\218\178\217\178\217\178\217\178\185\178\185\174\185\174\153\174\152\174x\170x\170X\170W\166W\1666\1666\1667\162\023\162\023\158\023\158\023\158\247\161\245\161\244\165\185\194\255\247\255\127\255{\255w\191\235\211\165\244\165\022\166\023\1667\1627\1667\166W\166W\170X\170x\170x\170\152\170\153\174\153\174\153\174\185\174\185\174\185\178\185\178\218\178\218\178\218\178\250\182\250\182\250\182\250\182\250\182\250\182\026\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\187<\191<\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191[\191[\191\092\191<\191<\191<\187;\187;\187;\187;\187\027\187\027\187\027\187\027\187\027\187\251\182\250\182\250\182\250\182\250\182\218\178\218\178\218\178\217\178\185\178\185\174\185\178\153\174\153\174x\170x\170x\170X\166W\1666\1666\1667\162\023\162\023\162\023\158\022\162\246\165\244\165\210\165\158\227\255{\255\127\255{\255{\255\243\243\169\243\1656\166\023\1667\1627\1667\166W\166W\170X\170x\170\152\170\153\174\153\174\153\174\185\174\185\174\185\178\186\178\218\178\218\178\218\182\250\182\250\182\250\182\251\182\251\182\026\187\027\187\027\187\027\187\027\187\027\187;\187;\187;\187<\187;\187;\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191<\191<\187<\187<\187\027\187\027\187\027\187\027\187\027\187\027\187\027\187\251\182\250\182\250\182\250\182\218\178\218\178\218\178\185\178\185\178\185\178\185\174\153\174\153\174x\170x\170X\170W\1666\1666\1667\162\023\162\023\162\023\158\022\162\246\165\212\165\178\165\255\247\255w\255\127\255{\255{\255{\151\198\243\169\244\165\022\1667\1667\166X\166X\166X\166w\170x\170x\170\152\170\153\174\153\174\185\174\185\178\185\178\218\178\218\178\218\182\218\178\250\182\250\182\251\182\250\182\251\182\026\183\026\187\027\187\027\187\027\187;\187;\187;\187;\191;\191[\191[\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191<\191<\191<\191;\187;\187\027\187\027\187\027\187\027\187\027\187\026\183\250\182\250\182\250\182\250\182\250\182\218\178\217\178\186\178\185\178\185\174\153\174y\174x\170x\170X\170W\1667\1666\166X\170\246\1578\162\023\158\246\161\021\170\210\165v\194\255{\255{\255\127\222{\255{\255{\191\231\177\165\021\170\022\1667\1667\1627\166X\166W\166x\170x\174x\170\152\170\153\174\185\174\185\174\185\178\185\178\218\178\218\182\218\182\250\182\250\182\250\182\251\182\251\182\251\182\026\187\027\187\027\187\027\187\027\187;\187;\187;\187;\191<\191[\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191|\195|\195|\195|\195|\195|\195|\195|\195\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191<\191<\191<\187;\187\027\187\027\187\027\187\027\187\027\187\026\183\027\183\250\182\250\182\250\182\250\182\218\178\218\178\186\178\185\178\185\174\153\174\153\174x\174x\170X\170W\166W\1706\166W\166\023\1628\162\023\158\023\166\021\170\210\165]\223\255{\255\127\255\127\255\127\255\127\255{\255{\019\178\020\1745\1707\1668\1668\166X\166X\166x\174x\174x\170\152\174\153\174\185\174\185\178\185\178\218\178\218\178\218\182\218\182\250\182\250\182\250\182\250\182\027\187\027\187\026\187\026\187\027\187\027\187;\187;\187;\187;\187[\191[\191\092\191\092\191\092\191\092\191\092\191\092\191|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191;\191;\187;\187;\187\027\187\027\187\027\187\026\187\026\183\250\182\250\182\250\182\250\182\218\178\217\178\218\178\185\178\185\174\153\174\153\174x\174x\170X\170W\170W\1706\1667\1627\1628\166\023\1587\170\244\169\209\173\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\247\027\215\210\1656\1747\170X\166X\166X\166X\166x\174x\174\152\170\153\174\153\174\185\174\185\178\186\178\218\178\218\178\218\182\250\182\250\182\250\182\250\182\027\187\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\191;\191\092\191\092\191\092\191\092\191\092\191\092\191|\191|\191|\195|\195|\195|\195|\195}\195}\195}\195}\195}\195}\195|\195|\195|\195|\195|\195|\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191\092\191;\191;\187;\187;\187;\187\027\187\027\187\027\187\026\183\250\182\250\182\250\182\250\182\218\178\218\178\185\178\186\178\153\174\153\174y\174x\170x\170X\170W\170W\1707\1628\1668\162\023\162\023\166\178\161\151\194\255{\255\127\255\127\255\127\255\127\222\127\255\127\255{\255\247\209\165\021\1707\166X\166X\166X\166x\170w\174x\174\152\170\152\174\185\174\185\178\185\178\185\178\218\178\218\178\250\182\250\182\250\182\250\182\251\182\027\187\027\187\027\187;\187;\187;\187;\187;\187;\191[\191\092\191\092\191\092\191\092\191\092\191|\191|\191|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\191|\191\092\191\092\191\092\191\092\191\092\191\092\191;\191;\187;\187;\187;\187\027\187\027\187\027\187\251\182\250\182\250\182\250\182\218\182\218\178\217\178\185\178\185\174\153\174\153\174x\174x\170X\170W\170W\1707\1628\1667\1628\162\022\166\210\165\191\235\255\127\255\127\255\127\255\127\255\127\255\127\222{\255\127\255w\152\194\020\170\022\1668\162X\166X\166x\170x\174x\174\152\174\153\174\185\174\185\178\185\178\218\178\218\178\218\182\250\182\250\182\250\182\250\182\027\187\027\187\027\187\027\187;\187;\187;\187;\187;\191\092\191\092\191\092\191\092\191\092\191\092\191|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195}\195}\195}\195}\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195|\195\092\191\092\191\092\191\092\191\092\191\092\191;\191;\191;\187;\187;\187\027\187\027\187\027\187\251\182\251\182\250\182\218\182\218\178\218\178\186\178\185\178\153\174\153\174y\174x\170x\170W\170W\170X\1668\162\023\1628\162\245\165V\186\255w\223{\255\127\255\127\255\127\255\127\255\127\222{\255\127\255{\191\231\020\1706\166X\166X\166X\166x\170x\174x\174\152\174\153\174\185\174\185\178\185\178\218\178\218\178\218\182\250\182\250\182\250\182\251\182\027\187\027\187\027\187\027\187;\187;\187;\187;\191[\191\092\191\092\191\092\191\092\191\092\191|\191|\195|\195|\195|\195|\195|\195|\195|\195|\195\157\195\157\195\157\195\157\195\157\195\157\195\157\195\157\195|\195|\195|\195|\195}\195}\195|\195|\195|\195|\191\092\191\092\191\092\191\092\191\092\191\092\191<\191;\191;\187;\187\027\187\027\187\027\187\250\182\250\182\250\182\250\182\218\182\218\178\186\178\185\178\153\174\153\174\153\174x\170x\170W\1746\170X\166X\166\023\1628\162\021\166^\219\255{\222{\222\127\222\127\255\127\221{\222\127\255\127\223{\255{\255\2436\178W\170X\166X\166X\166x\170x\174x\174\152\174\153\174\185\178\185\178\185\178\218\178\218\182\218\182\250\182\251\182\250\182\027\187\027\187\027\187\027\187\027\187;\187;\187;\191;\191\092\191\092\191\092\191\092\191\092\191|\195|\195|\195|\195|\195|\195|\195\157\195\157\195\157\195\157\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195\157\195\157\195\157\195}\195}\195}\195|\195|\195|\195|\195\092\191\092\191\092\191\092\191\092\191\092\191<\191;\187;\187\027\187\027\187\027\187\026\187\250\182\250\182\250\182\218\182\218\178\186\178\185\178\153\174\153\174\153\174x\174x\170x\174\022\166X\166X\1668\1627\162\022\170\255\243\255\247\255\127\254\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\223\235\176\165U\178W\178W\170Y\170Y\166\155\174w\178\152\182\152\178\185\178\185\178\218\178\219\178\251\182\251\182\251\182\251\186\251\186\027\187\026\187:\187:\187;\191;\191;\191\092\191\092\191\092\191\092\191\092\195|\195|\195|\195|\195|\195}\195\157\195\157\195\156\195\156\195\157\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195\157\195\157\195\156\195\156\195\157\195\157\195}\195|\195|\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191;\191;\191;\191\028\191\028\191\252\190\251\186\250\186\250\186\250\182\249\182\217\178\217\178\185\174\185\174\153\178y\178x\178X\178y\158\154\166X\166\021\1665\178\209\173\092\227\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255\247\151\1944\174\021\170x\1748\166\155\174z\170w\178\152\182\152\178\185\178\185\178\218\178\219\178\251\182\251\182\251\182\251\186\027\187\027\187\026\187:\187:\187;\191;\191;\191\092\191\092\191\092\191\092\191\092\195|\195|\195|\195|\195|\195}\195\157\195\157\195\156\195\157\195\157\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195\157\195\157\195\156\195\157\195\157\195}\195|\195|\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191;\191;\191;\191\028\191\028\191\252\190\251\186\250\186\250\186\250\182\249\182\217\178\217\178\217\178\185\178\153\178y\178x\178X\178\154\166Y\158x\166W\174\020\170v\190\255w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255\243\209\169V\182\021\170\154\178X\166z\174\152\178\153\178\153\178\185\178\218\178\218\178\218\178\251\182\251\182\251\186\251\186\251\186\027\187\027\187;\187;\187;\191;\191[\191\092\191\092\191\092\191\092\195\092\195|\195|\195|\195|\195}\195\157\195\157\195\157\195\157\195\157\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195\157\195\157\195\157\195\157\195\157\195}\195|\195|\195|\195|\195\092\195\092\195\092\191\092\191\092\191[\191;\191;\191<\187\028\191\027\187\251\190\250\186\250\186\250\186\250\182\217\178\217\178\185\178\185\174\185\174\153\174y\178y\174x\166x\1667\170V\178\210\169\158\231\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255\247\217\202\243\165\153\182X\170X\166\154\174\152\178\153\178\185\178\185\178\218\178\218\178\218\178\251\182\251\182\251\186\251\186\027\187\027\187\027\187;\187;\187;\191;\191\092\191\092\191\092\191\092\191\092\195\092\195|\195|\195|\195|\195}\195\157\195\157\199\157\199\157\195\157\195\157\199\157\199\157\199\157\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195\157\199\157\199\157\195}\195|\195|\195|\195|\195\092\195\092\195\092\191\092\191\092\191\092\191;\191;\191<\191\028\191\027\187\251\190\251\190\250\186\250\186\250\182\217\178\218\178\217\178\185\174\185\178\153\178y\178y\174X\166\153\170W\170\243\165\184\194\255w\223\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255\2475\182V\1826\170y\174y\174\153\174\154\174\186\174\186\174\218\178\218\178\218\182\218\182\250\186\250\186\251\190\251\190\027\187;\187<\187<\187;\191;\191\092\191\092\191\092\191\092\195\092\195|\195|\195|\195|\195}\195\157\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195\157\195}\195|\195|\195|\195|\195\092\195\092\195\092\191\092\191\092\191;\191;\191;\187;\187\027\187\027\187\251\190\250\186\250\186\250\186\218\182\218\178\218\174\186\174\186\174\154\174\153\174y\174w\174w\1784\174\019\178\255\243\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255w\255w\223\231\177\157\153\182Y\174y\174\153\174\154\174\186\174\186\174\218\178\218\178\218\182\218\182\250\186\251\186\251\190\027\191\027\187;\187<\187<\187;\191\092\191\092\191\092\191\092\191\092\195|\195|\195}\195}\195|\195}\195\157\195\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195}\195|\195}\195}\195|\195|\195\092\195\092\191\092\191\092\191\092\191;\191<\187;\187\027\187\027\187\251\190\250\186\250\186\250\186\250\182\218\178\218\178\186\174\186\174\186\174\153\174\153\174\152\174v\178\243\165~\219\255w\223\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\254\127\223{\255{\255w\184\198V\182w\182x\174\154\170\154\174\186\174\186\174\218\178\218\178\218\182\250\186\250\186\250\190\251\190\027\191\027\187\028\187<\187=\187\092\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\195\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\190\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\195}\195}\195}\195}\195|\195|\195\092\195\092\195\092\191\092\191\092\191\092\191[\187;\187;\187\027\187\027\191\251\190\250\186\251\186\251\182\219\178\218\178\218\174\186\174\186\174\153\174\153\174u\1824\182\183\198\255{\255w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\254\127\255\127\255{\255w\255\243\020\174w\182x\174\154\170\154\174\186\174\186\174\218\178\218\178\218\182\250\186\250\186\250\190\251\190\027\191\027\187\028\187<\187]\187\092\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\195\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\190\199\190\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\195}\195}\195}\195}\195|\195|\195\092\195\092\195\092\191\092\191\092\191\092\191[\187;\187;\187\027\187\027\191\251\190\251\186\251\186\251\182\219\182\218\178\218\174\186\174\186\174\153\174\153\174u\182\019\178\255\243\255\247\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\254\127\221{\255\127\223w\255{\255\247\241\173v\186y\178\153\174\186\178\251\182\185\174\152\174\027\187\250\186\250\186\250\190\251\190\027\191\027\191<\187<\187<\187\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\190\199\190\199\190\199\190\199\190\199\190\199\190\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191[\187[\187;\187\027\191\027\191\027\191\251\186\251\186\218\182\251\186\185\174\218\178\218\182\185\178w\174\153\182\017\178\190\239\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\254\127\254\127\221\127\255\127\255\127\255{\223\243\255\247~\2275\178\153\178x\170\186\178\153\174\251\182\251\186\151\170\027\191\250\186\250\190\251\190\027\191\027\191<\187<\187<\187\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\190\199\190\199\190\199\190\199\190\199\190\199\190\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191[\187[\187;\187\027\191\027\191\027\191\251\186\251\186\251\186\185\174<\191\185\174\185\178\185\182\218\1825\170[\219\255{\255{\255{\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\255\127\255\127\255\127\255\127\255{\255{\255{\216\2105\178\185\190w\178\152\182\218\186\184\178\250\182\218\182\251\186\251\186\027\187\027\191\027\191;\187<\187<\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\190\199\190\199\190\199\190\199\190\199\190\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191[\191;\191;\191\027\191\027\191\028\187\027\187\251\186\250\182\028\191\152\174\027\191\151\182v\182\151\186\217\202\255{\255\127\255{\223{\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\254\127\255\127\222\127\255\127\255\127\255\127\223w\255{\255w\219\202w\182\152\182\185\186\152\178\217\182\218\182\218\182\251\186\251\186\027\187\027\191\027\191;\187<\187<\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\190\199\190\199\190\199\190\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191\092\191[\191;\191;\191\027\191\027\191\028\187\027\187\251\186\028\187\217\178\027\187\184\182\184\186v\182\216\194\255\243\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\255\127\255\127\255\127\255\127\255w\184\202U\186\151\190\185\186\250\186\218\182\251\182\251\182\251\182\027\187\027\187\027\187;\191;\191;\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\190\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191<\191<\195<\191;\191\028\191\028\187\028\187\027\187\027\183\250\182\027\187\217\186\183\186u\186\216\202\255{\255\127\255{\255\127\255\127\255\127\223\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\255{\255\243\217\202v\186\185\186\184\178\251\186\218\182\251\182\251\182\027\187\027\187\027\187;\191;\191;\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\190\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191<\191<\195<\191;\191\028\191\028\187\028\187\027\187\027\183\027\187\250\182\216\182\183\186\216\198\255\247\255w\255{\255\127\255\127\255\127\255\127\223\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\190w\255\127\255\127\222\127\254\127\221\127\223\127\255{\255\247\217\202\151\182\218\186\152\174\028\183\252\182\252\182\028\183\028\187\027\187;\191:\191:\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191<\191<\195<\195<\191\028\191\028\187\028\187\027\183\027\183\250\182\250\186\216\186\249\198\255\247\255{\255\127\223\127\255\127\255{\255\127\223\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255\127\255\127\255\127\255\127\255\127\221\127\254\127\255\127\255{\255w\255\239\027\203\152\178\251\182\251\182\252\182\252\182\028\183\028\187\027\187;\191:\191:\191\092\191\092\191\092\191\092\195\092\195|\195|\195}\195}\195}\195}\195\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\189\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\195}\195}\195|\195|\195|\195\092\195\092\191\092\191\092\191<\191<\195<\195<\191\028\191\028\187\028\187\027\183\027\183\250\182\216\178;\199\255\231\255\243\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\223\127\191\127\255\127\255\127\255w:\215\182\190\214\186\024\187\249\182\249\182\027\187<\191\028\191\029\191>\195[\191[\191[\191\092\195\092\195|\195|\195|\195|\195|\195|\195}\199\157\199\157\199\157\199\157\199\158\195\158\195\158\195\158\195\189\195\190\199\189\199\189\199\189\199\189\199\189\199\190\195\190\195\158\195\158\195\158\195}\203}\203}\203}\199}\199|\199}\195|\195\092\191|\195|\195|\191|\191[\187[\187[\187}\183}\183\092\183;\187;\191\249\190\216\194\248\198\149\194[\223\255w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\223\127\223\127\223\127\222{\255{\255\247\190\223\215\186\248\186\025\187\026\187;\187<\191\028\191\028\191\028\191[\191[\191[\191\092\195\092\195|\195|\195|\195|\195|\195|\195}\199\157\199\157\199\157\199\157\199\158\195\158\195\158\195\158\195\189\195\189\195\189\199\189\199\189\199\189\199\189\195\190\195\158\195\158\195\158\195\158\195}\203}\203}\203}\199}\199|\199}\195|\195}\195|\195[\187[\187|\191|\191|\191|\187]\179<\175\092\183\092\187\027\191;\195\025\199\182\190\157\227\255\243\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\255\127\255\127\255\127\255{\255{\255{\255\239\150\194\216\198\250\198\249\190\249\190;\191\092\195<\191\092\191\092\191\092\191]\191]\191}\191}\195}\195|\195|\195|\195|\195\156\199\157\199\156\199\156\199\157\203}\203}\203\157\203\157\203\157\203\157\203\157\203\157\203\157\203\157\203\157\203\157\203}\203}\203}\203\188\191\188\191\156\191\156\191\156\191|\191|\191|\195}\195]\195\092\195\092\195]\195=\199<\195\028\195<\191<\191\026\191\027\195:\199\183\194\248\202\223\239\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\223\247\255w\255\247\255\243;\211\183\186\249\190;\195\250\186\250\186\092\191\092\191\092\191\092\191]\191]\191}\191|\195}\195|\195|\195|\195|\195|\199\156\199\156\199\156\199}\203}\203}\203}\203\157\203\157\203\157\203\157\203\157\203\157\203\157\203}\203}\203}\203}\203}\203\188\191\188\191\156\191\156\191\156\191|\191|\191|\195\092\191]\195]\199]\199<\195\028\195\028\195<\195<\191<\191\092\199\250\190\183\186\092\211\255\243\255\247\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255w\254{\255\127\255\127\255\127\223\127\223\127\255\127\255\127\191\239\248\206\182\190\026\199Z\199\025\187=\191=\191=\191]\191]\191]\191]\195|\195|\195}\195|\195|\195\156\195\156\195\156\195\156\195\157\191\157\191\189\191\189\191\188\191\189\195\188\195\188\195\188\195\188\195\189\195\188\191\189\191\189\191\157\191\157\195\156\199\156\199|\195|\195|\195|\195|\195|\195|\195|\195\092\191]\191\092\191\092\187\092\191}\195\216\194\026\207\216\202\248\206\191\235\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\223\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\247\255{\255{\255\127\255\127\255\127\255\127\255\127\223{\255\127\255{\255\243\157\219\025\195\024\191Z\195=\191=\191=\191]\191]\191]\191]\195|\195|\195|\195|\195|\195|\195\156\195\156\195\156\195\157\191\157\191\156\191\189\191\188\191\188\191\188\191\188\191\188\191\188\191\188\191\188\191\189\191\156\191\157\191\157\191|\195|\195|\195|\195|\195|\195|\195|\195|\195\092\191\092\191}\191}\195\092\191;\187;\183\026\203\216\198|\219\255\243\255\247\255w\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\254w\222w\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255w\255\235Z\215\247\194\027\199\027\199;\199<\199<\199\092\199\092\195\092\195|\195|\195}\195}\195\156\191\157\195\157\191\157\191}\199}\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\199}\199}\199}\199}\199}\199|\199|\199|\199|\199[\195[\195[\195|\199{\195:\1919\191:\191\025\219\223\243\255{\255\127\255{\255{\255\127\255\127\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254w\255{\255\127\255\127\255\127\255\127\223\127\223\127\223\127\255\127\255\127\255{\255w\255\243\255\243\255\235\027\199\027\199;\199<\199<\199\092\199\092\195\092\195|\195|\195|\195}\195\156\191\156\191\157\191\157\191}\199}\199}\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199\157\199}\199}\199}\199}\199}\199}\199}\199|\199|\199|\199|\199[\195[\195[\195Z\195\025\1879\187\156\203\255\219\255{\255{\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\247\223\235[\219\025\207\024\203\026\203:\203;\203[\203\092\199\092\195}\195}\195}\191}\191\158\195}\199}\199}\199}\199}\199}\199\157\199\157\199}\195}\195}\195}\199}\199}\199}\199|\199\157\187\157\187\157\187}\191}\191\092\195[\195:\199[\207\025\207\248\202\024\211{\227\222\243\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\247\255\247\255\243\255\239\157\223:\207\026\203:\203[\199\092\199|\199}\199\158\199}\195}\191}\191|\199|\199}\199}\199}\195}\195}\195}\195\158\199\158\199\157\199}\199}\199}\199\092\199\092\199\190\191\158\187\157\187}\187|\191\092\195|\199[\203\248\194Z\211\190\227\255\239\255w\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255{\223\239\157\227\025\207\025\203\026\199\026\199;\199<\199\092\203}\203|\199|\199|\199|\199\157\199\157\199\157\195\157\195|\191|\191|\191|\195|\195|\199|\199|\199\092\199<\199;\199;\199;\203;\207[\211[\219\255\239\255\247\255{\255\127\254\127\254\127\254\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255\247\255\243\255\243\255\239\255\227\190\215}\207\092\199;\195;\195|\199{\195|\195|\195|\195|\195|\191|\191\157\199\157\199\157\195|\195{\195[\195[\195[\195;\195<\199}\203\190\215\223\223\255\231\255\239\255\243\255\247\255w\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\255\127\255\127\255\127\255\127\222{\255{\255\127\255{\255{\255{\255w\255\247\255\243\255\239\223\239|\227\092\223[\219;\215;\215;\211;\211;\211\026\207\026\207:\211;\215\092\219}\227\157\231\158\231\223{\255\127\255\127\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255{\255{\255{\255{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\127\222\127\255\127\255\127\255\127\255\127\255{\255{\255{\255{\255{\255w\255w\255\247\255\247\255\247\255\247\255\247\255\243\255\243\255\243\255\243\255\243\255\243\255\243\255\243\255\243\255\243\255\247\255\247\255w\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255{\254{\254w\254{\255{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127"),

    green = image.new("^\000\000\000^\000\000\000\000\000\000\000\188\000\000\000\016\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\160\001\160\001\160\001\192\001\160\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\005\225\009$\022\136&\205\1822\203v\219\184\227\218\239\220\243\220\247\221\247\253\247\253\247\253\247\253\247\253\247\221\247\220\243\219\239\185\231v\2193\207\206\186\137&$\022\193\009\160\005\160\001\160\001\160\001\160\001\160\001\160\001\160\001\128\001`\001\000\001\160\000\192\000\192\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\193\005\002\014f\030\203\1780\199\150\219\185\235\220\243\253\247\253\247\253\247\221\247\253\247\253\247\253\247\253\247\221\247\221\247\253\247\253\247\253\247\253\247\221\247\253\247\253\247\253\247\220\243\219\239\151\2232\203\203\178F\030\226\013\160\005\160\001\160\001\160\001\160\001\160\001\160\001\128\001@\001\000\001\128\000 \000`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\005\225\009E\022\170\174/\195t\215\185\231\219\243\221\247\221\247\221\247\221\247\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\220\243\218\235\150\2190\199\170\174%\022\193\009\160\005\160\001\160\001\160\001\160\001\160\001\128\001@\001\224\000`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\005\192\005\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\005\002\014f\030\236\182Q\199\149\215\185\231\219\243\220\247\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\247\220\243\218\235\151\223Q\203\236\182F\030\226\013\160\005\160\001\160\001\160\001\160\001\128\001`\001\000\001\128\000`\000`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\002\014f\158\236\182P\199s\207\183\227\219\239\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\185\235\149\215Q\199\236\182F\158\226\013\160\001\160\001\160\001\160\001\160\001\128\001@\001\192\000@\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\225\009f\026\236\178O\195Q\199\149\215\186\235\220\243\220\243\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\220\243\219\239\151\223r\207O\195\235\178F\026\193\009\160\001\160\001\160\001\160\001\128\001`\001\224\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\224\005$\022\202\170.\191O\195Q\199\150\219\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\219\239\184\231s\207O\195.\191\170\170\004\018\160\005\128\001\160\001\160\001\160\001`\001\000\001 \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001\225\009\135\158\012\183O\191.\191Q\199\151\223\219\239\219\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\186\239\219\239\185\231s\207O\191N\191\012\183f\158\193\009\128\001\160\001\160\001\160\001\128\001 \001`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\001#\018\201\170-\187.\191-\187P\199\151\227\186\239\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\239\185\231s\207.\187.\187-\187\169\170\003\018\128\001\128\001\160\001\160\001\128\001 \001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\192\005E\022\235\174-\187-\187,\183O\195\151\223\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\185\231r\207-\187-\187-\187\235\174%\022\160\001\128\001\160\001\160\001\128\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\225\005\134\158\011\179-\183,\183\012\183.\191\149\219\186\235\185\235\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\186\235\152\227P\199,\183,\183-\183\011\179f\158\160\005\128\001\160\001\160\001\128\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\192\001\192\001\192\001\225\009\167\162\011\179,\183\012\179\012\179,\183s\211\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\150\219.\191\011\179\012\179,\183\011\179\135\158\193\005\128\001\160\001\160\001\128\001 \001`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\224\001\192\001\192\001\192\001\001\010\167\162\011\179\011\179\011\179\011\179\011\179P\195\151\227\184\231\152\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\227\184\231\185\231s\211\012\179\011\179\011\179\011\179\011\179\135\162\193\009\128\001\160\001\160\001\128\001 \001@\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\224\001\224\001\192\001\001\010\167\162\011\175\011\179\011\175\011\175\010\175\012\183u\215\184\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\151\227/\195\010\175\011\175\011\175\011\179\011\175\135\162\193\009\128\001\160\001\160\001\128\001\000\001\000\000\000\000\160\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001 \010\192\001\192\001\224\001\224\001\192\001\225\005\166\158\010\175\010\175\010\175\010\175\010\175\010\175P\199\151\227\151\227\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\152\227t\211\011\179\010\175\010\175\010\175\010\175\010\175\134\158\192\005\128\001\160\001\160\001`\001\192\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\192\001\224\001\224\001\192\001\224\005\133\154\234\170\010\171\010\171\010\171\010\171\233\170\011\179t\211\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\150\223.\191\233\170\010\171\010\171\010\171\010\175\233\170e\154\160\005\128\001\160\001\160\001@\001\160\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\224\001\192\001\192\001\224\001\224\001\192\001\224\001d\150\233\170\233\170\233\170\233\170\233\170\233\170\233\170.\187\150\219\150\223\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\151\223R\203\234\170\233\170\233\170\233\170\233\170\009\171\233\166D\150\160\001\160\001\160\001\160\001 \001`\000\000\000\000\000\000\000\000\000\000\000\000\000\192\001\192\001\192\001\224\001\224\001\224\001\192\001C\018\232\162\233\166\232\166\232\166\232\166\232\166\232\166\233\170Q\199\150\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219\150\219t\215\011\179\232\166\232\166\232\166\232\166\232\166\233\170\200\162\003\014\128\001\128\001\160\001\128\001\224\000 \000\000\000\000\000\000\000\000\000\192\001\128\001\192\001\192\001\224\001\224\001\192\001\002\010\198\158\232\166\232\166\232\166\232\166\232\166\232\166\232\166\233\170s\207u\219u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215-\187\231\162\232\166\232\166\232\166\232\166\232\166\232\166\166\158\225\009\128\001\160\001\160\001`\001\160\000\000\000\000\000\000\000\224\001\192\001\192\001\224\001\224\001\224\001\192\001\224\005\133\154\231\162\231\162\231\162\231\162\231\162\231\162\231\162\231\162\011\179t\211t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215t\215u\215/\195\231\162\231\162\231\162\231\162\231\162\231\162\231\162\231\162d\150\160\005\128\001\160\001\160\001 \001@\000\000\000\000\000\224\001\224\001\224\001\224\001\224\001\224\001\224\001C\146\231\162\231\162\231\162\231\162\231\162\231\162\231\162\231\162\198\158\012\183t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\215P\199\232\166\231\162\231\162\231\162\231\162\231\162\231\162\231\162\199\162#\142\160\001\128\001\160\001\128\001\192\000 \000\000\000\192\001\224\001\224\001\224\001\224\001\224\001!\010\165\154\230\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158-\187t\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211s\211Q\203\231\162\198\158\198\158\198\158\198\158\198\158\198\158\198\158\230\158\165\154\225\009\128\001\160\001\128\001@\001`\000\000\000\192\001\192\001\224\001\224\001\224\001\224\005\131\146\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158-\187s\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207s\207Q\203\231\162\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\230\158c\146\160\001\128\001\160\001\128\001\224\000 \000\192\001\224\001\224\001\224\001\224\001!\010\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154-\187R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207R\207Q\199\199\162\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\001\010\128\001\128\001\128\001`\001`\000\192\001\224\001\224\001\224\001\000\006\131\146\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\150\012\183R\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\203Q\2030\195\198\158\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154\197\154c\146\192\005\128\001\160\001\128\001\224\000\224\001\224\001\224\001\224\001A\138\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\010\175Q\203Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199Q\199.\191\197\154\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\196\150\001\138\128\001\160\001\128\001`\001\224\001\224\001\224\001\224\005\131\146\196\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\163\146\232\1660\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\1990\199\012\183\196\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\196\150\196\150b\142\160\001\128\001\160\001\128\001\224\001\224\001\224\001!\010\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\198\158/\1910\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\195/\1950\195\234\174\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\001\006\128\001\160\001\128\001\224\001\224\001\224\001b\142\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\164\150\012\183/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191/\191\199\162\162\142\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146\163\146B\142\160\001\160\001\160\001\224\001\224\001\000\006\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\233\170.\191\014\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191.\191\012\183\196\150\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\130\142\224\005\128\001\160\001\224\001\224\001A\138\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\197\154\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\013\187\014\187\233\166\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142\162\142!\134\128\001\160\001\224\001\224\001\129\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\162\142\234\170\013\187\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\183\013\187\012\183\196\150\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138\161\138a\138\160\001\128\001\224\001 \006\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\197\154\012\179\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\012\183\232\166\161\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\224\001\128\001\224\001@\134\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\134\161\138\200\166\012\179\011\179\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\235\178\011\179\234\174\163\146\128\134\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\128\138\160\138 \134\128\001\224\001`\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\163\146\233\170\235\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\234\174\197\154\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134@\134\160\001\000\002\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\164\150\233\170\234\174\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\170\234\174\199\162\129\138\128\130\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134\128\134`\134\192\001 \002\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\165\154\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\233\170\199\162\130\142\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\224\001@\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\129\134\165\154\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\200\166\199\158\162\142\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\000\130@\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\129\134\164\150\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\199\162\198\158\130\142\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130 \130`\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\164\146\198\158\199\162\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\198\158\199\158\199\158\165\154\129\138\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130@\130`\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\162\142\165\154\166\158\166\158\165\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\158\166\154\163\146\129\138\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130`\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\129\138\163\146\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\165\154\164\150\130\142\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130`\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\130\142\163\146\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\164\150\130\142\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\129\138\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\131\146\130\142\128\138\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\129\138\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\130\142\129\138\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\128\134\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\129\138\128\134\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\134\128\134\128\134\128\134\128\134a\134a\134a\134a\134a\134a\134a\134a\134\128\134\128\134\128\134\128\134\128\134\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\160\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\160\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\160\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\128\130\160\130\128\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\128\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\128\130\128\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\128\130\128\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130\160\130`\130`\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130@\130@\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130 \130 \002\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\000\002\000\002\160\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\160\130\192\001\224\001\128\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130`\130\160\001\224\001@\130\224\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\192\130\224\130 \130\128\001\224\001 \002\192\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\192\130\224\001\128\001\224\001\224\001\160\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\128\130\160\001\128\001\224\001\224\001`\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130 \130\128\001\160\001\224\001\224\001\000\002\192\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\192\130\192\001\128\001\160\001\224\001\224\001\224\001\128\130\000\131\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\224\130\000\131`\130\160\001\160\001\160\001\224\001\224\001\224\001 \002\224\130\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\224\130\224\001\128\001\160\001\128\001\224\001\224\001\224\001\224\001\160\130\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\128\130\160\001\128\001\160\001`\001\192\001\224\001\224\001\224\001 \130\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\130\128\001\160\001\128\001 \001`\001\224\001\224\001\224\001\224\001\160\130 \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\131 \131\128\130\160\001\128\001\160\001\128\001\128\000\224\000\160\001\224\001\224\001\224\001 \002\000\131 \131\000\131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131\000\131 \131\224\130\224\001\128\001\160\001\128\001 \001 \000\224\000`\001\192\001\224\001\224\001\224\001\128\130 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131`\130\160\001\128\001\160\001`\001\128\000\000\000`\000\224\000\160\001\224\001\224\001\192\001\000\002\224\130 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131@\131\192\130\192\001\128\001\160\001\128\001\000\001 \000\000\000\000\000\128\000@\001\192\001\224\001\224\001\224\001@\130\000\131@\131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131 \131@\131\000\131 \130\128\001\128\001\160\001`\001`\000\000\000\000\000\000\000\000\000\192\000\128\001\224\001\224\001\192\001\224\001\128\130@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131\128\130\160\001\128\001\160\001\128\001\192\000\000\000\000\000\000\000\000\000`\000\000\000\000\001\160\001\224\001\224\001\192\001\000\002\192\130@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131\192\130\192\001\128\001\160\001\128\001 \001@\000\000\000\000\000\000\000\000\000\000\000\000\000@\000@\001\192\001\224\001\224\001\192\001 \002\000\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131\000\131\000\002\128\001\160\001\160\001`\001`\000\000\000\000\000\000\000\000\000\000\000\000\000 \000\000\000\160\000`\001\192\001\224\001\192\001\192\001`\130 \131`\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131@\131`\131 \131 \130\128\001\160\001\160\001\128\001\160\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\224\000\128\001\192\001\224\001\192\001\192\001\128\130@\131`\131@\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131@\131`\131@\131`\130\160\001\128\001\160\001\128\001\224\000 \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\128\001\192\001\224\001\192\001\224\001\128\130@\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131@\131`\130\160\001\128\001\160\001\128\001 \001@\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\000\000\000 \001\160\001\192\001\224\001\192\001\192\001\128\130@\131\128\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131`\131\128\131@\131`\130\160\001\128\001\160\001\160\001@\001`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\224\003\000\000 \001\160\001\192\001\192\001\192\001\224\001\128\130`\131\128\131`\131`\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131`\131`\131\128\131@\131`\130\160\001\128\001\160\001\160\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000@\001\160\001\192\001\192\001\192\001\224\001\128\130@\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131@\131`\130\160\001\128\001\160\001\160\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000@\000@\001\160\001\192\001\192\001\192\001\192\001`\130 \131\160\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131 \131@\130\160\001\128\001\160\001\160\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000@\000 \001\160\001\192\001\192\001\192\001\192\001 \002\000\131\128\131\160\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\128\131\160\131\128\131\224\130\000\002\128\001\128\001\160\001\160\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\128\001\192\001\192\001\192\001\192\001\000\002\160\130`\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131`\131\160\130\192\001\128\001\160\001\160\001\160\001@\001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\224\000\128\001\192\001\192\001\192\001\192\001\224\001`\130 \131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131 \131@\130\160\001\128\001\160\001\160\001\128\001 \001`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\192\000@\001\192\001\192\001\192\001\192\001\192\001\000\002\160\130`\131\192\135\192\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\160\131\192\131\192\135`\131\160\130\224\001\160\001\160\001\160\001\160\001\128\001\224\000@\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\000 \001\160\001\192\001\192\001\192\001\192\001\192\001 \002\224\130\128\131\192\135\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\135\128\131\224\130\000\002\160\001\160\001\160\001\160\001\160\001`\001\160\000 \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000@\000\192\000`\001\192\001\192\001\192\001\192\001\192\001\192\001@\130\000\131\160\131\192\135\192\135\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\131\192\135\192\135\128\131\224\130 \130\160\001\128\001\160\001\160\001\160\001\128\001\000\001`\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\000 \001\128\001\192\001\192\001\192\001\192\001\192\001\192\001 \002\224\130\128\131\192\135\224\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\192\135\224\135\192\135\128\131\224\130 \002\160\001\160\001\160\001\160\001\160\001\160\001`\001\160\000 \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000 \000\160\000@\001\160\001\192\001\192\001\192\001\192\001\192\001\192\001\000\002\128\130 \131\160\131\192\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\192\135\160\131 \131\128\130\224\001\160\001\160\001\160\001\160\001\160\001\160\001`\001\000\001@\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000@\000\192\000`\001\160\001\192\001\192\001\192\001\192\001\192\001\160\001\192\001 \002\128\130 \131\128\131\160\131\192\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\224\135\192\135\160\131\128\131\000\131\128\130\000\002\160\001\160\001\160\001\160\001\160\001\160\001\160\001\128\001 \001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000 \000\192\000`\001\160\001\160\001\192\001\192\001\192\001\192\001\160\001\160\001\192\001\224\001@\002\160\130\224\130@\131`\131\128\131\160\131\160\131\192\131\192\135\192\135\192\135\192\135\192\131\160\131\160\131\128\131`\131@\131\224\130\128\130 \002\224\001\160\001\128\001\160\001\160\001\160\001\160\001\160\001\128\001`\001 \001\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"),


}
