-- Count down timer
-- John Powers  2011-02-12
-- 
-- Copyright (C) 2011 Texas Instruments
-- All rights reserved



time_remaining = 0 -- seconds


Color = {
    black = {   0,    0,    0},
    red   = {0xff,    0,    0},
    green = {   0, 0xff,    0},
    blue  = {   0,    0, 0xff},
    white = {0xff, 0xff, 0xff},
}


------------------------------------------------------------------ Utilities

function string:ischar1(n)
    local b = self:byte(n)
    return b and (b <= 0x7F or b >= 0xC0)
end


function clearWindow(gc, color)
    local w, h = platform.window:width(), platform.window:height()

    gc:setColorRGB(unpack(color))
    gc:fillRect(0, 0, w, h)
end






------------------------------------------------------------------ View
View = class()


function View:init(parent)
    self.children = {}
    if parent then
        table.insert(parent.children, self)
    end
end


function View:valid(content)
    return true
end


function View:isVisible()
    return self.visible
end


function View:mouseDown(x, y)
    for _, view in ipairs(self.children) do
        if view:isVisible() and view:contains(x, y) and view:mouseDown(x, y) then
            return true
        end
    end
    return false
end


function View:contains(x, y)
    return x >= self.x and x <= self.x + self:width() and
           y >= self.y and y <= self.y + self:height()
end


function View:width()
    return self.w
end


function View:height()
    return self.h
end


function View:paint(gc)
    -- Implemented in subclasses
end


function View:charIn(ch)
    if self.focus then
        self.focus:charIn(ch)
    end
end


function View:acceptsEnter()
    return false
end


function View:enterKey()
    if self.focus then
        if self.focus:acceptsEnter() then
            self.focus:enterKey()
        else
            self:defaultEnterHandler()
        end
    end
end


function View:escapeKey()
    -- Implemented in subclasses
end


function View:defaultEventHandler()
    -- Implemented in subclasses
end


function View:backspaceKey()
    if self.focus then
        self.focus:backspaceKey()
    end
end


function View:deleteKey()
    if self.focus then
        self.focus:deleteKey()
    end
end


function View:clearKey()
    if self.focus then
        self.focus:clearKey()
    end
end


function View:arrowLeft()
    if self.focus then
        self.focus:arrowLeft()
    end
end


function View:arrowRight()
    if self.focus then
        self.focus:arrowRight()
    end
end


function View:timer()
    -- Implemented in subclasses
end


function View:close()
    -- Implemented in subclasses
end







------------------------------------------------------------------ ImageButton
ImageButton = class(View)

function ImageButton:init(parent, image, command)
    View.init(self, parent)
    self.image = image
    self.x = 0
    self.y = 0
    self.visible = false
    self.command = command
end


function ImageButton:setxy(x, y)
    self.x = x
    self.y = y
end


function ImageButton:setwh(w, h)
    w = math.floor(w + 0.5)
    h = math.floor(h + 0.5)
    if self.image:width() ~= w or self.image:height() ~= h then
        self.image = self.image:copy(w, h)
    end
end


function ImageButton:width()
    return self.image:width()
end


function ImageButton:height()
    return self.image:height()
end


function ImageButton:setVisible(vis)
    self.visible = vis
end


function ImageButton:mouseDown(x, y)
    self:execute()
    return true
end


-- Runs command callback function when invoked
function ImageButton:execute()
    self.command()
end

function ImageButton:paint(gc)
    if self.visible then
        gc:drawImage(self.image, self.x, self.y)
    end
end






------------------------------------------------------------------ TextInputView
TextInputView = class(View)


function TextInputView:init(parent, fontsize)
    View.init(self, parent)
    self.content    = ""
    self.fontsize   = fontsize
    self.cursor     = 0
    self.focused    = true
    self.textcolor  = Color.black
    self.bkgcolor   = Color.white
end


function TextInputView:setxyw(x, y, w)
    self.x = x
    self.y = y
    self.w = w
end


function TextInputView:charIn(ch)
    local cursor = self.cursor
    local content = self.content
    if cursor > #content then
        self.content = content .. ch
    else
        self.content = content:sub(1, cursor-1) .. ch .. content:sub(cursor)
    end
    self.cursor = cursor + #ch
end


function TextInputView:cursorLeft()
    local cursor = self.cursor
    local content = self.content

    if cursor > 1 then
        repeat
            cursor = cursor - 1
        until content:ischar1(cursor)
        self.cursor = cursor
        return true
    end
    return false
end

TextInputView.arrowLeft = TextInputView.cursorLeft


function TextInputView:cursorRight()
    local cursor = self.cursor
    local content = self.content

    if cursor <= #content then
        repeat
            cursor = cursor + 1
        until cursor > #content or content:ischar1(cursor)
        self.cursor = cursor
        return true
    end
    return false
end

TextInputView.arrowRight = TextInputView.cursorRight


function TextInputView:del()
    local content = self.content
    local cursor  = self.cursor

    if cursor <= #content then
        local nc = cursor
        repeat
            nc = nc + 1
        until nc > #content or content:ischar1(nc)
        self.content = content:sub(1, cursor-1) .. content:sub(nc)
    end
end

TextInputView.deleteKey = TextInputView.del


function TextInputView:delLeft()
    local content   = self.content
    local cursor    = self.cursor
    if self:cursorLeft() then
        self:del()
    end
end

TextInputView.backspaceKey = TextInputView.delLeft


function TextInputView:clear()
    self.content = ""
    self.cursor = 1
end

TextInputView.clearKey = TextInputView.clear


function TextInputView:paint(gc)
    local textcolor = self.textcolor
    local bkgcolor  = self.bkgcolor
    local style = "r"
    if not self:valid() then
        textcolor = Color.white
        bkgcolor  = Color.red
        style = "b"
    end
    gc:setFont("sansserif", style, self.fontsize)
    local height = gc:getStringHeight(self.content)
    self.h = height
    gc:setColorRGB(unpack(bkgcolor))
    gc:fillRect(self.x-2, self.y-2, self.w+4, height+4)

    gc:setColorRGB(unpack(Color.black))
    gc:drawRect(self.x-2, self.y-2, self.w+4, height+4)
    gc:clipRect("set", self.x, self.y, self.w, height)
    gc:setColorRGB(unpack(textcolor))
    local newx = gc:drawString(self.content, self.x, self.y, "top")

    if self.focused then
        local cx = gc:getStringWidth(self.content:sub(1, self.cursor-1)) + self.x
        local cy = self.y
        gc:drawLine(cx, cy, cx, cy + height)
    end
    gc:clipRect("reset")
    return newx
end





------------------------------------------------------------------ TimeInputView
TimeInputView = class(TextInputView)


function TimeInputView:init(parent, initialvalue, ...)
    TextInputView.init(self, parent, ...)
    self.initialvalue = initialvalue
    self.content = initialvalue
    self.cursor  = #initialvalue + 1
end


function TimeInputView:clear()
    self.content = self.initialvalue
    self.cursor = #self.content
end


function TimeInputView:valid()
    local min, sec = self.content:match("^(%d+):(%d%d)$")
    if min and sec then
        sec = tonumber(sec)
        return sec >= 0 and sec < 60
    end
    return false
end


function TimeInputView:toSeconds()
    local minutes, seconds = self.content:match("^(%d+):(%d+)$")
    return tonumber(minutes or 0)*60 + tonumber(seconds or 0)
end






------------------------------------------------------------------ RunningClockView
RunningClockView = class(View)


function RunningClockView:init(time)
    View.init(self)
    self.time = time
    self.running = true
    self.startButton = ImageButton(self, startButton,
        function()
            self.running = true
            self.pauseButton:setVisible(true)
            self.startButton:setVisible(false)
            timer.start(1)
        end
    )
    self.pauseButton = ImageButton(self, pauseButton,
        function()
            self.running = false
            self.pauseButton:setVisible(false)
            self.startButton:setVisible(true)
            timer.stop()
        end
    )
    self.pauseButton:setVisible(true)
    self.stopButton  = ImageButton(self, stopButton,
        function()
            self:close()
            theView = SetClockView()
        end
    )
    self.stopButton:setVisible(true)
    timer.start(1)
end


function RunningClockView:charIn(ch)
    if ch == "s" then
        self.stopButton:execute()
    elseif ch == "p" then
        self.pauseButton:execute()
    end
end


function RunningClockView:enterKey()
    if self.running then
        self.pauseButton:execute()
    else
        self.startButton:execute()
    end
end


function RunningClockView:escapeKey()
    self.stopButton:execute()
end


function RunningClockView:close()
    timer.stop()
end


function RunningClockView:toMmSs()
    local minutes = math.floor(self.time / 60)
    local seconds = self.time % 60
    return ("%d:%02d"):format(minutes, seconds)
end


function RunningClockView:timer()
    self.time = self.time - 1
    if self.time <= 0 then
        timer.stop()
    end
    platform.window:invalidate()
end


function RunningClockView:switchToSetClockView()
    self:close()
    theView = SetClockView()
end


function RunningClockView:paint(gc)
    clearWindow(gc, Color.black)
    gc:setColorRGB(unpack(Color.white))
    gc:setFont("sansserif", "b", 144)
    local now = self:toMmSs()
    local tw = gc:getStringWidth(now)
    local th = gc:getStringHeight(now)
    local ww = platform.window:width()
    local wh = platform.window:height() 
    gc:drawString(now, (ww - tw)/2, (wh - th)*.4, "top")

    local bsize = math.floor(math.min(ww, wh)/5 + 0.5)
    local b
    if self.running then
        b = self.pauseButton
    else
        b = self.startButton
    end
    b:setwh(bsize, bsize)
    b:setxy(ww/2 - bsize - 10, wh - bsize - 10)
    b:paint(gc)
    self.stopButton:setwh(bsize, bsize)
    self.stopButton:setxy(ww/2 + 10, wh - bsize - 10)
    self.stopButton:paint(gc)
    
end






------------------------------------------------------------------ SetClockView
SetClockView = class(View)

function SetClockView:init(parent)
    View.init(self, parent)
    self.x = 0
    self.y = 0
    self.w = platform.window:width()
    self.h = platform.window:height()

    self.fontsize = 12
    self.inputfield = TimeInputView(self, "10:00", self.fontsize)
    self.focus = self.inputfield
    self.startButton = ImageButton(self, startButton, function() self:switchToRunningView() end)
    self.startButton:setVisible(true)
end


function SetClockView:defaultEnterHandler()
    if self.startButton:isVisible() then
        self.startButton:execute()
    end
end


function SetClockView:switchToRunningView()
    self:close()
    theView = RunningClockView(self.inputfield:toSeconds())
end


function SetClockView:paint(gc)
    clearWindow(gc, Color.white)

    gc:setFont("sansserif", "r", self.fontsize)
    gc:setColorRGB(unpack(Color.black))
    local x = gc:drawString("Enter count down time ", 10, 10, "top")

    -- Display input box
    self.inputfield:setxyw(x, 10, 60)
    self.inputfield:paint(gc)

    gc:setColorRGB(unpack(Color.black))
    gc:setFont("sansserif", "r", self.fontsize)
    gc:drawString(" mm:ss", x + 60, 10, "top")

    local b = self.startButton
    if self.inputfield:valid() then
        local ww = platform.window:width()
        local wh = platform.window:height()
        local min = math.min(ww, wh)
        self.startButton:setwh(min/3, min/3)
        b:setVisible(true)
        local w = b:width()
        local h = b:height()
        b:setxy((ww - w)/2, wh - h - 10)
        b:paint(gc)
    else
        b:setVisible(false)
    end
end






------------------------------------------------------------------ Event Handlers

function on.paint(gc)
    theView:paint(gc)
end


function on.charIn(ch)
    theView:charIn(ch)
    platform.window:invalidate()
end


function on.enterKey()
    theView:enterKey()
    platform.window:invalidate()
end


function on.escapeKey()
    theView:escapeKey()
    platform.window:invalidate()
end


function on.backspaceKey()
    theView:backspaceKey()
    platform.window:invalidate()
end


function on.deleteKey()
    theView:deleteKey()
    platform.window:invalidate()
end


function on.clearKey()
    theView:clearKey()
    platform.window:invalidate()
end


function on.arrowLeft()
    theView:arrowLeft()
    platform.window:invalidate()
end


function on.arrowRight()
    theView:arrowRight()
    platform.window:invalidate()
end


function on.mouseDown(x, y)
    theView:mouseDown(x, y)
    platform.window:invalidate()
end


function on.create()
    theView = SetClockView()
    platform.window:invalidate()
end


function on.timer()
    theView:timer()
    platform.window:invalidate()
end






------------------------------------------------------------------ Images


startButton = image.new(
"P\000\000\000P\000\000\000\000\000\000\000\160\000\000\000\016\000\001\000$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145$\145$\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145\004\145$\145$\145$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145$\145\004\145\004\145$\145D\145d\145\132\145\163\141\195\141\195\141\195\141\227\141\227\141\195\141\195\141\195\141\163\141\132\145d\145D\145$\145\004\145\004\145$\145$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145\004\145\004\145$\145d\145\164\145\227\141\004\146F\150\137\162\204\174\239\186\017\1992\2033\2073\2072\203\017\199\239\186\204\174\137\162F\150$\146\227\141\164\145d\145$\145\004\145\004\145$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\004\145$\145d\145\196\145\003\142E\150\170\166\016\195w\219\187\239\254\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\254\251\187\239w\219\016\195\170\166E\150\003\142\196\145d\145$\145\004\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\004\145$\145\133\145\004\146$\142\136\162\016\195\153\231\254\251\255\255\255\255\255\255\254\251\254\251\254\251\254\251\254\251\254\251\254\251\254\251\254\251\254\251\254\251\254\251\255\255\255\255\255\255\254\251\153\231\016\195\136\162$\142\228\145\133\145$\145\004\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145\004\145$\145\133\149\004\146$\142\137\1624\207\220\243\255\255\254\251\222\251\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\222\251\254\251\255\255\220\2434\207\137\162$\142\004\146\133\149$\145\004\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\004\145e\145\005\150$\142f\154\018\203\220\243\254\251\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\254\251\220\243\018\203f\154$\142\005\150e\145\004\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145$\145\197\149E\146$\142\171\174\153\231\254\251\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\220\243\254\251\153\231\171\174$\142E\146\197\149$\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145D\145\005\150D\146$\146\016\195\220\243\220\243\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\220\243\220\243\016\195$\146D\146\005\150d\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\133\145%\150D\146E\1503\207\220\243\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\187\239\220\2433\207E\150D\146%\150\133\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\165\145E\150E\146%\1503\207\188\243\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\186\235\188\2433\207%\150E\146E\150\165\145$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\197\149E\150E\150$\146\018\199\187\239\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\187\239\018\199$\146E\150E\150\197\149$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145$\145\165\149E\150E\150$\146\205\182\186\235\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\153\231\186\235\205\182$\146E\150E\150\197\149$\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145\165\145E\150E\150$\146h\162x\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227x\227h\162$\146E\150E\150\165\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145\132\145E\150E\150E\150$\146\018\203\152\227w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223\152\227\018\203$\146E\150E\150E\150\133\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145$\145d\145%\150E\150E\150$\146\137\166w\223w\223w\223w\223w\223w\223w\223W\2237\223W\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223w\223\137\166$\146E\150E\150%\150d\145$\145$\145$\145$\145$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\017$\145$\145$\145D\145\005\150E\150E\150E\150$\146\017\199w\223v\219v\219v\219v\219v\219V\219\023\219\023\219\023\219\023\2196\219V\219V\219V\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219w\223\017\199$\146E\150E\150E\150\005\150D\145$\145$\145$\145$\017$\017$\017$\017$\017$\017D\017D\017D\017D\017D\017D\145D\145D\145$\145\197\149E\150E\150E\150$\146G\154U\215V\219U\215U\215U\215U\215U\215V\215\246\218\246\218\246\218\246\218\246\218\022\2195\215U\215V\219V\219V\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219V\219V\219V\219V\219V\219V\219V\215V\215U\215U\215U\215U\215U\215U\215U\215U\215U\215V\219U\215G\154$\146E\150E\150E\150\197\149$\145D\145D\145D\145D\017D\017D\017D\017D\017D\017D\017D\017D\017D\145D\145D\145$\145d\145E\150E\150E\150E\150$\146\170\170V\215U\215U\215U\215U\215U\215U\2155\215\246\214\246\214\246\214\245\214\246\214\246\214\246\214\021\2155\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215V\215\170\170$\146E\150E\150E\150E\150d\145$\145D\145D\145D\145D\017D\017D\017D\017D\017D\017D\017D\017D\145D\145D\145D\145\005\150E\150E\150E\150E\150$\146\205\182U\215T\211T\211T\211T\211T\211T\2114\211\213\214\213\214\213\214\245\214\213\210\213\210\213\214\213\214\245\210\021\2114\211U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\215U\211U\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211U\215\205\182$\146E\150E\150E\150E\150\005\150D\145D\145D\145D\145D\017D\017D\017D\017D\017D\017D\017D\145D\145D\145$\145\165\145E\150E\150E\150E\150E\150$\146\239\190T\2113\2073\2073\2073\2073\2073\2074\207\212\210\212\210\212\210\154\235X\227\212\210\180\206\212\210\212\210\212\210\244\210\020\2114\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\2114\2114\2114\2114\2073\2073\2073\2073\2073\207T\211\239\190$\146E\150E\150E\150E\150E\150\165\145$\145D\145D\145D\145D\017D\017D\017D\017D\017D\017D\145D\145D\145D\145\005\150E\150E\150E\150E\150E\150%\150\240\1943\2073\2073\2073\2073\2073\2073\207\019\207\180\206\180\206\180\206\187\239\254\251\187\239\022\219\179\206\179\206\180\206\180\206\180\206\244\2063\2074\2114\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\2114\2073\2073\2073\2073\2073\2073\2073\2073\2073\2073\2073\2073\207\240\194%\150E\150E\150E\150E\150E\150\005\150D\145D\145D\145D\145D\017D\017D\017D\017D\017D\145D\145D\145$\145\165\145E\150E\150E\150E\150E\150E\150E\150\240\1943\203\018\203\018\2032\2032\2032\2032\203\019\203\179\206\179\206\147\202\187\235\254\247\254\247\253\247y\231\212\210\146\202\147\202\179\206\147\206\179\206\243\2063\2073\2073\207S\207S\207T\207T\207S\207S\207S\207S\2073\2073\2073\2073\2073\2073\2073\2073\2073\2073\2032\2032\2032\203\018\203\018\2033\203\240\194E\150E\150E\150E\150E\150E\150E\150\165\145$\145D\145D\145D\145D\017D\017D\017D\017D\145D\145D\145D\145\005\150E\150E\150E\150E\150E\150E\150%\146\239\186\018\203\017\199\018\199\018\203\018\2032\2032\203\018\203\147\202\147\202\146\202\187\235\254\247\253\247\253\247\254\247\253\2437\223\147\202r\198\146\202\147\202\147\202\178\202\243\2023\2073\2073\2073\2073\2073\207S\2073\2073\2073\2073\2073\2073\2073\2032\2032\2032\2032\2032\2032\203\018\203\018\203\018\199\017\199\018\203\239\186%\146E\150E\150E\150E\150E\150E\150\005\150D\145D\145D\145D\145D\017D\017D\017D\017D\145D\145D\145\132\145E\150E\150E\150E\150E\150E\150E\150%\146\205\182\017\199\017\199\017\199\017\199\017\199\017\199\017\199\018\199\146\198r\198r\198\186\235\254\247\253\247\253\247\253\247\253\247\254\247\187\239\245\214q\198q\198\146\198r\198\146\198\178\202\018\2032\2032\2033\2073\2073\2073\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\203\018\199\017\199\017\199\017\199\017\199\017\199\017\199\205\182$\146E\150E\150E\150E\150E\150E\150E\150\132\145D\145D\145D\145D\017D\017D\017D\145D\145D\145D\145\229\149E\150E\150E\150E\150E\150E\150E\150$\146\171\174\017\199\016\195\017\195\017\199\017\199\017\199\017\199\241\198q\198q\198q\194\186\235\253\247\253\243\253\243\253\243\253\243\253\243\253\247\253\247y\231\179\202Q\194q\194q\198q\198q\198\209\198\018\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\2031\1991\199\017\199\017\199\017\199\017\199\017\199\017\195\016\195\017\199\171\174$\146E\150E\150E\150E\150E\150E\150E\150\229\149D\145D\145D\145D\145D\017D\017D\145D\145D\145D\145%\150E\150E\150E\150E\150E\150E\150E\150$\146h\162\016\195\240\194\016\195\016\195\016\195\016\195\017\195\241\194q\194Q\194P\194\154\231\253\247\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\247\253\2437\223q\198P\190Q\194Q\194Q\194q\194\209\198\017\1992\1992\2032\2032\2032\1991\1991\1991\1991\199\017\199\017\199\017\199\017\199\017\195\016\195\016\195\016\195\016\195\016\195\136\162%\146E\150E\150E\150E\150E\150E\150E\150%\150D\145D\145D\145D\145D\017D\017D\145D\145D\145\132\145E\150E\150E\150E\150E\150E\150E\150E\150E\150F\154\238\186\240\194\240\190\016\195\016\195\016\195\016\195\240\194P\190P\190P\190\154\231\253\247\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\247\219\239\245\210P\190/\190P\190P\190P\190\144\194\241\194\017\199\017\1991\1991\1991\1991\1991\199\017\199\017\199\016\195\016\195\016\195\016\195\016\195\016\195\240\190\240\194\238\186F\154E\150E\150E\150E\150E\150E\150E\150E\150E\150\132\145D\145D\145D\145D\017D\145D\145D\145D\145\197\149E\150E\150E\150E\150E\150E\150E\150E\150E\150E\150\171\174\239\190\239\190\239\190\015\191\016\195\016\195\240\190O\190/\190/\186\153\231\253\247\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\243\253\247\153\231\146\202/\186/\186/\190/\190O\190\144\190\240\194\016\195\017\195\017\199\017\195\016\195\016\195\016\195\016\195\016\195\016\195\016\195\015\191\239\190\239\190\239\190\171\174E\150E\150E\150E\150E\150E\150E\150E\150E\150E\150\197\149D\145D\145D\145D\145D\145D\145D\145D\145\005\150E\150E\150E\150E\150E\150E\150E\150E\150F\150F\150h\158\238\186\239\190\239\190\239\190\015\191\015\191\207\190/\186/\186/\186\153\231\253\247\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\253\243\253\247\253\2437\219P\190\014\182/\186/\186/\186O\186\175\190\240\194\016\195\016\195\016\195\016\195\016\195\016\195\016\191\015\191\015\191\239\190\239\190\239\190\239\186h\162F\150F\154E\150E\150E\150E\150E\150E\150E\150E\150\005\150D\145D\145D\145D\145D\145D\145D\145d\145%\150E\150E\150E\150E\150E\150E\150E\150F\150F\154f\154f\154\171\174\239\190\238\186\239\186\239\190\015\191\207\190.\182\014\182\014\182\153\227\253\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\253\243\253\247\219\239\244\210\014\182\013\182\014\182\014\182\014\182O\186\175\190\239\190\015\191\015\191\015\191\015\191\015\191\015\191\239\190\239\190\238\186\239\190\171\174f\154f\154F\154F\150E\150E\150E\150E\150E\150E\150E\150%\150d\145D\145D\145D\145D\145D\145D\145\132\145E\150E\150E\150E\150E\150E\150E\150E\150F\154F\154f\154f\154\135\158\205\182\238\186\238\186\238\186\238\186\206\186\014\182\014\182\013\182\153\227\253\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\253\243\253\247\153\231\146\198\237\177\237\177\014\182\014\182\014\182N\182\206\186\239\190\239\190\239\190\239\190\238\186\238\186\238\186\238\186\237\182\135\158f\154f\154f\154F\154F\150E\150E\150E\150E\150E\150E\150E\150\132\145D\145D\145D\145D\145D\145D\145\165\145E\150E\150E\150E\150E\150E\150E\150F\150F\154f\154g\154g\158g\158\169\166\238\182\238\186\238\186\238\186\206\186\237\177\237\177\237\177\153\227\253\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\252\243\253\243\253\243W\219/\186\204\173\237\177\237\177\237\177\013\178m\182\206\186\238\186\238\186\238\186\238\186\238\186\238\186\169\166g\158g\158g\154f\154F\154F\150E\150E\150E\150E\150E\150E\150E\150\165\145D\145D\145D\145D\145D\145D\145\197\149E\150E\150E\150E\150E\150E\150F\150F\154f\154f\154g\158g\158\135\158\135\158\170\170\238\182\238\182\238\186\205\182\236\177\236\173\236\173x\227\253\243\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\243\253\243\251\239\244\206\237\177\203\173\236\173\236\177\204\173\013\178\141\182\205\182\205\182\205\182\237\182\170\170\135\158\135\158g\158g\158f\154f\154F\154F\150E\150E\150E\150E\150E\150E\150\197\149D\145D\145D\145D\145D\145D\145\229\149E\150E\150E\150E\150E\150E\150F\150F\154f\154g\154g\158\135\158\135\158\136\162\136\162\203\174\237\182\237\182\173\182\204\173\204\173\204\173x\223\253\243\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\243\253\243\186\231\145\194\203\169\203\169\204\173\204\173\204\173,\174\140\178\205\182\170\170\136\162\136\162\136\158\135\158g\158g\154f\154f\154F\154E\150E\150E\150E\150E\150E\150\229\149D\145D\145D\145D\145D\145D\145\229\149E\150E\150E\150E\150E\150E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\136\162\168\162\203\174\237\182\173\182\203\169\171\169\171\169x\223\253\243\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\243\253\243W\219.\182\170\165\171\169\203\169\171\169\203\169\009\166G\158h\162h\162\136\158\135\158g\158g\158f\154f\154F\154F\150E\150E\150E\150E\150E\150\229\149D\145D\145D\145d\145d\145D\145\005\150E\150E\150E\150E\150E\150F\150F\154f\154g\154g\158\135\158\135\158\136\162\136\162\168\162\169\166\169\166\202\170\172\178\171\169\170\169\170\165x\223\252\243\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\253\243\252\239\244\206\204\173\170\165\170\165h\157'\153g\153'\158g\158g\158g\158g\158g\158g\154f\154f\154F\150E\150E\150E\150E\150E\150\005\150D\145d\145d\145d\145d\145D\145\005\150E\150E\150E\150E\150E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\169\166\201\166\137\166\137\161\138\165\138\165w\223\252\243\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\252\239\253\247\219\235.\182G\153G\153G\153'\153\167\153G\158G\158g\158g\158g\158g\158f\154f\154F\154E\150E\150E\150E\150E\150\005\150d\145d\145d\145d\145d\145D\145\005\150E\150E\150E\150E\150F\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\169\166\202\170\137\166G\153h\157i\161w\219\252\243\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\252\239\253\243\186\231\144\190h\157G\153G\153G\153G\153\166\153\006\154G\158g\158g\158g\158g\158g\158f\154f\154F\154F\154F\154F\154E\150\005\150d\145d\145d\145d\145d\145D\145\005\150E\150E\150E\150E\150F\150F\154f\154g\154g\158\135\158\136\158\136\162\136\162\169\162\169\166\169\166\201\166\202\170\137\166G\153G\153G\153w\219\252\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\252\243\251\239\211\202\137\161&\149G\153G\153G\153G\153\134\153\198\153\230\153'\158g\158g\158\135\158\136\158\135\158g\158g\158g\158g\158g\158g\158F\150\005\150D\145d\145d\145d\145d\145D\145\229\149E\150E\150E\150E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\169\166\202\170\202\170\137\166G\153G\153G\153V\219\252\239\251\235\251\235\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\251\239\252\239\252\2396\215\203\169&\149'\149G\153G\153G\153f\153\133\149\165\149\198\153\007\158G\158g\158\136\158\136\162\137\162\136\162\136\162\136\162\136\162h\162h\162h\162F\154\229\145D\145d\145d\145d\145d\145D\145\229\149E\150E\150E\150E\150F\154f\154g\154g\158\135\158\135\158\136\162\136\162\168\162\169\166\169\166\201\166\202\170\202\170\138\166G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\239\252\243\152\223-\178&\149&\149G\153G\153G\153g\153\134\153\166\153\166\153\198\153\007\158'\158h\162h\162\137\166\170\166\169\166\169\166\137\166\137\166\137\166\137\166\137\166\137\166F\154\229\145D\145d\145d\145d\145d\145D\145\197\149E\150E\150E\150F\150F\154f\154g\154g\158\135\158\136\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\239\252\243\218\231\144\194g\153&\149G\153G\153G\153G\153\134\153\166\153\198\153\199\157\231\157'\158H\162h\162\136\162\170\170\171\170\170\170\170\170\170\170\170\170\170\170\170\170\138\170\138\170\138\170F\154\196\145D\145d\145d\145d\145d\145d\145\165\149E\150E\150E\150F\150F\154f\154g\154g\158\135\158\136\162\136\162\168\162\169\166\169\166\169\166\202\170\202\170\202\170\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\239\251\239\243\206\137\161&\149G\153G\153G\153G\153\135\153\166\153\199\157\231\157\007\158(\162H\162h\162\137\162\136\162\170\170\204\174\203\174\203\174\203\174\171\174\171\174\171\174\171\174\171\174\171\174\170\170E\150\164\145d\145d\145d\145d\145d\145d\145\132\149E\150E\150E\150F\150f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\201\166\202\170\202\170\202\170\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\239\252\239V\215\235\169&\149&\149G\153G\153G\153g\153\167\157\199\157\231\157\008\162(\162I\166i\166\137\166\137\166\169\166\203\174\236\178\204\178\204\178\204\178\204\178\204\178\204\178\204\178\204\178\172\178\172\178\171\174E\150\132\145d\145d\145d\145d\145d\145d\145\132\145%\150E\150E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\201\166\202\170\202\170\234\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\239\252\243\153\227N\182G\153&\149G\153G\153G\153g\153\167\157\199\157\232\161\008\162(\166I\166i\166\138\170\170\170\170\170\170\170\236\178\237\182\237\182\237\182\237\182\237\182\237\182\205\182\205\182\205\182\205\182\205\182\205\182\170\170%\146\132\145d\145d\145d\145d\145d\145d\145d\145\005\150E\150E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\243\218\235\177\194h\157&\149G\153G\153G\153g\153\167\157\200\161\232\161\008\162)\166I\166\138\170\170\170\170\170\202\170\202\170\202\170\237\182\014\187\014\187\238\186\238\186\238\186\238\186\238\186\238\186\238\186\238\186\206\186\206\186\238\186\137\166\004\146d\145d\145d\145d\145d\145d\145d\145d\145\197\149E\150E\150F\154f\154f\154g\158\135\158\135\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\239\252\239\020\207\170\165&\149G\153G\153G\153G\153\135\157\200\161\232\161\009\166)\166J\170\138\170\171\174\203\174\203\174\203\174\203\174\235\174\014\187\015\191\015\191\015\191\015\191\015\191\015\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\136\162\196\145d\145d\145d\145d\145d\017d\145d\145d\145\164\145E\150E\150F\154f\154f\154g\158\135\158\136\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\235\251\235\252\239\252\239w\219\236\173&\149&\149G\153G\153G\153\135\157\200\161\232\165\009\166)\166J\170\139\174\171\174\203\178\236\178\236\178\235\174\235\174\013\1830\1950\1950\195\016\195\016\195\016\195\016\195\016\191\016\191\016\191\015\191\016\191\239\190\239\190\240\194\239\190F\154\164\145d\145d\145d\145d\017d\017d\145d\145d\145\132\145%\150E\150F\154f\154f\154g\158\135\158\136\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\251\235\251\239\252\243\185\227O\186G\153&\149G\153G\153G\153g\157\200\161\233\165\009\166*\170j\170\139\174\171\178\204\178\236\178\236\178\012\179\236\178\236\178/\1911\1991\1990\1990\1950\1950\1950\195\016\195\016\195\016\195\016\195\016\195\016\195\016\195\016\195\017\199\205\182%\146\132\145d\145d\145d\145d\017d\017d\145d\145d\145d\145\229\149E\150F\154f\154f\154g\158\135\158\136\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\251\235\251\235\252\239\251\235\210\198h\157&\149G\153G\153G\153g\153\168\161\233\165\009\170*\170J\174\139\174\172\178\204\178\237\182\013\183\013\183\012\183\012\179\013\1870\195R\203Q\1991\1991\1991\1991\1991\1991\1991\1991\1991\199\017\199\017\199\017\199\017\199\017\1992\203\170\170\228\145d\145d\145d\145d\145d\017d\017d\017d\145d\145d\145\165\149E\150F\154f\154f\154g\158g\158\135\158\136\162\136\162\169\162\169\166\169\166\202\170\202\170\202\170\235\174\138\170G\153G\153G\153V\219\252\239\251\235\251\235\252\239\252\2395\211\202\165&\149G\149G\153G\153G\153\136\157\233\165\009\170*\170K\174\139\174\172\178\205\182\237\182\013\183\013\183-\183\013\183-\187P\195R\203R\203R\203R\203R\203R\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\2032\203\017\199g\154\164\145d\145d\145d\145d\017d\017d\017d\017d\145d\145d\145d\145%\150f\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\201\166\202\170\202\170\234\174\138\170G\153G\153G\153V\219\252\239\252\239\252\243x\223\013\178&\149&\149G\153G\153G\153\135\157\200\161\009\166*\170J\174\139\174\172\178\205\182\237\182\014\187.\187.\187-\183-\187P\195s\207s\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\2073\2073\2073\2073\2073\2073\2073\207T\211\238\186\005\146d\145d\145d\145d\145d\017d\017d\017d\017d\145d\145d\145d\145\197\149f\154f\154f\154g\158g\158\135\158\136\162\136\162\168\162\169\166\169\166\201\166\202\170\202\170\234\170\138\170G\153G\153G\153w\219\253\243\218\231p\190G\153&\149G\153G\153G\153g\153\168\161\233\165\009\166J\170k\174\172\178\205\182\237\186\014\187.\187.\187-\187.\187P\195s\207t\211t\211s\207s\207s\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207S\207T\207T\211T\211T\211T\211\137\166\196\145d\145d\145d\145d\145d\017d\017d\017d\017d\017d\145d\145d\145\132\145%\150f\154f\154g\158g\158\135\158\136\158\136\162\168\162\169\166\169\166\169\166\202\170\202\170\202\170\138\170G\153G\153G\153V\215\244\206\137\161&\149G\153G\153G\153G\153\135\157\200\161\009\166)\170J\174\139\178\204\182\237\182\014\187.\187.\187N\187O\191q\203t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211t\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211T\211U\215\016\195%\150\132\145d\145d\145d\145d\017d\017d\017d\017d\017d\017d\145\132\145\132\145d\145\197\149f\154g\158\136\162\136\162\136\162\136\162\136\162\136\162\168\162\169\166\169\166\202\170\202\170\202\170\138\170G\153G\153G\153\169\161&\149'\149G\153G\153G\153\135\157\199\157\232\161\009\166J\170k\174\172\178\204\182\013\183\013\187.\187O\191q\199s\207\149\215\149\215\149\215\149\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215U\215U\215U\215U\215U\215U\215v\219\169\166\196\145d\145\132\145\132\145d\145d\017d\017d\017\132\017\132\017\132\017\132\017\132\145\132\145d\145d\145%\150g\158\137\166\137\166\170\170\170\170\170\170\170\170\170\170\169\166\169\166\201\166\202\170\202\170\137\166G\153G\153G\153G\149G\153G\153G\153g\153\167\157\199\161\232\161)\166J\170\139\174\172\178\237\182\014\187/\195Q\199s\207\149\215\149\215\149\215\149\215\149\215\149\215\149\215\149\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215u\215v\219v\219v\219v\219v\219v\219v\219w\223\017\199%\150d\145d\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145d\145\165\149f\154\137\166\170\170\170\170\171\174\203\174\204\178\236\178\237\182\237\182\237\182\236\178\236\178\171\174h\157H\153G\153G\153G\153G\153\135\153\199\157\232\161\009\166J\170\140\178\205\182\015\1911\199R\207t\211\149\215\149\219\149\219\149\215\149\215\149\219\149\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219v\219w\219w\223w\223w\223v\219\136\162\164\145d\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145d\145\229\149\135\158\171\174\171\174\204\174\204\178\205\178\237\182\238\182\238\186\015\191\015\191\015\191\239\190\014\182\237\177\237\177\237\177\013\178M\182m\182\142\186\175\190\208\194\018\2033\207T\211t\215u\215\149\215\149\215\149\215\149\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\219\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\152\227\153\231\238\186\228\145d\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145&\150\169\166\204\178\204\178\205\182\237\182\238\186\238\186\015\191\015\191\015\191\016\1950\195\176\190O\186o\190\143\190\144\194\176\194\209\198\241\202\018\2033\207T\211T\211t\215u\215\149\215\149\215\150\219\150\219\150\219\150\219\150\219\150\219\150\219\150\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\152\223\152\227\152\227\152\227\152\227\152\227\152\227\153\231T\211&\150\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145d\145\165\149f\154\171\174\237\182\238\182\238\186\238\186\015\191\015\191\016\195\016\1951\1951\199\017\199\017\199\240\198\208\194\209\198\241\198\242\202\019\2073\207T\211t\211u\215u\215\149\215\150\219\150\219\150\219\150\219\150\219\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\223\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\153\231\153\231\153\231\186\235\151\223\136\162\164\145d\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145d\145\197\149g\158\205\182\239\190\239\190\015\191\016\195\016\195\016\1951\1991\1991\1992\203\018\203\018\203\018\203\018\203\019\2073\207T\211T\211u\215u\215\149\215\150\219\150\219\150\219\150\219\151\223\151\223\151\223\151\223\151\223\151\223\151\223\151\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\152\227\153\231\153\231\153\231\153\231\153\231\185\231\185\231\185\231\186\235\186\235\171\174\196\145d\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145d\145\229\149\136\162\239\186\016\195\016\195\016\195\017\1991\1991\1992\2032\2032\2033\2073\2073\2073\207T\211T\211u\215u\215u\215\150\219\150\219\150\219\150\219\151\223\151\223\151\223\151\223\151\223\151\227\152\227\152\227\152\227\184\227\184\227\184\227\184\227\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\187\239\187\239\238\186\196\145d\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\229\149\169\166\016\195\017\199\017\1991\1992\2032\2032\203S\207S\207S\207T\211T\211T\211u\215u\215u\215v\219\150\219\150\219\150\219\151\223\151\223\151\223\151\223\151\227\152\227\152\227\184\227\184\227\184\227\184\227\185\231\185\231\185\231\185\231\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\220\243\015\191\228\145d\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\229\149\170\166\017\1992\2032\2032\203S\207S\207S\207T\211T\211T\211u\215u\215u\215v\219v\219\150\219\150\223\151\223\151\223\151\223\151\223\152\227\152\227\184\227\184\227\184\227\185\231\185\231\185\231\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\187\239\187\239\187\239\220\243\221\243\016\195\228\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\229\149\169\1661\199S\207S\207S\207T\211T\211T\211u\215u\215u\215v\219v\219\150\219\151\223\151\223\151\223\151\223\152\227\152\227\152\227\152\227\184\227\185\231\185\231\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\187\239\219\239\219\239\220\243\220\243\220\243\221\247\220\243\239\190\228\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\197\149\136\1621\199U\215T\211T\211U\215u\215u\215v\219v\219v\219\151\223\151\223\151\223\151\223\152\227\152\227\152\227\152\227\185\231\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\219\239\219\239\219\239\220\243\220\243\220\243\220\243\220\243\220\243\221\247\254\251\219\239\204\178\196\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\164\145G\158\015\187u\215v\219v\215v\219v\219v\219w\223\151\223\151\223\152\227\152\227\152\227\152\227\152\227\185\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\219\239\219\239\219\239\220\243\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\254\251\255\255\151\223i\166\164\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\005\150\203\174T\211w\223w\223w\223\151\223\151\223\152\227\152\227\152\227\152\227\153\231\185\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\187\239\187\239\219\239\219\239\219\239\220\243\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\221\247\221\247\254\251\255\255\253\2471\199\005\150\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\196\145G\158\015\191w\223\153\231\152\227\152\227\152\227\153\231\153\231\185\231\185\231\186\235\186\235\186\235\186\235\186\235\187\239\187\239\219\239\219\239\219\239\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\221\247\221\247\254\251\254\251\254\251\255\255\255\255v\219i\166\164\145\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\229\149\137\1661\199\152\227\186\235\186\235\185\231\186\235\186\235\186\235\186\235\186\235\187\239\187\239\187\239\219\239\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\221\247\253\247\254\251\254\251\254\251\254\251\255\255\255\255\255\255\152\227\172\178\228\145\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\132\145\229\149i\1661\199\151\223\187\239\219\239\187\239\187\239\187\239\187\239\219\239\220\243\220\243\220\243\220\243\220\243\221\247\221\247\221\247\221\247\253\247\254\251\254\251\254\251\254\251\255\255\255\255\255\255\253\247v\219\171\174\229\149\131\141\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\132\145\196\145'\158\204\178S\207\185\231\220\239\221\247\221\247\221\247\221\247\221\247\221\247\221\247\221\247\253\247\254\251\254\251\254\251\255\255\255\255\255\255\255\255\255\255\253\247\151\223\239\190H\162\196\145\131\145\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\132\145\132\145\132\145\132\145\132\145\132\145\132\145\132\145\196\145'\158\139\174\016\195U\215\152\227\219\239\221\247\253\247\254\251\255\255\255\255\255\255\255\255\254\251\253\247\187\235\151\2232\203\173\182(\162\196\145\131\141\132\145\132\145\132\145\132\145\132\145\132\145\132\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\145\164\145\164\145\164\145\164\145\164\145\132\145\132\145\132\145\132\145\164\145\196\145\229\149'\158i\166\139\174\172\178\173\182\173\182\172\178\140\174j\170'\162\230\153\196\145\163\141\131\141\132\145\132\145\132\145\164\145\164\145\164\145\164\145\164\145\132\145\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\164\017\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\132\145\132\145\132\145\132\145\132\145\131\145\131\141\131\141\131\141\131\141\131\141\131\145\132\145\132\145\132\145\132\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\164\017\164\017\164\017\164\017\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\017\164\017\164\017\164\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\164\017\164\017\164\017\164\017\164\017\164\017\164\017\164\017\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\145\164\017\164\017\164\017\164\017\164\017\164\017\164\017\164\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017\132\017"
)

pauseButton = image.new(
"Q\000\000\000P\000\000\000\000\000\000\000\162\000\000\000\016\000\001\000\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o{o{s{o[k{o{o{o{o\156o|o{o{o{o\156s\156\243\156\243{\239{\239Z\235Y\235Z\235Z\239Z\239{\239\155\243\189\247\156\243\156s{o{o{o|o{o{o\156s{o{o{o{o{o\156s{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o{o{s{o[k{o{o{o{o\156s\156s\156\243{\243Y\235\214\218r\2061\198\207\185\174\181\174\177\173\177\174\177\174\177\174\177\174\181\239\1891\198s\206\247\222Z\235\156\243\156\243\156s{s{o\156s{o{o{o{o{o\156s{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o{o{s{oZk{o\155s\156\243\156\243Z\235\247\222s\206\174\181\141\169\142\169\241\1733\182v\190\151\194\184\194\184\198\184\198\184\198\184\194\151\194u\186\019\182\208\173\141\169\141\173\239\185\148\210\248\226Z\239\156\247\189\247{o{o{o{o{o\156s{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o{o{s{o{o\156\243z\239\148\210\206\181\174\1733\186u\190\151\194\249\202\027\207\027\207\027\207\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\250\202\027\207\027\207\250\202\217\198\150\190U\186\018\182\141\173\239\189\214\218\156\243\156\243{o{o{o\156s{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o{o\156s\156\243Z\235s\206\140\173\174\169v\190\250\202\250\202\250\202\250\202\250\202\250\202\026\207\027\211<\215\092\219\092\223}\227}\227}\227\092\223\092\219;\215\027\211\027\207\250\202\250\202\250\202\250\202\027\203\217\1984\182m\165\173\181\181\218{\239\156s{s\156s{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\155s{o{o\156\243Z\235s\206\173\173\208\173\184\194\027\207\250\202\250\202\250\202\027\211<\215\092\223\157\231\190\239\190\239\190\243\190\243\222\243\222\243\222\243\222\243\222\243\222\243\190\243\190\239\158\235}\227\092\219<\215\027\207\250\202\250\202\250\206\027\207v\190\142\169\207\181\181\214{\239\189\247{o{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s{s\155s\156\243R\202K\165\018\182\218\202\027\207\250\202\250\202\027\207\092\219\157\231\190\239\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\239}\231<\219\251\206\250\202\250\206\027\207\184\198\175\173k\173\214\218\189\247{o{o{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156s\189\2479\231\206\181\175\169\217\198\027\207\250\202\250\202\027\211}\227\190\239\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\158\235\092\223\027\207\250\202\250\202\027\207\151\190m\1691\198z\239\156\243{s|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o{o{o\156\2431\198\174\173U\186\027\207\250\202\250\202;\215}\227\190\239\190\243\190\243\190\239\190\239\190\243\190\243\190\243\190\243\190\243\190\243\190\243\222\243\222\243\222\243\222\243\222\243\222\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\239\190\239\190\243\222\243\158\235}\227\027\211\250\202\250\206\250\202\018\182\141\173\213\218\157\247|s{o|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{k\156s{\239\239\189\207\173\250\202\027\207\250\202\027\207}\227\190\239\190\243\190\239\190\239\190\239\190\239\190\239\190\243\190\243\190\243\190\243\190\243\190\243\191\243\191\243\191\243\191\243\191\243\191\243\191\243\191\243\190\243\190\243\190\243\190\243\190\243\190\243\190\239\190\239\190\239\190\239\190\239\190\239\190\243\190\239\092\223\250\206\250\202\027\207\184\194l\169R\206\156\243\156s|o\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o9\231\173\177\208\173\251\206\250\202\250\202<\215\157\235\190\243\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\243\190\243\191\243\191\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\190\243\191\243\191\243\190\243\190\243\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\243\157\231\027\211\250\202\250\202\217\202m\165\016\194\156\243\156s\156s\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\189\247\024\227\141\173v\190\027\207\250\202\251\206|\227\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\191\239\191\239\191\239\191\239\191\243\223\243\223\243\223\243\223\243\223\243\223\243\223\243\223\243\191\243\191\239\191\239\191\239\191\239\191\239\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\092\223\250\206\250\202\250\202\019\182\206\185{\239\156w\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\189\247\024\231K\169v\190\027\207\250\202\027\207}\231\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\223\239\223\243\223\243\223\243\223\239\157\235[\227:\223\025\219\025\219:\223[\227}\231\190\239\223\243\223\243\223\243\223\243\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239}\227\251\206\250\202\027\207\241\177\173\181{\243\189{\157{\157{\157{\157{\157{\157{\157{\157{\157{\157{\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\189w9\231l\169u\190\027\207\250\202\027\207\157\231\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\223\243\223\243\157\235:\223\214\2102\194\207\185\140\173J\169J\169J\169J\169J\169J\169k\173\173\181\017\190\149\206\025\219|\231\223\243\223\243\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239}\227\027\207\250\202\027\207\241\177\207\185\156\243\156w\156s\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\157sZ\235\141\173\151\194\027\207\250\202\027\207\157\231\190\239\190\235\190\239\190\239\190\239\190\239\190\239\191\239\191\239\191\239\223\239\223\243[\227S\198k\173J\169J\169k\173\173\181\206\185\239\189\016\194\016\194\016\194\016\194\239\189\207\189\173\181\140\177j\173J\169J\169\240\185\025\219\191\239\223\243\190\239\190\239\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\235\190\239}\231\027\207\250\202\027\207\019\182\239\189\156\243\156s\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156\243\174\181\018\178\027\207\250\202\027\207}\227\190\235\158\235\190\235\190\235\190\235\190\239\190\239\190\239\190\239\223\239\223\243\092\231\017\190J\169J\169\173\181\239\1891\198s\206\148\210\181\214\214\218\247\222\247\222\247\222\247\222\214\218\214\218\148\210s\206R\202\239\189\173\181k\173)\165\174\181\248\218\223\239\223\243\191\239\190\239\190\239\190\239\190\239\190\239\190\235\190\235\158\235\190\239}\227\250\206\250\202\250\202\142\169R\202\156\247\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156w\156w\156w\156w\156w\156w\156w\188w\188\247\206\185\018\182\027\207\250\202\250\202\092\223\158\235\158\235\158\235\190\235\190\235\190\235\190\239\190\235\190\239\223\243[\227\016\190k\173k\173\206\1892\198\148\210\247\2229\231Z\235{\239\156\243\156\243\189\247\189\247\189\247\189\247\156\243\156\243{\239Z\2359\231\247\222\181\214R\202\239\189\140\177J\169\206\181\248\214\223\239\191\239\190\239\190\239\190\239\190\235\190\235\190\235\158\235\158\235\158\235\092\219\250\202\250\206\027\207\142\169s\206\190\247\157w\157s\157s\157s\157s\157s\157s\255\127\156s\156s\156s\156s\156s\156s\156s\189\247\181\218\175\169\027\207\027\207\251\202;\211\158\235\158\235\158\235\190\235\190\235\190\235\190\235\190\235\191\239\223\239\247\214k\173J\169\206\185R\202\181\214\024\227{\239\156\243\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\156\243{\2399\231\214\218s\206\016\194k\177)\165S\198\191\239\191\239\190\235\190\235\190\235\190\235\158\235\158\235\158\235\158\235\158\235;\211\027\207\250\206\217\198\173\173Z\235\157w\157s\157s\157s\157s\157s\157s\255\127|s|s|s|s|s|s\156sZ\235l\173\184\194\250\202\251\202\027\207}\227\158\235\158\235\158\235\158\235\158\235\158\235\190\235\191\239\158\235R\194)\165\173\1811\198\181\214\024\227{\239\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247{\2399\231\214\218R\202\206\185J\169\206\181\092\227\223\239\190\235\190\235\158\235\158\235\158\235\158\235\158\235\158\235}\231\027\211\250\202\027\2073\182\206\185\156\243\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156\243\173\1813\182\027\207\250\202\027\207\092\219\158\235\158\235\158\231\158\235\158\235\158\235\158\235\191\239}\231\207\185j\173\239\189\148\210\024\227{\239\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\156\2439\231\214\2181\198\140\177k\173\025\219\223\239\190\235\158\235\190\235\158\235\158\231\158\231\158\235\158\235\092\223\251\206\250\206\027\207\174\169R\202\189\247\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\189\247\148\214m\165\027\207\250\202\027\207\027\211\157\231\158\231\158\235\158\231\158\231\158\231\158\231\191\235|\227\173\177k\1731\198\181\214Z\235\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247{\239\214\218R\202\173\181k\169\249\214\223\239\158\235\158\235\158\235\158\231\158\231\158\235\157\231}\227\027\207\251\206\250\206\217\198K\1659\235\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156\243\173\177\151\190\250\206\250\206\027\207\092\215\158\231\158\231\158\231\158\231\158\231\158\231\191\235}\227\174\181k\1731\198\214\218Z\235\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251{\239\024\227s\206\173\181k\173\249\214\191\235\158\231\158\231\158\231\158\235\158\231\158\231\157\231<\215\027\207\250\206\027\207\019\182R\198\189\247\156s\156s\156s\156s\255\127\156s\156s\156s\156s\189\247\181\214\241\177\027\207\250\202\027\207\027\207}\223\158\231\158\231\158\231\158\231\158\231\158\231\158\231\239\185\140\177R\202\247\222\156\243\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\2479\231s\206\206\185\140\173[\223\191\235\158\231\158\235\158\231\158\231\158\231\157\231}\227\027\207\027\207\250\202\250\202\141\1699\231\189w\156s\156s\156s\255\127\157s\157s\157s\189s\156\243\140\177v\190\250\206\251\206\027\207<\211}\227\157\227\158\231\158\231\158\231\158\231\191\235S\194k\1731\198\247\222{\239\222\251\255\255\255\255\255\255\255\255\255\255\181\214c\140\132\144\132\144\132\144\132\144\132\144c\140\173\181\255\255\255\255\255\255\255\255{\239\165\148\132\144\132\144\132\144\132\144\132\144c\140\231\156\222\251\255\255\255\255\255\255\255\255\255\255\189\247\024\227s\206\173\181\174\181\157\227\190\235\158\231\158\231\158\231\158\231\157\227}\231<\215\027\207\250\206\027\207\241\173\016\194\189\247\156s\156s\156s\255\127\156o\156o\156o\189\243\148\210\142\169\027\207\250\202\027\207\027\207\092\219\157\227\158\227\158\227\158\231\158\231\191\231\182\206k\173\016\194\247\222{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\156\243\024\227s\206\139\1771\194\158\231\158\231\158\231\158\227\158\227\158\227\157\227]\223\027\207\251\206\250\206\217\198l\1658\231\156s\156s\156s\255\127\156s\156s\156s\156\243l\173u\190\251\206\251\206\027\207\028\207}\223}\227\158\227\158\227\158\227\158\231}\227\173\177\016\194\181\214{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\222\251\156\243\247\2221\198\139\177\215\210\191\231\158\227\158\227\158\227~\227}\227}\223\027\211\027\207\250\206\027\207\209\1730\198\156\247{o{o\255\127{o{o\156s\247\222l\165\250\202\250\202\027\207\027\207<\211}\223~\227~\227\158\227\158\227\191\231t\198\140\177\147\2109\231\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251{\239\214\218\206\185\239\185\158\227\158\227\158\227\158\227~\227}\227}\227<\215\027\207\027\207\250\206\184\194k\169{\239\156s\156s\255\127{o{o\189\247\016\1943\182\027\207\250\206\027\207\027\207<\215}\227~\227~\227\158\227\159\227\026\215k\1731\198\024\227\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\231\148\210\140\177\149\198\191\231\158\227\158\227~\227}\227}\227\092\215\027\207\027\207\250\206\027\207\175\173\247\222\189w\156s\255\127\156s\156s\156\243\240\185\184\198\250\202\027\207\027\207\028\207\092\215}\223~\223~\227~\227\159\227\017\190\173\181\181\214{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\247\222\015\194\173\177}\223\158\227~\227~\223}\223}\223]\215\027\207\027\207\250\206\026\207U\186s\206\156\243{o\255\127\156s\157s9\231\240\181\250\202\250\202\027\207\027\207<\207]\215}\223~\223~\223\158\227\026\215\140\1771\1989\231\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235\148\210\140\177\150\202\159\227~\223~\223~\223}\223]\219\028\207\027\207\027\207\250\206\151\194\206\181{\239{o\255\127\155s\156\243\148\210\175\173\027\207\250\206\027\207\027\207<\211]\215}\223~\223~\223\159\227S\194\173\181\181\214{\239\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\247\222\239\193\206\181}\223~\223~\223}\223}\223]\219<\207\027\207\027\207\250\206\250\202l\1698\231\156s\255\127\156s\156\243\239\189\018\182\027\207\251\206\027\207<\211<\211]\215}\223}\223~\223~\223\017\190\016\194\247\222\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\231s\206\173\181\027\215~\223}\223}\223}\223]\219<\211\028\207\027\207\250\206\027\207\174\169\148\214\157\247\255\127\156s{\239\140\173\151\194\250\206\251\206\027\207<\211<\211]\215}\219}\219~\223;\215\206\185\148\210Z\235\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\214\218\206\185\182\202\159\223}\223}\219}\219]\215<\211<\211\027\207\251\206\027\207\018\182\239\193\189\247\255\127\156\2439\231l\169\217\198\250\206\027\207\027\207<\211<\211]\215}\219}\219\158\223\183\202\206\185\214\218\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\227\016\1982\190\158\223}\219}\219}\219]\215<\211<\211\027\207\251\206\026\207v\190\173\177{\239\255\127\156\243\214\218m\165\250\202\250\206\027\207\028\207<\211<\211]\211]\219}\219\158\223S\194\016\194\024\227\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235s\206\239\185]\219~\219}\219]\219]\215<\211<\211\027\207\027\207\250\206\184\198l\169Z\235\255\127\156\243s\206\174\169\027\207\250\206\027\207\028\207<\211<\211]\211]\219}\219]\219\239\185R\202Z\235\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255{\239\181\214\206\185\026\211~\219]\219]\219]\215<\211<\211\027\207\027\207\250\206\250\202l\165\024\227\255\127{\2391\198\208\173\027\207\250\206\027\207\028\207<\211<\211]\211]\215~\219<\215\206\181\148\210{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\247\222\206\185\216\206~\219]\219]\219=\211<\211<\211\027\207\027\207\250\206\250\202m\169\214\218\255\127{\239\239\189\018\182\027\207\027\207\027\207\028\207<\211=\211]\211]\215~\219\026\211\206\185\181\214\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\227\239\189\150\202~\219]\219]\215=\211<\211<\211\027\207\027\207\250\206\027\207\174\169\181\214\255\127Z\235\206\1854\182\027\207\251\206\027\207<\211<\211<\211]\211]\215~\219\249\206\206\185\214\218\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\2479\231\015\194u\198~\219]\219]\215]\211<\211<\211\028\207\027\207\251\206\027\207\175\173s\210\255\1279\235\206\185T\186\251\206\251\206\027\207<\211<\211<\211]\211]\215~\219\217\206\206\185\247\222\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\181\214\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\231\016\194t\194~\219]\215]\215]\215<\211<\211\028\207\027\207\251\206\027\207\208\173s\206\255\1279\235\206\185T\186\251\206\251\206\027\207<\211<\211=\211]\215]\215~\215\217\206\239\189\247\222\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128B\136\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\2310\198t\194~\219]\215]\215]\215<\211<\211<\207\027\207\250\206\027\207\208\173R\206\255\127Z\235\206\1854\182\027\207\251\206\027\207<\211<\211=\211]\215]\215~\215\249\206\239\189\247\222\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128B\136\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\2310\198t\198~\219]\215]\215]\215<\211<\211<\211\027\207\250\206\027\207\207\173s\206\255\127Z\235\239\189\019\182\027\207\026\207\027\207<\211<\211=\211]\215]\215~\219\026\207\239\189\247\222\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128B\136\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\2519\231\016\198\150\198~\219]\215]\215]\215<\211<\211\028\211\027\207\250\206\027\207\174\169\148\210\255\127{\239\016\194\241\177\027\207\026\207\027\207<\211<\211=\215]\215]\215^\219\027\211\239\189\214\218\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\2479\231\016\194\216\206~\219]\215]\215]\215<\215<\211\028\211\027\207\250\206\251\206\141\169\181\218\255\127{\2391\202\175\169\027\207\026\207\027\207<\211<\211=\215]\215]\215]\219]\215\016\190\181\214\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\227\016\194\250\210~\219]\219]\215]\215<\215<\211\028\211\027\207\250\206\250\202l\165\247\222\255\127\188\247\148\214\141\165\027\207\250\206\027\207<\211<\211<\215]\215]\215]\219~\219R\194\148\210{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\247\222\016\194<\215~\219]\219]\215]\215<\215<\211\027\211\027\207\250\206\217\198k\1699\231\255\127\156\243\247\226K\165\250\202\250\206\027\207<\211<\211<\215]\215]\215]\219~\223\182\202R\202Z\235\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255{\239\181\214S\198~\219~\219]\219]\219]\215<\215<\211\027\211\027\207\027\207\151\194\140\173{\239\255\127\156\243Z\235k\169\184\194\027\207\027\207;\211<\211<\215]\215]\219]\219~\219\026\2111\1989\231\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235s\206\182\202~\223~\219]\219]\219]\215<\215<\211\027\211\027\207\027\2074\182\173\181\156\243\255\127\156s\156\243\173\181T\186\027\207\027\207\027\211<\211<\215]\215]\219]\219~\219]\219S\198\214\218\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\181\214\000\128\000\128\000\128\000\128\000\128\000\128\000\128J\169\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128B\136\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\2271\202\026\211~\223}\219]\219]\219]\215<\215<\211\027\211\027\207\027\207\207\173R\202\156\243\255\127\155s\156\2431\198\208\173\027\207\027\207\027\211<\211<\215]\215]\219]\219}\219~\223\149\202\148\210z\239\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\181\214\000\128\000\128\000\128\000\128\000\128\000\128\000\128)\165\255\255\255\255\255\255\255\255{\239\000\128\000\128\000\128\000\128\000\128\000\128\000\128\132\144Z\235\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\214\2181\194\092\219~\223}\219]\219]\219]\215<\215<\211\027\211\250\206\250\206l\165\214\222\156\243\255\127\155s\156\243\214\218\207\173\027\207\250\206\027\211<\211<\215]\215]\219]\219}\219\158\223\216\2101\198\024\227\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\181\214\000\128\000\128\000\128\000\128\000\128\000\128\000\128k\173\255\255\255\255\255\255\255\255{\239!\132\000\128\000\128\000\128\000\128\000\128\000\128B\136Z\235\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235s\206t\202\158\223}\223}\219]\219]\219]\215<\215<\211\027\211\251\206\184\198l\173Z\235\156s\255\127{o\156sZ\235\207\181\217\198\250\206\027\207<\211<\215\092\215]\219}\219}\223~\223~\223R\198\181\214\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128k\173\255\255\255\255\255\255\255\255{\239!\132\000\128\000\128\000\128\000\128\000\128\000\128B\136Z\235\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\2271\198;\219\158\227}\223}\223]\219]\219\092\215<\215<\211\027\207\027\207v\190\016\194\156\243\156s\255\127\156s\156s\156\243\206\185u\190\027\207\027\207\027\211<\215\092\215]\219]\219}\223~\223\159\227\248\210R\202{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\214\218\000\128\000\128\000\128\000\128\000\128\000\128\000\128\008\161\255\255\255\255\255\255\255\255{\239!\132\000\128\000\128\000\128\000\128\000\128\000\128c\140{\239\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251{\239\148\210t\202\159\227~\223~\223}\223]\219]\219\092\215<\211\027\211\027\207\027\207\241\177\148\210\156\243\155o\255\127\156s\156s\189\247s\210\174\169\027\207\027\207\027\211<\211\092\215]\219]\219}\223~\223~\227~\227t\202\024\231\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128\231\156\255\255\255\255\255\255\255\255{\239!\132\000\128\000\128\000\128\000\128\000\128\000\128B\136{\239\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\227R\202;\219\158\227~\223~\223}\223]\219\092\219<\215<\211\027\211\026\207\249\202K\1659\231\156s\156s\255\127\156s\156s\156sY\235*\165\184\198\027\207\027\211;\211<\215\092\219]\219}\223~\223~\227\158\227;\219\148\210Z\235\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128\008\161\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128B\136\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235s\210\215\210\159\231~\227~\223~\223}\223]\219\092\219<\215;\211\027\207\027\2073\186\173\181\156\243\156s\156s\255\127\156s\156s\156s\156\243\239\189\208\173\027\211\027\207;\211<\215\092\215]\219}\223~\223~\227\158\227\159\227\149\206\148\210z\235\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128)\165\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243\214\218R\202|\223\158\227~\227~\227~\223}\223]\219\092\215<\215\027\211\027\207\027\207l\165\148\214\156\243\156s\156s\255\127\156s\156s\156s\156\243\024\227K\165\184\198\027\207\027\211<\215\092\215]\219}\223}\223~\227\158\227\158\227}\227s\202\181\218\156\243\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\210\000\128\000\128\000\128\000\128\000\128\000\128\000\128)\165\255\255\255\255\255\255\255\255Z\235\000\128\000\128\000\128\000\128\000\128\000\128\000\128c\140\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\024\227R\202;\219\159\231\158\227\158\227~\227}\223}\223]\219<\215;\215\027\211\027\211T\186\140\177{\239\156s\156s\156s\255\127\156s\156s\156s\156s\156\243\016\1943\186\027\211\027\211<\211<\215\092\219}\223}\223~\227\158\227\158\227\190\231\092\223S\202\024\227\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\214\218\132\144\132\144\165\148\165\148\132\144\132\144c\140\173\181\255\255\255\255\255\255\255\255{\239\198\152\132\144\165\148\198\152\165\148\165\148\132\144\008\161\222\251\255\255\255\255\255\255\255\255\255\255\255\255\189\247Z\235s\206\248\214\191\231\158\227\158\227\158\227~\227}\223}\223\092\219<\215;\211\026\207\027\211\141\169\148\214\188\247\156s\156s\156s\255\127\156s\156s\156s\156s\156\243\024\227\141\173\250\202\027\207\027\211<\215\092\219]\219}\223~\227\158\227\158\227\158\231\191\231\026\219s\206\024\227\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222\251\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251Z\235\148\210\214\210\191\231\158\231\158\231\158\227~\227}\227}\223]\219\092\219<\215\027\211\027\211\150\194\206\181{\239\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156\243\239\189\208\173\027\211\027\211;\215<\215]\219}\223}\223~\227\158\227\158\231\158\231\191\235\025\219R\206\024\227\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247Z\235\147\210\182\210\158\231\158\231\158\231\158\231\158\227}\227}\223]\223\092\219<\215;\215\027\211\027\207l\165\181\218\189\247\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\189w9\231J\165\183\198\027\211\027\211<\215\092\219]\223}\223~\227\158\227\158\231\158\231\158\231\191\235\025\219s\206\024\227\156\243\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247Z\235\148\214\215\214\190\235\158\231\158\231\158\231\158\231\158\227~\227}\223]\219\092\219<\215\027\211;\2113\182\140\177{\239\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156\243\148\210\141\169\250\206\027\211;\215\092\215\092\219}\223}\227}\227\158\231\158\231\158\231\158\231\191\235;\223\148\210\214\218{\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\156\243\024\227\148\210\248\218\190\235\190\235\158\231\158\231\158\231\158\227}\227}\227}\223\092\219<\215;\215\027\211\183\198k\169\024\227\156\243\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\157s\156\243\239\1893\186;\211\027\211<\215\092\219]\223}\223}\227\158\227\158\231\158\231\158\231\158\235\191\235}\231\182\210\181\214Z\235\156\243\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247{\239\214\218\148\210[\227\191\235\190\235\158\231\158\231\158\231\158\231\157\227}\227}\223]\219<\219;\215\027\211\027\211\175\173\148\210\189\247\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156\2439\231k\169\150\194;\211;\215<\215\092\219]\223}\223}\227\158\227\158\231\158\231\158\231\158\235\191\235\191\235\025\223\181\214\246\222Z\239\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247{\239\024\227\181\214\215\214\157\231\191\235\190\235\158\235\158\231\158\231\158\231\158\227}\227}\223]\223\092\219<\215;\211;\211\018\182\206\185\156\243\189s\189s\189s\189s\189s\189s\189s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\157\243\214\218K\165\184\198;\211;\215<\219\092\219]\223}\227}\227\158\231\158\231\158\231\158\235\190\235\190\239\191\239\157\235\025\219\149\210\214\2189\231\156\243\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247\156\243Z\235\247\222\180\214\215\214|\231\191\239\191\239\158\235\158\235\158\235\158\231\158\231\157\227}\227}\227]\223\092\219<\215;\215;\2153\186\140\1739\231\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156\243s\210\141\169\027\211;\211;\215\092\219\092\223}\223}\227}\227\157\231\158\231\158\235\190\235\190\235\190\235\191\235\191\239}\231\248\218\181\214\213\218\024\227Z\235{\243\156\247\189\247\222\251\222\251\222\251\222\251\222\251\222\251\189\247\189\247\156\243Z\239\024\227\214\218\181\214\215\218[\227\191\239\191\239\190\235\190\235\190\235\158\235\158\231\158\231\158\231}\227}\227]\223\092\219<\219;\215<\215\183\198k\169\247\226\156\243\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156\243R\202\207\173\250\206;\215;\215\092\219\092\223}\223}\227\157\227\158\231\158\231\158\231\158\235\190\235\190\235\190\239\191\239\223\239\158\235:\223\247\218\247\218\247\218\247\222\247\222\024\2278\2319\2319\2319\231\024\227\024\227\247\222\215\218\247\218\247\218\025\223|\231\191\239\191\239\190\235\190\235\190\235\158\235\158\235\158\235\158\231\157\231}\227}\227}\223\092\223\092\219;\215<\215\184\198\174\177\214\218\156\243\156s\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\189w\156\2430\198l\169\250\206;\215;\215\092\219\092\223}\223}\227}\227\157\231\158\231\158\231\158\235\158\235\190\235\190\235\190\239\191\239\223\239\223\239\191\239\157\235|\231:\227\025\223\248\218\248\218\248\218\248\218\024\223:\223[\227\157\235\190\239\191\239\223\239\191\239\190\239\190\235\190\235\190\235\158\235\158\235\158\231\158\231\158\231\157\231}\227]\223\092\223<\219;\215<\215\183\198k\169\180\214\156\243\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{\239\016\198\175\177\027\211;\215;\215\092\219\092\223}\223}\227}\227\157\231\158\231\158\231\158\231\158\235\190\235\190\235\190\235\190\239\190\239\191\239\191\239\191\239\223\239\223\239\223\239\223\239\191\239\223\239\223\239\223\239\223\239\191\239\191\239\191\239\190\239\190\239\190\235\190\235\190\235\158\235\158\235\158\231\158\231\157\231}\227}\227}\227\092\219<\219;\215<\215\216\202l\169\148\214\156\243{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156\243\156\243R\202\173\173\183\198\092\219;\215<\219\092\219\092\223}\223}\227}\227\157\231\158\231\158\231\158\235\158\235\190\235\190\235\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\239\191\239\191\239\190\239\190\239\190\239\190\239\190\239\190\239\190\235\190\235\158\235\158\235\158\235\158\231\158\231\157\231}\227}\227}\223\092\223\092\223\092\219;\215;\215T\190\173\177\214\218\156\243\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{\239\148\210\140\177\150\198<\215;\215<\215\092\219\092\223}\223}\227}\227}\231\157\231\158\231\158\231\158\235\158\235\158\235\190\235\190\235\190\235\190\235\190\239\190\239\190\239\190\239\190\239\190\239\190\239\190\235\190\235\190\235\190\235\158\235\158\235\158\235\158\231\158\231\157\231}\227}\227}\227]\223\092\223\092\219;\215;\215<\219\018\186\140\177\247\226\156\243\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156\243\247\222\140\177\018\186\250\206<\215;\215\092\219\092\223\092\223}\223}\227}\227}\231\157\231\158\231\158\231\158\231\158\235\158\235\158\235\190\235\190\235\190\235\190\235\190\235\190\235\190\235\190\235\190\235\158\235\158\235\158\235\158\231\158\231\158\231\157\231}\227}\227}\227}\223\092\223\092\219<\219;\215<\215\216\202\208\177\239\1899\231\156\243{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156\243Z\2351\198\140\173\150\198<\215;\215;\215\092\219\092\219\092\223}\223}\227}\227}\227}\227\157\231\157\231\158\231\158\231\158\231\158\231\158\235\158\235\158\235\158\235\158\231\158\231\158\231\158\231\158\231\157\231\157\231}\227}\227}\227}\223]\223\092\223\092\219<\219;\215<\215;\2112\186l\173\147\210{\239\156s{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s\156\243\156\243\214\218\140\177\207\177\216\202<\215<\215;\219\092\219\092\219\092\223]\223}\223}\227}\227}\227}\227}\231\157\231\157\231\157\231\157\231\157\231\157\231\157\231\157\231}\231}\231}\227}\227}\227}\227}\223]\223\092\223\092\219\092\219;\219<\215<\215\150\194\141\173\206\185\024\231\156\243\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o\156s\156\2439\231R\202\174\1812\186\183\198<\215<\219;\215<\219\092\219\092\219\092\223\092\223}\223}\223}\227}\227}\227}\227}\227}\227}\227}\227}\227}\227}\227}\223]\223\092\223\092\223\092\219\092\219<\219;\215<\219\027\211u\194\017\186\173\181\148\210Z\235\156\243{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o{o{o{o\156\2439\231R\202\140\177\207\177\150\194\027\211<\215<\215;\215<\219\092\219\092\219\092\219\092\223\092\223\092\223\092\223\092\223\092\223\092\223\092\223\092\223\092\223\092\219\092\219\092\219<\219;\219;\215<\215<\215\250\206T\190\141\173\173\181\148\214{\239\156\243{o{o{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o|s{o{o{o{\239\156\243Z\235s\206\140\177\141\1732\186\184\202\026\211<\215\092\219\092\219<\219<\219<\219\092\219\092\219\092\219\092\219\092\219\092\219<\219<\219<\219\092\219\092\219\092\219;\215\250\210\151\198\241\181l\173\206\185\181\214{\239\156\243\156s\156s{o{o{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o|s{o{o{o{o{s\156\243\156\239Z\235\247\2221\198\240\189\240\185\240\181\017\182u\194\183\202\249\206\027\211;\215<\215<\215<\215<\215<\215;\215\026\211\249\206\183\198T\190\240\181\240\181\240\189\240\189s\206\024\227{\239\156\243{o\156s{o\156s{o{o{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o|s{o{o{o{o{s{o{o\156s\156s\156\243{\239\024\231\181\2141\202\206\185\140\177\140\173\173\177\207\177\207\181\240\181\240\181\240\181\207\177\174\177\141\173\140\173\173\177\239\189R\206\214\2189\231{\243\156\243{o\156s{o{o\156s{o\156s{o{o{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s\156s{o\156s\156s{o|s{o{o{o{o{s{o{o{o{o{o{o{\239\156\243{\239{\239Z\235\024\227\246\222\181\214\148\210\147\210\147\210\148\210\148\214\213\218\247\2229\231Z\235{\239\156\243\156\243{o{o{o{o{o{o{o\156s{o\156s{o{o{o\156s{o{o{o\156s\156s{o{o{o{o{o{o{o{o{o{o{o\255\127"
)

stopButton = image.new(
"P\000\000\000P\000\000\000\000\000\000\000\160\000\000\000\016\000\001\000\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\189w\222{\189w\189w\189w\222{\189\247\222\251\189\247\189\247\156\243\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\222\251\189\247\189\247\222\251\222{\222{\222{\222{\222{\222{\255\127\255\127\222{\255\127\255\127\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\189w\222{\189w\189\247\189\247\189\247\189\247\189\247\189\247\222\251\222\251\222\251\222\251\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222\251\222\251\189\247\222\251\222\251\222\251\222\251\222{\222{\255\127\255\127\222{\255\127\255\127\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\189\247\222\251\189\247\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222\251\222\251\255\127\255\127\222{\255\127\255\127\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222{\222{\222{\222{\222\251\189\247\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\255\255\222{\255\127\255\127\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222{\222{\222\251\189\247\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\255\255\255\127\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222{\222\251\189\247\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222{\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222{\222{\222\251\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222{\222\251\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222\251\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\255\127\222\251\189\247\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247\189\243\156\239\156\239\156\239\156\239\156\235\156\239\156\239\156\239\156\239\156\243\189\243\189\247\222\251\222\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\222\251\189\247\189\247\255\255\255\255\255\255\255\255\254\255\223\255\223\255\255\251\255\255\255\255\255\255\255\255\223\255\190\247\157\243\156\239\155\239{\235Z\2399\239\247\242\214\242\181\242\181\242\181\242\181\242\213\242\247\2429\243Z\239{\235\155\239\156\239\157\243\190\247\223\255\255\255\255\255\255\251\254\251\254\255\254\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\251\255\251\254\255\254\255\254\255\255\251\223\251\189\247\156\239\155\235Z\239\247\242\148\242\017\250\206\249\140\249k\253k\253J\253J\253J\253j\253k\253l\249\140\249\206\249\015\250s\246\247\242Z\239{\239\156\239\189\243\222\251\255\251\255\255\255\255\255\251\255\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\223\255\255\251\254\251\255\255\223\251\157\247{\239{\235\024\239s\246\206\249\139\253j\253J\253K\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253J\253k\253\206\249t\246\024\243{\239|\239\157\247\223\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\254\251\255\255\223\255\189\247|\239{\239\247\242\016\250k\249J\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253j\253K\253\140\249\016\250\248\242z\235\155\239\189\247\254\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\254\255\222\251\156\239\155\235\024\2431\246l\249K\253j\253k\253k\253l\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253J\253l\253\016\2508\239\155\235\156\239\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\254\255\255\255\223\255\255\255\189\247|\239Z\243s\246\140\253J\253j\253k\253l\253k\253k\253k\253k\253k\253l\253l\253\139\253\139\253\139\253\139\253\140\253\140\253l\253l\253\140\253\140\253\140\253k\253k\253k\253k\253l\253l\253j\253J\253\140\249s\242Z\239\156\235\189\247\254\255\255\255\223\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\255\189\247\222\251\255\255\255\255\255\255\255\255\255\255\223\255\255\255\255\255\255\251\157\243\156\2398\239\239\249K\253k\253k\253k\253k\253k\253\139\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\139\253\140\253\140\253k\253k\253k\253k\253k\253k\253j\253J\253\240\245\024\243\155\239\156\243\255\255\255\255\255\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\222{\222{\222{\255\127\222\251\189\247\255\255\255\255\255\255\255\255\255\255\254\255\223\255\223\255\254\251\156\243\156\239\214\242\141\249J\253j\253l\253l\253k\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253l\253l\253l\253k\253j\253J\253\141\249\214\242\156\239\156\243\254\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\189\247\222\251\255\255\255\255\255\255\255\255\255\255\223\255\255\251\254\251\156\243\155\239\149\246k\253k\253k\253k\253\139\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\139\253l\253l\253\139\253k\253K\253k\253\180\242{\239\157\243\223\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\255\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\156\243{\239\148\242k\249j\253k\253l\253\140\253\139\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\141\253\141\253\141\253\141\253\141\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\139\253\140\253l\253l\253l\253k\253j\253k\253\148\242{\239\156\239\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\222{\222{\222\251\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\251\157\243|\239\148\246k\253j\253k\253k\253\139\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\172\253\172\253\172\253\172\253\173\253\173\253\173\253\173\253\173\253\173\253\172\253\172\253\172\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\139\253\140\253k\253k\253k\253k\249\148\246\156\239\156\243\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\222{\222{\222{\222{\222{\255\127\255\127\255\127\255\127\255\255\189\247\222\251\255\255\255\255\254\251\255\255\223\255\254\255\255\255\189\247\156\239\214\246k\253j\253k\253\139\253\140\253\140\253\140\253\140\253\140\253\141\253\141\253\141\253\141\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\141\253\141\253\141\253\140\253\140\253\140\253\140\253\140\253\140\253\139\253k\253\139\249j\253L\253\214\242\156\239\189\247\255\255\255\255\223\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\255\222\251\222\251\255\255\255\255\255\255\255\255\255\251\255\251\255\255\222\251|\2398\239\141\249k\253k\253k\253\140\253\140\253\140\253\140\253\141\253\141\253\141\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\141\253\140\253\140\253\140\253\140\253\140\253\140\253l\253\140\253\139\253J\253\141\249\024\247\156\239\222\251\255\255\255\251\255\251\255\255\255\255\255\255\255\255\222\251\222\251\255\255\255\127\255\127\255\127\255\127\255\127\255\127\255\255\189\247\222\251\255\255\255\255\255\255\255\255\254\255\254\251\223\255\157\239z\243\239\249J\253k\253l\253\140\253\140\253\140\253\141\253\141\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\140\253\140\253\140\253\140\253\140\253l\253\139\253k\253J\253\239\245z\243\157\243\254\251\255\251\223\255\255\255\255\255\255\255\255\255\255\255\222\251\255\255\255\127\255\127\255\127\222{\222{\222{\189\247\222\251\255\255\255\255\255\255\255\251\255\251\254\255\255\255\189\247\155\239s\246K\253j\253\139\253\140\253\140\253\140\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\205\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\205\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\141\253\140\253\172\253\140\253l\253l\253k\253J\253s\242\155\239\189\243\223\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\222\251\255\127\255\127\255\127\255\127\255\127\255\255\189\247\222\251\222\251\222\251\222\251\223\251\223\255\254\251\223\251\156\239\024\243\140\253k\253l\253\140\253\140\253\141\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\206\253\206\253\205\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\172\253\140\253\140\253\140\253k\253k\253\140\249\024\243\157\239\222\251\254\251\222\251\222\251\222\251\222\251\222\251\222\251\222\251\255\255\255\127\255\127\255\127\255\127\222\251\189\247\255\255\222\251\222\251\222\251\223\255\222\255\254\255\189\243\155\239\017\246J\253k\253\140\253\141\253\141\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\141\253\172\253\172\253\140\253\139\253k\253K\253\016\246\156\239\188\243\254\251\222\251\222\251\222\251\222\251\222\251\222\251\189\247\189\247\222{\222{\222{\222\251\156\243\189\247\222\251\222\251\222\251\222\251\222\251\222\251\222\251\156\243\023\243k\253k\253\140\253\140\253\172\253\173\253\173\253\173\253\173\253\173\253\205\253\205\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\172\253\140\253\140\253\139\253k\253k\253\023\243\156\243\222\251\222\251\222\251\222\251\222\251\222\251\222\251\189\247\189\247\222\251\222{\189w\189\247\156\243\189\247\189\247\189\247\222\251\222\251\222\251\223\251\188\243|\2391\246J\253k\253\140\253\140\253\173\253\173\253\173\253\173\253\173\253\205\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\238\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\173\253\173\253\173\253\141\253\141\253\140\253\139\253J\2531\246\156\239\189\247\222\251\222\251\222\251\189\247\189\247\189\247\189\247\156\243\222\251\255\127\222{\189\247\156\243\189\247\189\247\189\247\189\247\189\247\189\247\222\247\155\2399\243\140\253\139\253\140\253\140\253\173\253\173\253\173\253\173\253\205\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\238\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\205\253\173\253\173\253\173\253\173\253\173\253\141\253\140\253\139\253k\253\248\242{\235\189\247\189\247\189\247\189\247\189\247\189\247\189\247\156\243\189\247\222{\222\251\156\243\156\243\189\247\189\247\189\247\189\247\189\247\189\247\155\243Z\235\206\241\165\252)\253\140\253\173\253\206\253\206\253\206\253\174\253\206\253\206\253\206\253\206\253\206\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\015\254\015\254\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\238\253\206\253\206\253\206\253\206\253\206\253\206\253\174\253\173\253\173\253\173\253\173\253\173\253\139\253)\253\165\252!\248\141\241Z\231\156\243\189\247\189\247\189\247\189\247\189\247\189\247\189\247\156\243\189w\222\251\156\243\189\247\189\247\189\247\189\247\189\247\189\247\189\247z\239\025\231\132\248\000\252!\252\132\252\232\252J\253\140\253\205\253\206\253\238\253\238\253\238\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\015\254\015\254\015\254\015\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\015\254\016\254\015\254\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253k\253)\253\199\252d\252\034\252\000\252\000\252\133\244\024\231{\235\189\247\189\247\189\247\189\247\189\247\189\247\189\247\156\243\222\251\189\247{\239\189\247\189\247\189\247\189\247\189\247\189\247\189\243{\235\147\238\000\248\000\252 \252B\252C\252c\252\132\252\198\248\008\249J\253\140\253\173\253\206\253\239\253\015\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\016\254\015\254\239\253\238\253\205\253\140\253k\253)\253\231\248\165\252d\252c\252B\252B\252B\252\000\252\000\252\000\244\147\238{\231\189\243\189\247\188\247\157\247\189\247\189\247\189\247{\239\189\247{\239{\239\156\243\156\243\156\243\156\243\156\243\189\247\156\243{\231\206\241\000\252\000\252!\252c\252c\252d\252\132\252\132\252\133\252\133\252\165\252\198\252\231\252\008\253J\253k\253\140\253\173\253\206\253\238\253\239\253\015\254\016\254\016\254\016\254\016\254\016\2540\2540\2540\2540\254\016\254\016\254\016\254\015\254\015\254\239\253\238\253\205\253\173\253\140\253J\253)\253\008\253\231\252\166\252\133\252\133\252d\252d\252d\252c\252c\252c\252B\252!\252\000\252\000\252\206\241{\231\156\239\157\247\188\243\157\247\156\243\156\243\189\247{\239\156\243{\239\156\243\156\243\156\243\156\243\156\243\156\243\156\243{\239Z\231\009\245\000\252 \252B\252d\252\132\252\132\252\132\252\132\252\165\252\165\252\165\252\197\252\198\252\198\252\198\252\198\252\231\252\231\252\008\253\008\253)\253)\253J\253J\253K\253k\253k\249k\249k\249k\249k\249K\253J\253J\253*\253)\253\008\253\008\253\231\252\231\252\199\252\198\252\198\252\198\252\197\252\165\252\165\252\165\252\164\252\164\252\132\252\132\252c\252c\252b\252A\252\000\252\000\252\008\241Z\235{\239\157\243\156\243\156\243\156\243\156\243\156\243{\239\156\243{\239{\239\156\243\156\243\156\243\156\243\156\243\156\243{\2358\235\132\248\000\252!\252c\252\132\252\132\252\132\252\164\252\165\252\165\252\165\252\198\252\198\252\198\252\231\252\231\252\231\252\007\253\007\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\008\253\007\253\231\252\231\252\231\252\231\252\230\252\198\252\198\252\165\252\165\252\165\252\164\252\132\252\132\252\132\252d\252c\252B\252 \252\000\252\132\248\025\235{\235\156\243\156\243\156\243\156\243\156\243\156\243\156\243Z\235{\239{\239\156\243\156\243\156\243\156\243\156\243\156\243{\235\214\238\034\248\000\252\034\252b\252\132\252\132\252\165\252\165\252\165\252\198\252\198\252\198\252\198\252\231\252\231\252\007\253\007\253\008\253\008\253\008\253\009\253(\253)\253)\253)\253)\253)\253)\253)\253)\253)\253)\253)\253)\253)\253)\253(\253\008\253\008\253\008\253\008\253\008\253\231\252\231\252\231\252\199\252\198\252\198\252\166\252\165\252\165\252\133\252\132\252\132\252\132\252c\252B\252!\252\000\252!\252\215\234z\235\156\239\156\243\156\243\156\243\156\243\156\243{\239{\239{\239{\239{\239{\239{\239{\239{\239{\239{\231\147\238\000\252\000\252B\252c\252\132\252\164\252\165\252\165\252\198\252\198\252\198\252\231\252\231\252\231\252\231\252\008\253\008\253\008\253(\253)\253)\253)\253)\253)\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253)\253)\253)\253)\253(\253\008\253\008\253\008\253\231\252\231\252\231\252\231\252\198\252\198\252\198\252\165\252\165\252\165\252\132\252\132\252\132\252c\252!\252\000\252\000\252\148\238{\235{\239{\243{\239{\239{\239{\239{\239Z\235Z\235{\239{\239{\239{\239{\239{\239{\239{\231R\238\000\252\000\252B\252\131\252\165\252\165\252\165\252\198\252\198\252\198\252\231\252\231\252\231\252\007\253\008\253\008\253\008\253)\253)\253)\253)\253I\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253J\253)\253)\253)\253)\253\008\253\008\253\008\253\008\253\231\252\231\252\231\252\198\252\198\252\198\252\165\252\165\252\132\252\132\252\132\252c\252B\252\000\252\000\252Q\238{\235{\239{\239{\239{\239{\239{\239{\239Z\235Z\235{\239{\239{\239{\239{\239{\239{\239{\2311\238\000\252 \252B\252\132\252\165\252\165\252\165\252\198\252\198\252\231\252\231\252\231\252\008\253\008\253\008\253(\253)\253)\253)\253J\253J\253J\253J\253J\253j\253k\253k\253k\253k\253k\253k\253k\253k\253k\253J\253J\253J\253J\253J\253)\253)\253)\253)\253\008\253\008\253\008\253\231\252\231\252\231\252\198\252\198\252\166\252\165\252\165\252\132\252\132\252c\252B\252\000\252\000\2520\238{\235{\239{\239{\239{\239{\239{\239{\2399\231Z\235{\239{\239{\239{\239{\239{\239{\239{\231\016\242\000\252 \252C\252\132\252\165\252\165\252\198\252\198\252\231\252\231\252\231\252\008\253\008\253\008\253)\253)\253)\253J\253J\253J\253J\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253k\253j\253J\253J\253J\253J\253)\253)\253)\253\008\253\008\253\008\253\231\252\231\252\231\252\198\252\198\252\165\252\165\252\165\252\132\252\132\252B\252\000\252\000\252\015\238{\235Z\239{\235{\239{\239{\239{\239{\2399\231Z\235Z\235Z\235Z\235Z\235Z\235Z\235Z\235{\231\016\242\000\252!\252c\252\164\252\166\252\198\252\198\252\230\252\231\252\231\252\008\253\008\253\008\253)\253)\253)\253J\253J\253J\253k\253k\253k\253k\253k\253k\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253k\253k\253k\253k\253k\253K\253J\253J\253J\253)\253)\253)\253\009\253\008\253\008\253\231\252\231\252\231\252\198\252\198\252\165\252\165\252\165\252\132\252B\252\001\252\000\252\015\238{\231Z\239z\235[\239Z\235Z\235Z\235Z\2359\231Z\235Z\235Z\235Z\235Z\235Z\235Z\235Z\235{\2311\242\000\252!\252c\252\164\252\198\252\198\252\198\252\231\252\231\252\008\253\008\253\008\253)\253)\253)\253J\253J\253J\253k\253k\253k\253k\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253\140\253k\253k\253k\253k\253J\253J\253J\253*\253)\253)\253\008\253\008\253\008\253\231\252\231\252\198\252\198\252\197\252\165\252\165\252\132\252B\252\001\252\000\252\016\242{\231Z\239Z\235Z\235Z\235Z\235Z\235Z\235\024\227Z\235Z\235Z\235Z\235Z\235Z\235Z\235Z\235{\231R\238\000\252!\252c\252\164\252\198\252\198\252\231\252\231\252\231\252\008\253\008\253)\253)\253*\253J\253J\253j\253k\253k\253k\253\140\253\140\253\140\253\140\253\140\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\172\253\140\253\140\253\140\253\140\253k\253k\253k\253J\253J\253J\253)\253)\253)\253\008\253\008\253\231\252\231\252\231\252\198\252\198\252\165\252\165\252\132\252B\252 \252\000\252R\238{\235Z\235Y\235Z\235Z\235Z\235Z\235Z\235\024\227Z\2359\231Z\235Z\235Z\235Z\235Z\235Z\235{\235\148\238\000\252!\252c\252\165\252\198\252\198\252\231\252\231\252\008\253\008\253)\253)\253)\253J\253J\253J\253k\253k\253k\253\140\253\140\253\140\253\172\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\173\253\140\253\140\253\140\253\140\253k\253k\253k\253J\253J\253J\253)\253)\253(\253\008\253\008\253\231\252\231\252\198\252\198\252\198\252\165\252\132\252B\252\000\252\000\252\148\238{\235Z\235Y\235Z\235Z\235Z\235Z\2359\2319\231Z\2359\2319\2319\2319\2319\2319\2319\231Z\235\247\238B\248\000\252B\252\165\252\199\252\231\252\231\252\008\253\008\253\008\253)\253)\253J\253J\253J\253k\253k\253\139\253\140\253\140\253\173\253\173\253\173\253\173\253\173\253\173\253\206\253\206\253\206\253\206\253\206\253\206\253\205\253\173\253\173\253\173\253\173\253\140\253\140\253\140\253l\253k\253k\253J\253J\253J\253)\253)\253\008\253\008\253\008\253\231\252\231\252\198\252\198\252\165\252\132\252B\252\000\252\034\252\246\238Z\231:\2319\235:\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\231Z\2319\239\166\244\000\252B\252\133\252\231\252\231\252\231\252\008\253\008\253)\253)\253J\253J\253J\253k\253k\253\139\253\140\253\140\253\173\253\173\253\173\253\173\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\141\253\140\253\140\253l\253k\253k\253J\253J\253)\253)\253)\253\008\253\008\253\231\252\231\252\230\252\198\252\165\252\132\252B\252\000\252\165\2489\235Y\2319\2319\2319\2319\2319\2319\231\024\2279\231{\2399\2319\2319\2319\2319\2319\2319\2319\231{\235)\245\000\252B\252\132\252\231\252\231\252\008\253\008\253(\253)\253)\253J\253J\253k\253k\253\139\253\140\253\140\253\173\253\173\253\173\253\206\253\206\253\206\253\206\253\206\253\238\253\239\253\239\253\239\253\238\253\238\253\206\253\206\253\206\253\206\253\205\253\173\253\173\253\173\253\140\253\140\253\140\253k\253k\253J\253J\253)\253)\253\008\253\008\253\008\253\231\252\231\252\198\252\198\252\132\252B\252\000\252)\245[\2399\2319\2279\2319\2319\2319\2319\231\024\2279\231\156\243\024\2279\2319\2319\2319\2319\2319\2319\231|\235\206\241\000\248B\252\131\252\199\252\231\252\008\253\008\253)\253)\253J\253J\253J\253k\253k\253\140\253\140\253\172\253\173\253\173\253\206\253\206\253\206\253\206\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\239\253\238\253\206\253\206\253\206\253\206\253\173\253\173\253\173\253\140\253\140\253k\253k\253j\253J\253J\253)\253)\253\008\253\008\253\231\252\231\252\230\252\165\252c\252!\252\000\252\238\241|\2359\2319\227\025\2318\2319\2319\2319\231\247\222\156\243\189\2479\231\024\227\024\227\024\227\024\227\024\227\024\2278\231{\235\180\242!\248!\252c\252\198\252\007\253\008\253\009\253)\253)\253J\253J\253k\253k\253\140\253\140\253\140\253\173\253\173\253\206\253\206\253\206\253\239\253\239\253\239\253\239\253\239\253\015\254\016\254\240\253\239\253\239\253\239\253\239\253\239\253\239\253\206\253\206\253\206\253\173\253\173\253\140\253\140\253\140\253k\253k\253J\253J\253)\253)\253\008\253\008\253\231\252\231\252\230\252\165\252c\252 \252!\248\181\238{\231\024\227\024\227\024\227\024\227\024\227\024\227\024\227\247\222\156\243\189\2479\231\024\227\024\227\024\227\024\227\024\227\024\227\024\227Z\231Z\235\165\248\000\252c\252\165\252\007\253\008\253)\253)\253I\253J\253j\253k\253k\253\140\253\140\253\173\253\173\253\206\253\206\253\206\253\239\253\239\253\239\253\015\254\015\254\015\254\016\254\016\254\016\254\016\254\015\254\016\254\239\253\239\253\239\253\238\253\206\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253)\253)\253)\253\008\253\008\253\231\252\199\252\132\252B\252\000\252\165\248Z\239Z\231\024\227\024\227\024\227\024\227\024\227\024\227\024\227\247\222\156\243\255\255{\239\024\227\024\227\024\227\024\227\024\227\024\227\024\227\025\231\156\235\173\241\000\252C\252\132\252\231\252\008\253)\253)\253J\253J\253k\253k\253\140\253\140\253\140\253\173\253\173\253\206\253\206\253\239\253\239\253\239\253\240\253\016\254\016\254\016\2541\2541\2541\2541\254\016\254\016\254\016\254\240\253\239\253\239\253\238\253\206\253\206\253\173\253\173\253\172\253\140\253\140\253k\253k\253J\253J\253)\253)\253\008\253\008\253\231\252\198\252\131\252\034\252\000\252\173\245|\2359\231\024\227\024\227\024\227\024\227\024\227\024\227\247\2229\231\222\251\255\127\156\243\247\222\024\227\024\227\024\227\024\227\024\227\024\227\024\227{\235\214\238B\248A\252d\252\198\252\008\253)\253)\253J\253J\253k\253k\253\140\253\140\253\173\253\173\253\205\253\206\253\206\253\239\253\239\253\240\253\016\254\016\2541\254R\254s\254\148\254\148\254s\254R\2541\254\016\254\016\254\239\253\239\253\239\253\206\253\206\253\205\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253)\253)\253\008\253\008\253\232\252\197\252b\252!\252B\244\214\242{\235\024\227\024\227\024\227\024\227\024\227\024\227\024\227\214\218{\239\255\127\255\127\222\251\024\227\247\222\024\227\024\227\024\227\024\227\024\223\247\2269\231|\239)\249\000\252c\252\165\252\008\253)\253*\253J\253J\253k\253l\253\140\253\140\253\173\253\173\253\206\253\206\253\239\253\239\253\015\254\016\254\016\2542\254s\254\181\254\214\254\214\254\214\254\214\254\181\254s\254Q\254\016\254\016\254\015\254\239\253\239\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253*\253)\253\009\253\008\253\231\252\164\252B\252\000\252)\245{\2399\231\248\222\024\227\024\227\024\227\024\227\024\227\247\222\214\218\222\251\222{\189w\189\2479\231\247\222\247\222\247\222\247\222\247\222\247\226\247\226\024\223|\239\148\242!\252B\252\132\252\199\252)\253)\253J\253k\253k\253\140\253\140\253\173\253\173\253\205\253\206\253\206\253\239\253\239\253\015\254\016\2541\254s\254\181\254\247\254\024\2559\2559\255\024\255\247\254\181\254s\2541\254\016\254\015\254\239\253\239\253\206\253\206\253\173\253\173\253\141\253\140\253l\253k\253k\253J\253*\253)\253)\253\008\253\198\252c\252!\252!\252\147\242|\235\024\227\247\222\247\222\247\222\247\222\247\222\024\227\214\218\024\227\255\255\255\127\156s\189w{\239\247\222\247\222\247\222\247\222\247\222\247\222\248\222\246\2229\235{\235*\245\000\252c\252\166\252\008\253I\253j\249k\253k\253\140\253\140\253\173\253\173\253\205\253\206\253\206\253\239\253\239\253\016\254\016\254R\254\181\254\247\2549\255Z\255{\255{\255Z\2559\255\247\254\148\254R\254\016\254\016\254\239\253\239\253\206\253\206\253\173\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253)\253)\253\231\252\165\252B\252\000\252*\245{\239Y\227\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218{\239\222{\222{\222{\222{\222\251\024\227\247\222\247\222\247\222\247\222\247\222\247\222\246\222\023\223|\235\214\238B\252B\252\132\252\199\252*\253J\253k\253k\253\140\253\140\253\173\253\173\253\206\253\206\253\206\253\239\253\239\253\016\2541\254s\254\214\254\024\255[\255\156\255\189\255\189\255\156\255Z\255\024\255\214\254s\2541\254\016\254\239\253\239\253\239\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253)\253\008\253\198\252c\252A\252B\248\182\242|\235\023\223\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218{\239{o{o\222{\222{\222{{\239\214\218\247\222\247\222\247\222\247\222\247\222\248\222\247\2228\231\156\239\205\245!\248B\252\165\252\008\253J\253k\253k\253\140\253\140\253\173\253\173\253\206\253\206\253\239\253\239\253\239\253\016\2541\254t\254\214\2549\255{\255\189\255\255\255\255\255\189\255{\255\024\255\214\254s\2541\254\016\254\015\254\239\253\239\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253J\253)\253\231\252\132\252B\252\000\252\205\245\156\2359\227\247\222\246\222\247\222\247\222\247\222\247\222\247\222\181\2149\231\189w\189w\189w\189w\189w\189w\189\247\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\218\246\222z\235Z\239\232\248!\252\131\252\198\252)\253k\253k\253\140\253\140\253\173\253\173\253\206\253\206\253\239\253\239\253\239\253\016\2541\254s\254\214\2548\255{\255\189\255\222\255\254\255\189\255{\255\024\255\214\254s\2541\254\016\254\015\254\239\253\239\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253)\253\232\252\165\252c\252 \252\231\244Z\239Z\235\247\222\247\218\247\222\247\222\247\222\247\222\247\222\214\218\214\218\156\243\156s\156s\156s\222{\222{\222{\255\127{\239\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\218\023\227|\239\246\242d\248B\252\132\252\198\252I\253k\253\139\253\140\253\173\253\173\253\205\253\206\253\238\253\239\253\015\254\016\2540\254s\254\181\254\024\255Z\255\156\255\188\255\189\255\155\255Z\255\024\255\181\254r\254\016\254\016\254\239\253\239\253\239\253\206\253\206\253\173\253\173\253\140\253\140\253k\253k\253J\253\008\253\198\252\131\252B\252c\248\214\238\156\239\248\226\215\222\247\222\247\222\247\222\247\222\247\222\247\222\181\2149\231\189w\156s\156s\156s\255\127\255\127\255\127\255\127\222\251\247\222\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\2229\227\187\235s\238C\248b\252\165\252\231\252*\253k\253\140\253\141\253\173\253\206\253\206\253\206\253\239\253\239\253\016\254\016\254R\254\148\254\214\254\024\255Z\255Z\255[\255Z\255\024\255\214\254\148\254Q\254\016\254\016\254\239\253\239\253\206\253\206\253\173\253\173\253\141\253\140\253\140\253k\253J\253)\253\198\252\131\252B\252B\248s\242\156\243\025\231\247\218\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\189\247\189w\189w\189w\189w\255\127\255\127\255\127\255\127\255\255{\239\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\222:\231\156\2390\242c\248c\252\165\252\232\252J\253\140\253\172\253\173\253\174\253\206\253\206\253\239\253\239\253\240\253\016\2541\254s\254\181\254\214\254\247\254\024\255\024\255\247\254\214\254\148\254s\2541\254\016\254\016\254\239\253\239\253\206\253\206\253\173\253\173\253\141\253\140\253\139\253j\253)\253\199\252\132\252b\252A\2521\242\157\2439\231\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\181\214Z\235\255\127\222{\222{\222{\222{\255\127\255\127\255\127\255\127\255\127\255\2559\231\214\218\247\222\247\222\247\222\247\222\247\222\247\222\215\222\248\218Z\231\156\239\016\242C\252c\252\164\252\231\252J\253\140\253\173\253\174\253\206\253\206\253\239\253\239\253\240\253\016\254\016\2541\254S\254\148\254\181\254\213\254\213\254\181\254\148\254R\2541\254\016\254\016\254\239\253\239\253\238\253\206\253\206\253\173\253\173\253\172\253\140\253k\253)\253\199\252\132\252B\252A\248\239\245\156\239Z\231\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\247\222\189\247\189w\189w\189w\189w\189w\255\127\255\127\255\127\255\127\255\127\255\127\189\247\214\218\247\222\247\222\247\222\247\222\247\222\247\222\248\218\246\222\247\222[\235\156\2391\246d\248c\252\196\252\231\252)\253k\253\173\253\173\253\206\253\206\253\239\253\239\253\239\253\015\254\016\2541\254Q\254R\254s\254s\254R\2542\254\017\254\016\254\016\254\239\253\239\253\239\253\206\253\206\253\173\253\173\253\173\253\140\253j\253\008\253\198\252\132\252C\252c\2481\242\156\239Z\235\247\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\156\243\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\222{\255\127{\239\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\246\222\247\226Z\231\156\239\147\238\134\248c\252\165\252\198\252(\253k\253\140\253\205\253\206\253\207\253\239\253\239\253\239\253\015\254\016\254\016\254\016\2541\2541\254\016\254\016\254\016\254\016\254\240\253\239\253\239\253\239\253\206\253\206\253\173\253\173\253\140\253J\253\008\253\166\252\132\252B\252\132\248s\242\157\243Z\231\247\218\215\226\247\222\247\222\247\222\247\222\247\222\247\222\247\222\181\214{\239\255\255\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\222{\222{\222{\222{\255\2559\231\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222Z\231\156\239\246\242)\245C\252\132\252\198\252\232\252)\253k\253\173\253\206\253\206\253\239\253\239\253\239\253\015\254\015\254\016\254\016\254\016\254\016\254\016\254\240\253\239\253\239\253\239\253\207\253\206\253\206\253\173\253\140\253k\253\009\253\199\252\165\252\131\252B\252\009\245\247\242\156\243:\235\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\181\214\024\227\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\222{\222{\222{\222{\222{\222{\255\127\222\251\247\222\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\2189\227\156\239z\239\239\241\165\248\131\252\165\252\230\252\008\253*\253k\253\173\253\206\253\239\253\239\253\240\253\239\253\239\253\015\254\016\254\015\254\239\253\239\253\239\253\239\253\206\253\206\253\173\253\140\253J\253(\253\230\252\197\252\164\252C\252\164\248\238\245Z\239\156\2399\231\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\222\251\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\251\247\222\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\222\024\227{\235\156\239\247\242k\245\164\248\132\252\166\252\231\252\231\252)\253J\253\139\253\140\253\173\253\206\253\206\253\206\253\206\253\206\253\206\253\206\253\173\253\173\253\140\253k\253J\253)\253\232\252\198\252\165\252c\252\132\248l\245\247\242\156\243{\235\024\227\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\189\247\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\255\189\247\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\218\247\2229\227\156\239\156\239\214\242\140\245\165\248\132\252\165\252\198\252\231\252\008\253(\253)\253J\253K\253k\253k\253k\253k\253J\253J\253)\253\008\253\231\252\199\252\165\252\131\252\131\252\166\248\140\245\213\242\156\243}\2399\227\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\214\156\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{{\239\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\218\247\222\247\222\247\218\247\222Y\231\156\239\156\239\024\239\016\242I\245\198\244\133\252\165\252\198\252\198\252\231\252\231\252\231\252\231\252\231\252\231\252\231\252\198\252\198\252\165\252\133\252\132\252\165\248)\245\016\246\023\243\156\243\156\239Z\231\247\226\246\218\247\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\189\247\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222{\189w\156\243\247\222\214\218\247\222\247\222\247\222\247\222\247\222\246\222\247\222\247\222\247\222\247\222\247\218\247\2269\235{\235\188\239{\243\246\242R\246\173\241*\249\231\252\198\252\166\252\165\252\165\248\165\252\165\252\165\252\197\252\231\252)\245\173\241Q\246\246\242{\243\189\239|\2399\231\247\222\247\218\215\222\246\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\156\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\255\189\247\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\024\227Z\231|\235\157\239\156\243{\2439\243\247\242\180\242s\242s\242s\242s\242\181\242\247\2429\243{\243\156\243\156\239{\235Z\235\024\227\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\181\214\181\2149\231\189w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189w9\2319\231\181\214\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\246\222\023\2278\227Y\231z\231{\235\156\239\156\239\156\243\156\239\156\239\156\239|\235z\235Y\2358\227\024\223\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\2149\231\189\247\156s\189w\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\189\2479\231\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\218\246\222\246\222\247\222\247\222\247\222\247\226\247\226\247\226\247\222\247\222\247\222\215\222\247\222\247\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\024\227\222\251\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{\239\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\246\222\246\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\218\247\218\246\218\246\218\247\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\214\214\218\189\247\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{oZ\2359\231\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\246\222\246\222\247\222\247\222\247\222\247\222\215\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\214\214\218Z\235{\239\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{o9g\156s{\239\024\227\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\214\214\218\214\218Z\235\222{{o\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{o9g{o{o{o\189\2479\231\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\181\214\214\2189\231\156\2439g9g\189w{o\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{o9g{o{o{o\189w{o\156\2439\231\214\218\214\218\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\181\214\214\218\024\2279\231\156\243\156s\156s\024c9g\189w{o\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{o9g{o{o{o\189w{o\156sZk9g{\239\247\222\247\222\214\218\214\218\214\218\214\218\214\218\214\218\214\218\247\222\247\222\247\222\247\222\247\222\247\222\214\218\214\218\214\218\214\218\181\214\214\218\214\218\024\227Z\235{\239\156\243{o\156s{o\156s\024c9g\189w{o\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189wZk\156s\156s{o9g{o{o{o\189w{o\156sZk9g{o\024c{oZ\235Z\235\024\2279\231\181\214\247\222\214\218\214\218\181\214\181\214\181\214\181\214\214\218\214\218\247\222\214\2189\231\024\227Z\2359\231\189\247\156s\156s\156sZk\156s{o\156s\024c9g\189w{o\222{\189w\189w\255\127\255\127\156s\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127"
)
