-- Copyright (C) 2010, Texas Instruments Incorporated
-- All rights reserved
--
-- factoring-trinomials.lua
--
-- This Nspire app demonstrates factoring trinomials geometrically.
-- 
-- John Powers  2010-06-25



----------------------------------------------------------------------------- Grid

Grid = {}

Grid.spacing = 0.05

-- Returns unit coordinates nearest grid point
function Grid.nearestUnitCoords(ux, uy)
    local spacing = Grid.spacing
    return math.floor(ux / spacing + 0.5) * spacing,
           math.floor(uy / spacing + 0.5) * spacing
end






----------------------------------------------------------------------------- Square

-- The square represents x*x geometrically

Square = class()

Square.length   = Grid.spacing * 10
Square.x        = Grid.spacing * 3
Square.y        = Grid.spacing * 3

function Square:init()
    self.x          = -Square.length
    self.y          = Square.y
    self.showHandle = false
    self.visible    = true
    self.leftslots  = {}
    self.topslots   = {}
    self.intersections = {}
    thePaintList:add(self)
    animation.zipto(self, Square.x, Square.y)
end


function Square:setxy(ux, uy)
    self.x = ux
    self.y = uy
end


function Square:intersectionCount()
    return #self.leftslots * #self.topslots
end


function Square:filledIntersectionCount()
    local count = 0
    for k, v in pairs(self.intersections) do
        count = count + 1
    end
    return count
end


function Square:paint(gc)
    local wx, wy = util.toWindowCoords(self.x, self.y)
    local ww, wh = util.toWindowCoords(self.length, self.length)

    -- Draw square interior
    --gc:setColorRGB(0x90, 0x90, 0x90)
    gc:setColorRGB(0x66, 0xAA, 0xFF)    -- light blue
    gc:fillRect(wx, wy, ww, wh)

    -- Draw border
    gc:setPen("medium", "smooth")
    gc:setColorRGB(0, 0, 0)
    gc:drawRect(wx, wy, ww, wh)

    -- Draw dimensions
    gc:setFont("serif", "i", 10)
    local label = "x"
    local root = #self.leftslots
    if root > 0 then
        label = label .. "-" .. root
    end
    gc:drawString(label, wx - gc:getStringWidth(label) - 5, wy + wh/2, "baseline")

    label = "x"
    root = #self.topslots
    if root > 0 then
        label = label .. "-" .. root
    end
    gc:drawString(label, wx + ww/2 - gc:getStringWidth(label)/2, wy - 2, "bottom")
end


function Square:roots()
    return #self.leftslots, #self.topslots
end


function Square:contains(ux, uy)
    return ux >= self.x and ux <= (self.x + self.length) and
           uy >= self.y and uy <= (self.y + self.length)
end


-- Returns true if the ux,uy coordinates are in the upper right
-- triangular region of the square.
function Square:upperRightContains(ux, uy)
    return ux - self.x >= uy - self.y
end


-- A tile was dropped on square. We are interested in it if
-- it aligns with an edge of the square and occupies a slice
-- not already occupied by another tile.
function Square:receiveTile(tile)

    if tile.type == "LongTile" then
        local nslot, nux, nuy

        -- Push horizontal tiles to the left edge and stacked above other tiles
        if tile.orientation == 0 then
            nslot, nux, nuy = self:findEmptyHSlot()
            tile.slots = self.leftslots
            tile.slot = nslot
            self.leftslots[nslot] = tile
    
        else -- Push vertical tiles to the top edge and stacked to the left of other tiles
            nslot, nux, nuy = self:findEmptyVSlot()
            tile.slots = self.topslots
            tile.slot = nslot
            self.topslots[nslot] = tile
        end
        animation.zipto(tile, nux, nuy, function() tile:setInside(); self:coverIntersections() end )

    elseif tile.type == "SmallTile" then
        local slot, nux, nuy = self:findEmptyIntersection()
        if slot ~= nil then
            tile.slots = self.intersections
            tile.slot = slot
            self.intersections[slot] = tile
            animation.zipto(tile, nux, nuy, function() tile:setInside() end )
        else
            tile:returnHome()
        end
    end
end


-- Remove a tile from the square
function Square:bounceTile()
    for _, tile in pairs(self.intersections) do
        tile:setOutside()
        tile:returnHome()
    end
    self.intersections = {}

    self:packTiles()
    self:coverIntersections()
end



-- Places small tiles over the intersection of horizontal and vertical
-- tiles on the square.
function Square:coverIntersections()
    local it = IntersectionTile.tiles
    while #it > 0 do
        local i = it[#it]
        i:discard()
    end
    for h = 1, #self.leftslots do
        local y = self.y + self.length - h*Grid.spacing
        for v = 1, #self.topslots do
            local x = self.x + self.length - v*Grid.spacing
            IntersectionTile(x, y)
        end
    end
end


function Square.auxpack(slots, coordf)
    -- Move tiles down to fill empty slots
    local nslot = 1
    while nslot < table.maxn(slots) do
        -- Find first empty slot
        if slots[nslot] == nil then                 -- found an empty slot
            local higher = nslot + 1
            while higher <= table.maxn(slots) do               -- look for an occupied slot
                local occupant = slots[higher]
                if occupant ~= nil then             -- found an occupied slot
                    slots[nslot] = occupant         -- move it down
                    slots[higher] = nil
                    occupant.slots = slots
                    occupant.slot = nslot
                    animation.zipto(occupant, coordf(nslot))
                    break
                end
                higher = higher + 1
            end
        end
        nslot = nslot + 1
    end
end
    
    
function Square:packTiles()
    -- Move horizontal tiles down to fill empty slots
    self.auxpack(self.leftslots, function(n) return self.x, self.y + self.length - Grid.spacing*n end)

    -- Move vertical tiles right to fill empty slots
    self.auxpack(self.topslots, function(n) return self.x + self.length - Grid.spacing * (n-1), self.y end)
end


-- Returns slot number and ux,uy location for a horizontal tile
function Square:findEmptyHSlot()
    local e = #self.leftslots + 1
    return e, self.x, self.y + self.length - e * Grid.spacing
end


-- Returns slot number and ux,uy location for a vertical tile
function Square:findEmptyVSlot()
    local e = #self.topslots + 1
    return e, self.x + self.length - (e-1) * Grid.spacing, self.y
end


-- Returns slot and ux,uy location of an empty intersection
function Square:findEmptyIntersection()
    for row = 1, #self.leftslots do
        for col = 1, #self.topslots do
            local slot = row .. "-" .. col
            if self.intersections[slot] == nil then
                return slot, self.x + self.length - col*Grid.spacing,
                             self.y + self.length - row*Grid.spacing
            end
        end
    end
    return nil
end






----------------------------------------------------------------------------- Tile

Tile = class()

Tile.handleRadius = 0.008



function Tile:init(ux, uy, uw, uh, visible)
    self.x      = ux
    self.y      = uy
    self.w      = uw
    self.h      = uh
    self.staging_x = ux
    self.staging_y = uy
    self.color  = {0x80, 0x80, 0x80}
    self.border = "smooth"
    self.angle  = 0
    self.showHandle = true
    self.visible = visible

    table.insert(self.tiles, self)  -- keep track of all tiles
    thePaintList:add(self)
end


-- Discards a tile.
function Tile:discard()
    thePaintList:remove(self)
    table.removeObj(self.tiles, self)
end


function Tile:setxy(ux, uy)
    self.x = ux
    self.y = uy
end


function Tile:returnHome()
    if self.visible then
        animation.zipto(self, self.staging_x, self.staging_y)
    end
end


-- Show a tile by sliding it in from the bottom of the window
function Tile:show()
    if not self.visible then
        local ux, uy = util.toUnitCoords(0, platform.window:height())
        self.x = self.staging_x
        self.y = uy
        self.visible = true
        self:returnHome()
    end
end


-- Hide a tile by sliding it off the bottom of the window
function Tile:hide()
    if self.visible then
        local ux, uy = util.toUnitCoords(0, platform.window:height())
        animation.zipto(self, self.x, uy, function() self.visible = false end )
    end
end



-- Returns a tile that contains wx, wy window coordinates.
-- Returns nil if none found.
function Tile.findTileAtWindowCoords(wx, wy)
    local ux, uy = util.toUnitCoords(wx, wy)
    return Tile.findTileAtUnitCoords(ux, uy)
end


-- Looks for a tile that contains ux,uy.
function Tile.searchTileList(list, ux, uy)
    for _, t in ipairs(list) do
        if t:contains(ux, uy) then
            return t
        end
    end
    return nil
end


-- Returns tile at ux,uy
function Tile.findTileAtUnitCoords(ux, uy)
    return Tile.searchTileList(LongTile.tiles, ux, uy) or Tile.searchTileList(SmallTile.tiles, ux, uy)
end


function Tile:setInside() end       -- implemented by subclasses
function Tile:setOutside() end


-- Handles picking up a tile
function Tile:pickup()
    -- Remove myself from square
    if self.slot then
        self:setOutside()
        self.slots[self.slot] = nil
        self.slot = nil
    end
end

-- Handles dropping a tile on mouseUp
function Tile:drop(wx, wy)
    local ux, uy = Grid.nearestUnitCoords(util.toUnitCoords(wx, wy))

    if theSquare:contains(ux, uy) then
        theSquare:receiveTile(self)
    else
        animation.zipto(self, ux, uy)
    end    
end


function Tile:paint(gc)
    local angle = self.angle
    local x     = self.x
    local y     = self.y
    local hr    = self.handleRadius
    local hd    = hr + hr
    local w     = self.w
    local h     = self.h

    local sa    = math.sin(angle)
    local ca    = math.cos(angle)

    -- Upper right corner
    local urx = x + ca * w
    local ury = y - sa * w

    -- Lower right corner
    local lrx = urx + sa * h
    local lry = ury + ca * h

    -- Lower left corner
    local llx = x + sa * h
    local lly = y + ca * h

    local corners = table.toWindowCoords({x, y, urx, ury, lrx, lry, llx, lly, x, y})

    -- Draw tile interior
    gc:setColorRGB(unpack(self.color))
    gc:fillPolygon(corners)

    -- Draw tile border
    gc:setPen("thin", self.border)
    gc:setColorRGB(0, 0, 0)
    gc:drawPolyLine(corners)

    -- Draw handle
    if self.showHandle then
        local wx, wy = util.toWindowCoords(x - hr, y - hr)
        local ww, wh = util.toWindowCoords(hd, hd)
        gc:fillArc(wx, wy, ww, ww, 0, 360)
    end
end


-- Returns true if the tile handle contains ux,uy unit coordinates
function Tile:contains(ux, uy)
    return self.showHandle and math.sqrt((ux - self.x)^2 + (uy - self.y)^2) <= 2*self.handleRadius
end






----------------------------------------------------------------------------- LongTile

-- A long tile is x by 0.1x in size so it covers 1/10 of the square. Ten
-- tiles exactly cover the square.

LongTile = class(Tile)
LongTile.type = "LongTile"

LongTile.tiles = {}

LongTile.maxtiles = 9

function LongTile:init(ux, uy)
    Tile.init(self, ux, uy, Square.length, 0.1*Square.length, false)
    self.orientation = 0
    self:setOutside()
end


function LongTile:setxy(ux, uy)
    if not self.rotating then
        local vert = -math.pi/2
        if theSquare:upperRightContains(ux, uy) then
            if self.angle ~= vert then
                animation.rotate(self, vert)
                self.orientation = vert
            end
        elseif self.angle ~= 0 then
            animation.rotate(self, 0)
            self.orientation = 0
        end
    end
    self.x = ux
    self.y = uy
end



-- Sets color and border of tile when it is outside the square
function LongTile:setOutside()
    self.color = {0x99, 0xBB, 0xAA} --    {0x50, 0x50, 0x50}
    self.border = "smooth"
end


-- Sets color and border of tile when it is inside the square
function LongTile:setInside()
    self.color = {0xFF, 0xFF, 0xFF}
    self.border = "dashed"
end


function LongTile:hide()
    self:pickup()
    Tile.hide(self)
end






----------------------------------------------------------------------------- SmallTile

-- A small tile is 0.1x by 0.1x in size so it covers 1/100 of the square. One hundred
-- tiles exactly cover the square.

SmallTile = class(Tile)
SmallTile.type = "SmallTile"

SmallTile.tiles = {}

SmallTile.maxtiles = 12     -- a multiple of 4


function SmallTile.visibleCount()
    local count = 0
    for _, tile in ipairs(SmallTile.tiles) do
        if tile.visible then
            count = count + 1
        end
    end
    return count
end


function SmallTile:init(ux, uy)
    Tile.init(self, ux, uy, 0.1*Square.length, 0.1*Square.length, false)
    self:setOutside()
end


function SmallTile:setInside()
    self.color = {0xFF, 0xFF, 0xFF}
end


function SmallTile:setOutside()
    self.color = {0x99, 0xBB, 0xAA}    -- {0x80, 0x80, 0x80}
end






----------------------------------------------------------------------------- IntersectionTile

-- An intersection tile colors the intersection of a vertical and horizontal tile.

IntersectionTile = class(Tile)
IntersectionTile.type = "IntersectionTile"

IntersectionTile.tiles = {}

function IntersectionTile:init(ux, uy)
    Tile.init(self, ux, uy, 0.1*Square.length, 0.1*Square.length, true)
    self.showHandle = false
    self.color = {0x66, 0xAA, 0xFF}    -- {0x50, 0x50, 0x50}
end





----------------------------------------------------------------------------- Text

Trinomial = class()

function Trinomial:init()
    self.visible = true
    thePaintList:add(self)
end


SUPER_2 = string.uchar(0xb2)


function Trinomial:paint(gc)
    local ux = Square.x + Square.length/2
    local wx, wy = util.toWindowCoords(ux, 0)

    local text = "x" .. SUPER_2 .. "-"
    local b = var.recall("b")
    if b > 1 then
        text = text .. b
    end
    text = text .. "x+" .. var.recall("c")

    local icount = theSquare:intersectionCount()
    local fcount = theSquare:filledIntersectionCount()
    local tcount = SmallTile.visibleCount()

    if icount == fcount and fcount == tcount then
        local root1, root2 = theSquare:roots()
        text = text .. "=(x-" .. root1 .. ")(x-" .. root2 .. ")"
    end

    local length = gc:getStringWidth(text)

    gc:setColorRGB(0, 0, 0)
    gc:setFont("serif", "i", 10)
    gc:drawString(text, wx - length/2, wy, "top")
end


----------------------------------------------------------------------------- Animation

animation = {}

animation.list = {}

-- Starts the animation
function animation.start()
    if #animation.list == 1 then
        timer.start(0.1)
    end
end


-- Causes an object to zip to unit coordinates ux,uy
function animation.zipto(obj, ux, uy, arrivalf)
    -- Add object to list of animated objects
    table.insert(animation.list, function() return animation.zipstep(obj, ux, uy, arrivalf) end)
    animation.start()
end


-- Moves an object along the line from its location to its destination.
-- Return true each time it is called during the animation until the object
-- arrives at its destination. Calls "arrivalf" when the object arrives at
-- its destination.
function animation.zipstep(obj, ux, uy, arrivalf)
    local ox, oy = obj.x, obj.y
    local dx = ux - ox
    local dy = uy - oy
    local dist = math.sqrt(dx*dx + dy*dy)
    local keepgoing = true

    local step
    if dist > 0.1 then
        step = 0.1
    elseif dist > 0.01 then
        step = 0.01
    else
        step = dist
        keepgoing = false
    end

    local xstep, ystep
    if dx == 0 then
        xstep = 0
        if dy >= 0 then
            ystep = step
        else
            ystep = -step
        end
    else
        local aa = dy / dx
        local denom = 1 / math.sqrt(1 + aa*aa)
        xstep = step * denom            -- step * cos(atan(dy/dx))
        ystep = step * aa * denom       -- step * sin(atan(dy/dx))
        if dx < 0 then
            xstep = -xstep
            ystep = -ystep
        end
    end

    obj:setxy(ox + xstep, oy + ystep)
    if not keepgoing and arrivalf then
        arrivalf()
    end
    return keepgoing                    -- keep going until obj arrives at destination
end


-- Causes an object to rotate to a new angular orientation.
function animation.rotate(obj, angle)
    -- Is obj already rotated to target angle?
    if obj.angle ~= angle then
        local step = (angle - obj.angle)/5
        obj.rotating = true
        table.insert(animation.list, function() return animation.rotatestep(obj, angle, step) end)
        animation.start()
    end
end


-- Rotates an object about handle.
-- Returns true until the target angle is reached.
function animation.rotatestep(obj, angle, step)
    local keepgoing = true
    local newangle

    if math.abs(angle - obj.angle) > math.abs(step) then
        newangle = obj.angle + step
    else
        newangle = angle
        keepgoing = false
        obj.rotating = false
    end

    obj.angle = newangle

    return keepgoing
end


function animation.step()
    -- Step each object in animation list
    local newlist = {}
    for _, astep in ipairs(animation.list) do
        if astep() then
            table.insert(newlist, astep)
        end
    end

    -- All animations complete?
    if #newlist == 0 then
        timer.stop()
    end

    -- Keep new list of animated objects
    animation.list = newlist

    platform.window:invalidate()
end








----------------------------------------------------------------------------- Utilities

util = {}

-- Converts unit square coordinates to window coordinates
function util.toWindowCoords(ux, uy)
    local w = platform.window
    local scale = math.min(w:height(), w:width())
    return ux * scale, uy * scale
end


-- Returns a table of unit square coordinates converted to window coordinates
function table.toWindowCoords(t)
    local w = platform.window
    local scale = math.min(w:height(), w:width())
    local newt = {}

    for i, v in ipairs(t) do
        newt[i] = scale * v
    end

    return newt
end


-- Converts window coordinates to unit square coordinates
function util.toUnitCoords(wx, wy)
    local w = platform.window
    local scale = math.min(w:height(), w:width())
    return wx / scale, wy / scale
end


-- Returns a function that tracks mouse movements. Obj is updated to
-- follow the mouse.
function util.mouseTracker(obj, wx, wy)
    local oldx, oldy = util.toUnitCoords(wx, wy)
    return function(wx, wy)
        local ux, uy = util.toUnitCoords(wx, wy)
        obj:setxy(obj.x + ux - oldx, obj.y + uy - oldy)
        oldx = ux
        oldy = uy
        platform.window:invalidate()
    end
end


-- Extend table module to remove an object from a table
function table.removeObj(t, obj)
    for i, o in ipairs(t) do
        if o == obj then
            return table.remove(t, i)
        end
    end
end






----------------------------------------------------------------------------- Paint List

PaintList = class()

function PaintList:init()
    self.paintlist = {}
end


function PaintList:paint(gc)
    for _, obj in ipairs(self.paintlist) do
        if obj.visible then
            obj:paint(gc)
        end
    end
end


function PaintList:add(obj)
    table.insert(self.paintlist, obj)
end


function PaintList:remove(obj)
    table.removeObj(self.paintlist, obj)
end


function PaintList:bringToFront(obj)
    for i, o in ipairs(self.paintlist) do
        if o == obj then
            table.remove(self.paintlist, i)
            table.insert(self.paintlist, o)
            return
        end
    end
end


thePaintList = PaintList()






----------------------------------------------------------------------------- Event handlers

function on.create()
    local b = var.recall("b")
    local c = var.recall("c")

    if not b then
        var.store("b", 1)           -- x^2 + bx + c = 0
        b = 1
    end
    var.monitor("b")

    if not c then
        var.store("c", 1)
        c = 1
    end
    var.monitor("c")

    theSquare = Square()

    local x = Square.x
    local dy = Grid.spacing
    local y = Square.y + Square.length + dy

    -- Create tiles for coefficient "b"
    for i = 1, 5 do
        LongTile(x, y)
        y = y + dy
    end
    x = Square.x + Square.length
    y = Square.y + Square.length + dy
    for i = 1, 4 do
        LongTile(x, y)
        y = y + dy
    end

    -- Create tiles for coefficient "c"
    local dx = Grid.spacing
    y = Square.y + 5*dy
    for row = 1, SmallTile.maxtiles/4 do
        x = Square.x + Square.length + dx
        for col = 1, 4 do
            SmallTile(x, y)
            x = x + dx
        end
        y =  y + dy
    end

    for t = 1, b do
        local tile = LongTile.tiles[t]
        tile:show()
    end
    for t = 1, c do
        local tile = SmallTile.tiles[t]
        tile:show()
    end

    Trinomial()    
end


function on.mouseDown(wx, wy)
    -- Look for a tile to grab
    local tile = Tile.findTileAtWindowCoords(wx, wy)

    if tile ~= nil then
        tile:pickup()
        if tile.type == "LongTile" then
            theSquare:bounceTile()
        end
        thePaintList:bringToFront(tile)

        -- Move tile around with mouse pointer
        on.mouseMove = util.mouseTracker(tile, wx, wy)

        -- Drop tile
        on.mouseUp = function(wx, wy)
            on.mouseMove    = nil
            on.mouseUp      = nil
            tile:drop(wx, wy)
        end
    end
end


function on.paint(gc)
    thePaintList:paint(gc)
end


function on.varChange(varlist)
    local b = var.recall("b")
    local c = var.recall("c")

    -- Make sure variable was not deleted
    if not b or not c then
        return -3           -- veto deleted variable
    end

    -- Assure numeric values
    if type(b) ~= "number" or type(c) ~= "number" then
        return -2           -- veto type change
    end

    -- Assure b in [1..9]
    if b < 1 or b > LongTile.maxtiles then
        return -1           -- veto range
    end

    -- Assure c in [1..12]
    if c < 1 or c > SmallTile.maxtiles then
        return -1           -- veto range
    end

    -- Make some tiles (in)visible
    local btiles = LongTile.tiles
    for t = 1, b do
        local tile = btiles[t]
        tile:show()
    end
    for t = b+1, LongTile.maxtiles do
        local tile = btiles[t]
        tile:hide()
    end

    local ctiles = SmallTile.tiles
    for t = 1, c do
        local tile = ctiles[t]
        tile:show()
    end
    for t = c+1, SmallTile.maxtiles do
        local tile = ctiles[t]
        tile:hide()
    end
    theSquare:bounceTile()

    return 0        -- Okay
end


function on.timer()
    animation.step()
end
