-- Draw the Mandelbrot set
-- mandelbrot.lua
-- Author: John Powers  2010-10-21


ZoomScaleFactor = 4




------------------------------------------------------------------ ColorModel
ColorModel = class()

ColorModel.R = 6
ColorModel.G = 6
ColorModel.B = 7

ColorModel.totalColors = ColorModel.R * ColorModel.G * ColorModel.B

function ColorModel:init()
    for color = 0, ColorModel.totalColors-1 do
        local r = ColorModel.col(math.floor(color / (ColorModel.G * ColorModel.B)), ColorModel.R)
        local g = ColorModel.col(math.floor(color / ColorModel.B), ColorModel.G)
        local b = ColorModel.col(color, ColorModel.B)
        table.insert(self, {r, g, b})
    end
end


function ColorModel:get(ndx)
    return self[ndx+1]
end


function ColorModel.col(i, r)
    if r == 1 then
        return 0
    end
    return math.floor(255*(i % r) / (r-1))
end


theColors = ColorModel()






------------------------------------------------------------------ Mandelbrot

Mandelbrot = class()

Mandelbrot.STEP_SIZE = 30

function Mandelbrot:init(x1, y1, x2, y2)
    self.limit = 4
    self.maxCount = theColors.totalColors

    self.x1 = x1 or -2
    self.x2 = x2 or  1
    self.y1 = y1 or -1.5
    self.y2 = y2 or  1.5

    self.pixels = {}
    self.paintRow = 1
end


function Mandelbrot:resize()
    self.dx = (self.x2 - self.x1) / platform.window:width()

    local halfy = platform.window:height()/2
    local cy = (self.y1 + self.y2)/2
    self.y1 = cy - halfy * self.dx
    self.y2 = cy + halfy * self.dx

    document.markChanged()
end


function Mandelbrot:setCenter(cx, cy)
    local halfx = (self.x2 - self.x1)/2
    self.x1 = cx - halfx
    self.x2 = cx + halfx

    local halfy = (self.y2 - self.y1)/2
    self.y1 = cy - halfy
    self.y2 = cy + halfy
end


function Mandelbrot:scale(s)
    local cx = (self.x1 + self.x2)/2
    local nw = (self.x2 - self.x1)/2 * s
    local cy = (self.y1 + self.y2)/2
    local nh = (self.y2 - self.y1)/2 * s

    self.dx = self.dx * s
    self.x1 = cx - nw
    self.x2 = cx + nw
    self.y1 = cy - nh
    self.y2 = cy + nh
end


function Mandelbrot:toMcoord(wx, wy)
    return wx*self.dx + self.x1, wy*self.dx + self.y1
end


function Mandelbrot:startCalc()
    self.calcRow = 0
    return coroutine.create(function() self:calc() end)
end


function Mandelbrot:calc()
    local maxCount = self.maxCount
    local yieldCount = 0
    self.pixels = {}
    for wy = 0, platform.window:height()-1 do
        self.calcRow = wy
        local pixrow = {}
        for wx = 0, platform.window:width()-1 do
            local x, y = self:toMcoord(wx, wy)
            local cx = x
            local cy = y
            local lp = 0
            repeat
                lp = lp + 1
                cx, cy = cx*cx - cy*cy + x, 2*cx*cy + y
            until lp >= maxCount or (cx*cx)+(cy*cy) > self.limit

            if lp >= maxCount then
                lp = 0
            end
            table.insert(pixrow, string.char(lp))
        end
        table.insert(self.pixels, table.concat(pixrow))
        yieldCount = yieldCount + 1
        if yieldCount >= self.STEP_SIZE then
            coroutine.yield()
            yieldCount = 0
        end
    end
end


function Mandelbrot:paint(gc)
    if self.paintRow >= #self.pixels then
        self.paintRow = 1
    end
    for wy = self.paintRow, #self.pixels do
        local pixrow = self.pixels[wy]
        for wx = 1, #pixrow do
            local color = pixrow:sub(wx, wx):byte()
            pixel(gc, wx-1, wy-1, color)
        end
    end
end


theMandelbrotSet = Mandelbrot()






------------------------------------------------------------------ Event Handlers


function on.resize()
    theMandelbrotSet:resize()
    theCalculation = theMandelbrotSet:startCalc()
    timer.start(0.01)
end


function on.paint(gc)
    theMandelbrotSet:paint(gc)
    if theCalculation then
        timer.start(0.01)
    end
end


-- Zooms in at clicked point
function on.mouseDown(wx, wy)
    local cx, cy = theMandelbrotSet:toMcoord(wx, wy)
    theMandelbrotSet:setCenter(cx, cy)
    theMandelbrotSet:scale(1/ZoomScaleFactor)
    on.resize()
end


function on.charIn(char)
    -- Zoom out?
    if char == "o" then
        theMandelbrotSet:scale(ZoomScaleFactor)
        on.resize()

    -- Reset?
    elseif char == "r" then
        theMandelbrotSet = Mandelbrot()
        on.resize()
    end
end


function on.timer()
    timer.stop()
    if theCalculation then
        local startRow = theMandelbrotSet.calcRow
        if not coroutine.resume(theCalculation) then
            theCalculation = nil
        end
        local stopRow = theMandelbrotSet.calcRow
        if startRow ~= stopRow then
            platform.window:invalidate(0, startRow, platform.window:width(), stopRow - startRow)
            print(("timer: row %d to %d"):format(startRow, stopRow))
        end
    end
end


function on.save()
    local m = theMandelbrotSet
    return {x1 = m.x1, y1 = m.y1, x2 = m.x2, y2 = m.y2}
end


function on.restore(state)
    theMandelbrotSet = Mandelbrot(state.x1, state.y1, state.x2, state.y2)
end



------------------------------------------------------------------ Utility routines

function pixel(gc, x, y, color)
    gc:setColorRGB(unpack(theColors:get(color)))
    gc:fillRect(x, y, 1, 1)
end

