-- multiplying-fractions.lua
-- Interactive fraction multiplication model
-- 
-- Copyright (C) 2010, Texas Instruments Incorporated
-- All rights reserved
--
-- John Powers  2010-08-29







------------------------------------------------------------------------------ Image
Image = {}

Image.rawImages = {
    knobLeft    = "\013\000\000\000\017\000\000\000\000\000\000\000\026\000\000\000\016\000\001\000\147\250\189\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\238\249\238\249\246\254\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\237\249\237\249\172\249\238\2498\255\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\172\249\172\249\171\249\138\249\171\2490\250\156\255\255\127\255\127\255\127\255\127\255\127\255\127\171\249\138\249\246\254\238\249j\249\171\249\171\249\180\254\222\127\255\127\255\127\255\127\255\127\171\249\138\249\023\255\023\255\180\254\172\249\138\249\138\249\205\249\023\255\255\127\255\127\255\127\171\249\138\249\023\255\023\255\023\255\023\255Q\250\138\249\138\249\171\249\015\250z\255\255\127\171\249\138\249\023\255\023\255\023\255\246\254\023\255\246\254\015\250I\249I\249'\249'\245\171\249\138\249\023\255\023\255\023\255\023\255\023\255\247\254\214\250\205\241\006\229\230\220\230\224\171\249\138\249\023\255\023\255\023\255\247\254\246\2500\242\007\225\197\208\164\200I\2139\243\171\249\138\249\023\255\023\255\247\250s\246I\229\197\212\164\200\197\196\148\230\222\127\255\127\171\249j\249\023\255\180\250\139\233\229\216\164\200\164\196\239\217\222{\255\127\255\127\255\127\171\249I\249\238\241\230\220\197\208\164\196j\205{\247\255\127\255\127\255\127\255\127\255\127I\249\007\241\229\216\164\196\007\201\247\234\255\127\255\127\255\127\255\127\255\127\255\127\255\127'\245\230\220\230\200R\226\222\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\006\229\238\225\156\251\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\1279\251\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127",
    knobRight   = "\014\000\000\000\017\000\000\000\000\000\000\000\028\000\000\000\016\000\001\000\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127z\255\237\249\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\189\255\180\250\205\249H\249\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\255\023\2550\250\238\249j\249\007\237\255\127\255\127\255\127\255\127\255\127\255\127\255\127Z\255r\2500\250\237\249\205\249'\245\230\220\255\127\255\127\255\127\255\127\255\127\156\255\147\2500\250\238\249\205\249r\250\246\254\006\233\229\216\255\127\255\127\255\127\222\255\247\2540\250\015\250\205\2490\250\246\254\023\255\247\250\006\229\229\216\255\127\255\1279\255Q\250\015\250\237\249\238\249\213\254\023\255\023\255\023\255\247\250\006\229\229\216j\249\205\249\238\249\237\249\205\249\147\254\023\255\023\255\023\255\023\255\023\255\247\250\006\229\229\216I\241j\249I\249I\249Q\250\246\254\023\255\023\255\023\255\023\255\023\255\247\250\006\229\229\216\254\255\180\246'\229\006\229\006\233j\241\147\250\023\255\023\255\023\255\246\254\023\251\006\233\229\216\255\127\255\127\222\255\015\234\229\216\230\224\006\233\172\241\213\250\023\255\023\255\247\250\006\229\229\216\255\127\255\127\255\127\255\127z\251\139\225\230\220\230\224'\237\015\246\247\250\247\250\006\229\229\216\255\127\255\127\255\127\255\127\255\127\255\127\246\242(\225\230\220\006\229H\237R\246\006\229\229\216\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\255r\238\229\216\230\224\230\224\230\220\197\212\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\156\251\205\229\197\212\197\212\197\208\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\1279\247I\217\197\204\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\222\127\181\238",
    knobTop     = "\017\000\000\000\014\000\000\000\000\000\000\000\034\000\000\000\016\000\001\000>\194\030\190>\194>\194>\194>\194>\194>\194>\194>\194>\194>\194>\194\253\185\189\177}\169\222\218_\235\221\181\253\185\253\185\253\185\253\185\253\185\253\185\253\185\253\185\253\185\221\181\189\177}\169:\161x\173\255\127\255\127>\194}\169\189\177\030\227\030\227?\227\030\227?\227\030\227\030\227?\227\158\206\024\157\245\152<\231\255\127\255\127\159\243\092\165]\165>\194\030\227\030\227\030\227\254\222\030\227\030\223\254\222X\165\244\152\181\181\255\127\255\127\255\127\255\127\157\206\092\165}\169\222\214\030\227\030\227\030\227\030\227\030\223\027\190\246\152\210\148\157\243\255\127\255\127\255\127\255\127\223{\156\173\092\165\253\185\030\227\254\222\030\227\030\223\221\2188\161\211\152X\202\255\127\255\127\255\127\255\127\255\127\255\127\254\222\092\165\092\165\190\210\030\227\030\227\030\223\186\181\245\152\019\161\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\221\181\092\165\157\173\030\227\030\223\157\206\022\157\210\148\218\218\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127^\235\092\165\092\165^\202\254\222y\169\244\152t\173\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127]\202\092\165|\169<\198\246\152\210\148]\235\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\191w|\169:\1618\161\211\152\023\194\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\221\2149\161\245\152\210\148\222{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\024\157\244\152\186\214\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127;\161\245\152\223{\255\127\255\127\255\127\255\127\255\127\255\127\255\127",
    knobBottom  = "\017\000\000\000\014\000\000\000\000\000\000\000\034\000\000\000\016\000\001\000\255\127\255\127\255\127\255\127\255\127\255\127\255\127}\169z\173\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\253\185\253\185_\235\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\159\243>\194\221\181\221\181\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127~\206\253\185\189\177}\169\222\218\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127>\227>\194\221\181>\198]\165|\169\223{\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\223\251^\198\253\185^\198\254\222}\169\092\165}\202\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\255\127\254\218\030\190\253\185\254\222\030\227^\198]\165\092\165\159\243\255\127\255\127\255\127\255\127\255\127\255\127\255\127\159\243>\194\253\185~\206\030\227\030\227\254\222\157\177\092\165\253\185\255\127\255\127\255\127\255\127\255\127\255\127\255\127\158\206\253\185\253\185\030\227\030\227\030\227\030\227\158\206\092\165\092\165\030\227\255\127\255\127\255\127\255\127\255\127_\235>\194\221\181\190\214\030\227\030\227\030\227\030\227\030\227\221\181\092\165\156\173\255\127\255\127\255\127\255\127\255\127^\202\253\185>\194\030\227\030\227\030\227\030\227\254\222\030\227\222\214]\165\092\165\157\210\255\127\255\127\255\127\254\222\253\185\189\177\222\214\030\227\030\223\030\223\254\222\030\223\030\223\030\223\028\190:\161Z\165\159\243\255\127\191\247\189\177]\1659\161\022\157\022\157\246\152\022\157\246\152\246\152\022\157\246\152\246\152\022\157\022\157\026\190\255\127\030\194:\161\246\152\211\152\210\148\209\148\209\148\209\148\209\148\209\148\209\148\209\148\210\148\209\148\210\148\244\152=\231",

    sliderWellRed = "\145\000\000\000\006\000\000\000\000\000\000\000\034\001\000\000\016\000\001\000\255\127\247\242\139\217\197\204\197\204\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\197\208\229\216\172\229\023\247\255\127\247\242\197\208\197\208\197\212\229\216\229\216\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\230\220\230\220\230\220\230\220\230\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\229\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\220\230\224\230\224\006\233'\237\023\255\239\233\230\220\230\224\006\229\006\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\237\007\241'\245'\249H\249Q\250\015\242\007\237'\245'\249H\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249I\249i\249j\249\139\249\171\249r\250\023\251H\249i\249\138\249\171\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\172\249\204\249\205\249\205\249\237\249Y\255\255\1278\2550\250\204\249\205\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\238\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\237\249\238\249\238\249\238\249r\250Z\255\255\127",
    sliderWellBlue = "\145\000\000\000\006\000\000\000\000\000\000\000\034\001\000\000\016\000\001\000\255\127\251\222\149\177\210\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\211\152\243\152\244\152\245\152\185\177\029\227\255\127\252\222\244\152\244\152\245\152\245\152\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\246\156\023\157\024\157\025\161;\161\030\227\025\190\022\157\023\157\025\157:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161:\161;\161<\161]\165}\169^\198\027\194:\161<\161]\165}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169}\169\157\173\157\173\189\177\189\177~\206\030\227]\165}\169\189\177\189\177\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\221\181\253\185\253\185\253\189_\235\255\127>\231>\194\221\181\253\185\253\185\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\253\189\029\190\029\190\158\206_\235\255\127",

    beadBlue    = "\009\000\000\000\009\000\000\000\000\000\000\000\018\000\000\000\016\000\001\000\156s|\239\029\227\189\214^\198}\206\220\218\092\235\156s|\239\253\218~\202^\202>\194\253\185\189\177]\202\092\235\253\222^\198^\198>\194\253\185\189\177}\169\092\165\155\210}\206\030\190\253\189\221\181\157\177}\169<\161\025\161\216\181\221\181\189\177\189\177}\169]\165;\161\024\157\245\152\243\152\029\190]\165\092\165;\161\025\161\022\157\243\152\209\148\148\177\156\210:\161\024\157\023\157\244\152\210\148\175\148\174\144V\202\092\235\025\194\245\152\211\152\176\148\174\144\140\144\211\185[\231\156s[\235x\206\147\173\174\148q\173U\202[\235\156s",
    beadRed     = "\009\000\000\000\009\000\000\000\000\000\000\000\018\000\000\000\016\000\001\000\156s{\243\024\247\180\2500\250r\250\213\246Z\243\156s{\243\214\246Q\250Q\250\015\250\205\249\139\249Q\246Z\243\247\2460\2500\250\015\250\205\249\139\249I\249'\245\147\242r\250\238\249\237\249\172\249\138\249H\249'\245\006\233\172\229\172\249\171\249\138\249i\249'\249\007\237\230\228\229\216\197\208\014\246H\249'\245\007\241\006\233\230\220\197\208\164\200\139\209\147\242\007\237\006\229\230\220\197\212\164\200\164\192\131\184R\222Z\243\015\234\197\216\197\204\164\196\131\184\131\180\206\205Y\239\156sZ\239r\226\139\209\164\184j\197R\214Y\239\156s",

    checkboxSelected = "\013\000\000\000\013\000\000\000\000\000\000\000\026\000\000\000\016\000\001\000\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255s\206k\173\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\016\194)\165\239\189\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\181\214\008\161\148\2101\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243{\239)\165\206\185\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\206\185J\169{\239\156\2431\198\008\161\008\161\255\255\156\243{\2391\198{\239\181\214\008\161\024\227\156\243\156\2431\198\008\161\008\161\255\255\156\243s\206\008\1611\198)\1651\198\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\024\227\008\161\008\161J\169{\239\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\173\181\008\161\181\214\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\2439\231\247\222\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\2551\1981\1981\1981\1981\1981\1981\1981\1981\1981\198\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161",
    checkboxUnselected = "\013\000\000\000\013\000\000\000\000\000\000\000\026\000\000\000\016\000\001\000\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\2551\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\255\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\2431\198\008\161\008\161\255\2551\1981\1981\1981\1981\1981\1981\1981\1981\1981\198\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161\008\161",
}

for name, raw in pairs(Image.rawImages) do
    Image[name] = image.new(raw)
end






------------------------------------------------------------------------------ Color

Color = {
    black       = {   0,    0,    0},
    gray        = {0x80, 0x80, 0x80},
    redFrac     = {0xF4, 0x4D, 0x3F},
    redArea     = {0xFB, 0xD1, 0xCE},
    blueFrac    = {0x4B, 0x53, 0xEC},
    blueArea    = {0xD1, 0xD3, 0xF9},
    redBlueArea = {0xCE, 0xB0, 0xD4},

    minorLine   = {0x92, 0x92, 0x92},
    majorLine   = {0x26, 0x26, 0x26},
}






------------------------------------------------------------------------------ Layout
Layout = {
    gridMinX    = 300,
    gridMinY    = 50,
    gridWidth   = 240,
    gridHeight  = 240,

    symbolicX    = 20,
    symbolicY    = 20,

    frac1Slider = {20, 80},
    frac2Slider = {20, 120},

    buttonShowFractionModel     = {20, 180},
    buttonShowCalculation       = {20, 250},
}






------------------------------------------------------------------------------ Model
Model = class()

function Model:init()
    self.f1 = {numerator = 2, denominator = 3}
    self.f2 = {numerator = 1, denominator = 4}

    self.maxv = 3

    self.showFractionModel = false
    self.showCalculation   = false
end


function Model:divisions(fraction)
    return fraction.denominator * self.maxv
end


theModel = Model()






------------------------------------------------------------------------------ InteractiveObjectManager
InteractiveObjectManager = class()

function InteractiveObjectManager:init()
    self.objectList = {}
end


-- Adds an object to the list of interactive objects
function InteractiveObjectManager:add(obj)
    table.insert(self.objectList, obj)
end


-- Activates an object that has an onMouseDown handler and contains wx,wy
function InteractiveObjectManager:onMouseDown(wx, wy)
    for _, o in ipairs(self.objectList) do
        local md = o.onMouseDown
        if md and o:contains(wx, wy) then
            md(o, wx, wy)
            break
        end
    end
end


theInteractiveObjects = InteractiveObjectManager()






------------------------------------------------------------------------------ SymbolicDisplay
SymbolicDisplay = class()

function SymbolicDisplay:init(model)
    self.model  = model
end


-- Displays a fraction at (x,y). The font size and style must already be set
-- before calling this paint routine
-- 
-- Returns the wx offset after the fraction and wy as the middle of the fraction height.
function SymbolicDisplay.paintFract(gc, frac, color, pen, x, y)
    gc:setColorRGB(unpack(color))
    local ch = gc:getStringHeight("1")

    local n  = tostring(frac.numerator)
    local d  = tostring(frac.denominator)

    local nwid = gc:getStringWidth(n)
    local dwid = gc:getStringWidth(d)
    local divwid  = math.max(nwid, dwid) + 10

    -- Paint numerator
    gc:drawString(n, x + (divwid - nwid)/2, y, "top")

    -- Paint denominator
    y = y + ch
    gc:drawString(d, x + (divwid - dwid)/2, y, "top")

    -- Paint dividing line
    gc:setPen(pen, "smooth")
    gc:setColorRGB(unpack(Color.black))
    gc:drawLine(x, y, x + divwid, y)

    return x + divwid + gc:getStringWidth("0"), y
end


function SymbolicDisplay:paint(gc)
    gc:setFont("sansserif", "b", 12)

    local wx, wy = SymbolicDisplay.paintFract(gc, self.model.f1, Color.redFrac, "medium", Layout.symbolicX, Layout.symbolicY)

    gc:setColorRGB(unpack(Color.black))
    wx = gc:drawString(string.uchar(0xB7), wx, wy, "middle") + 8  -- middle dot

    wx, wy = SymbolicDisplay.paintFract(gc, self.model.f2, Color.blueFrac, "medium", wx, Layout.symbolicY)

    gc:setColorRGB(unpack(Color.black))
    wx = gc:drawString("=", wx, wy, "middle") + 8

    SymbolicDisplay.paintFract(gc, {numerator="?", denominator="?"}, Color.black, "medium", wx, Layout.symbolicY)
end


theSymbolicDisplay = SymbolicDisplay(theModel)






------------------------------------------------------------------------------ Checkbox
Checkbox = class()

function Checkbox:init(model, selector, label, x, y)
    self.model      = model
    self.selector   = selector
    self.x          = x
    self.y          = y
    self.label      = label

    local image     = Image.checkboxUnselected
    self.w          = image:width()
    self.h          = image:height()

    theInteractiveObjects:add(self)
end


function Checkbox:isSelected()
    return self.model[self.selector]
end


function Checkbox:contains(wx, wy)
    return wx >= self.x and wx < self.x + self.w and
           wy >= self.y and wy < self.y + self.h
end


function Checkbox:showCustomInfo(gc)
end


function Checkbox:paint(gc)
    local image
    if self:isSelected() then
        image = Image.checkboxSelected
    else
        image = Image.checkboxUnselected
    end
    gc:drawImage(image, self.x, self.y)
    gc:setFont("sansserif", "r", 7)
    gc:setColorRGB(unpack(Color.black))
    gc:drawString(self.label, self.x + image:width() + 5, self.y + image:height()/2, "middle")
    self:showCustomInfo(gc)
end


function Checkbox:onMouseDown(wx, wy)
    self.model[self.selector] = not self.model[self.selector]
    platform.window:invalidate()
    document.markChanged()
end




btnSFM = Checkbox(theModel, "showFractionModel", "Show fraction model", unpack(Layout.buttonShowFractionModel))

function btnSFM:showCustomInfo(gc)
    if self:isSelected() then
        local twiceShaded = self.model.f1.numerator * self.model.f2.numerator
        local rectPerUnitSquare = self.model.f1.denominator * self.model.f2.denominator

        gc:setFont("sansserif", "r", 7)
        gc:setColorRGB(unpack(Color.black))
        local ch = gc:getStringHeight("1")
        local x  = self.x + self.w + 5
        local y  = self.y + self.h + 4
        gc:drawString("Total rectangles shaded twice = " .. twiceShaded, x, y, "top")
        gc:drawString("Rectangles per unit square = " .. rectPerUnitSquare, x, y + ch, "top")
    end
end




btnSC  = Checkbox(theModel, "showCalculation", "Show calculation", unpack(Layout.buttonShowCalculation))

function btnSC:showCustomInfo(gc)
    if self:isSelected() then
        local f1 = self.model.f1
        local f2 = self.model.f2

        local x = self.x + 4
        local y = self.y + self.h + 4
        gc:setFont("sansserif", "b", 9)
        local wx, wy = SymbolicDisplay.paintFract(gc, f1, Color.redFrac, "medium", x, y)

        gc:setColorRGB(unpack(Color.black))
        wx = gc:drawString(string.uchar(0xB7), wx, wy, "middle") + 4

        wx, wy = SymbolicDisplay.paintFract(gc, f2, Color.blueFrac, "medium", wx, y)

        gc:setColorRGB(unpack(Color.black))
        wx = gc:drawString("=", wx, wy, "middle") + 4

        local n = f1.numerator * f2.numerator
        local d = f1.denominator * f2.denominator
        wx = SymbolicDisplay.paintFract(gc, {numerator=n, denominator=d}, Color.black, "medium", wx, y)

        local gcd = math.gcd(n, d)
        if gcd ~= 1 then
            n = n / gcd
            d = d / gcd
            wx = gc:drawString("=", wx, wy, "middle") + 4
            SymbolicDisplay.paintFract(gc, {numerator=n, denominator=d}, Color.black, "medium", wx, y)
        end
    end
end






------------------------------------------------------------------------------ Knob
-- A Knob can be grabbed with the mouse to move a slider

Knob = class()

function Knob:init(parent, image, x, y)
    self.parent         = parent
    self.image          = image
    self.x              = x
    self.y              = y
    self.dx, self.dy    = self:origin()
    self.w              = image:width()
    self.h              = image:height()

    theInteractiveObjects:add(self)
end


function Knob:setxy(wx, wy)
    self.parent:setxy(wx, wy)
end


function Knob:paint(gc)
    gc:drawImage(self.image, self.x - self.dx, self.y - self.dy)
end


function Knob:contains(wx, wy)
    local x = self.x - self.dx
    local y = self.y - self.dy
    return wx >= x and wx <= x + self.w and
           wy >= y and wy <= y + self.h
end


function Knob:onMouseDown(wx, wy)
    on.mouseMove = util.mouseTracker(self, wx, wy)

    on.mouseUp = function(wx, wy)
        on.mouseMove        = nil
        on.mouseUp          = nil
        self:drop()
    end
end


function Knob:drop()
end






------------------------------------------------------------------------------ VerticalKnob
VerticalKnob = class(Knob)

showTopFractionGuide = false


function VerticalKnob:paint(gc)
    self.x = self.parent:wcoord()
    Knob.paint(self, gc)
end


function VerticalKnob:onMouseDown(wx, wy)
    Knob.onMouseDown(self, wx, wy)
    showTopFractionGuide = true
    platform.window:invalidate()
end


function VerticalKnob:drop()
    showTopFractionGuide = false
    platform.window:invalidate()
end






------------------------------------------------------------------------------ HorizontalKnob
HorizontalKnob = class(Knob)

showRightFractionGuide = false


function HorizontalKnob:paint(gc)
    self.y = self.parent:wcoord()
    Knob.paint(self, gc)
end


function HorizontalKnob:onMouseDown(wx, wy)
    Knob.onMouseDown(self, wx, wy)
    showRightFractionGuide = true
    platform.window:invalidate()
end


function HorizontalKnob:drop()
    showRightFractionGuide = false
    platform.window:invalidate()
end






------------------------------------------------------------------------------ SimpleKnob
SimpleKnob = class(Knob)


function SimpleKnob:origin()
    return self.image:width()/2, self.image:height()/2
end


function SimpleKnob:paint(gc)
    self.x = self.parent:wcoord()
    Knob.paint(self, gc)
end






------------------------------------------------------------------------------ TopKnob
TopKnob = class(VerticalKnob)

function TopKnob:init(parent, wx, wy)
    VerticalKnob.init(self, parent, Image.knobTop, wx, wy)
end


-- Returns the origin coordinates.
function TopKnob:origin()
    return self.image:width()/2 - 1, self.image:height()
end


function TopKnob:paint(gc)
    VerticalKnob.paint(self, gc)
    if showTopFractionGuide then
        local parent = self.parent
        gc:setFont("sansserif", "r", 7)
        SymbolicDisplay.paintFract(gc, parent.fraction, parent.color, "thin", self.x + self.w/2 + 6, self.y - self.h - 22)
    end
end






------------------------------------------------------------------------------ BottomKnob
BottomKnob = class(VerticalKnob)

function BottomKnob:init(parent, wx, wy)
    VerticalKnob.init(self, parent, Image.knobBottom, wx, wy)
end


-- Returns the origin coordinates.
function BottomKnob:origin()
    return self.image:width()/2 - 1, 0
end






------------------------------------------------------------------------------ LeftKnob
LeftKnob = class(HorizontalKnob)

function LeftKnob:init(parent, wx, wy)
    HorizontalKnob.init(self, parent, Image.knobLeft, wx, wy)
end


-- Returns the origin coordinates.
function LeftKnob:origin()
    return self.image:width(), self.image:height()/2
end






------------------------------------------------------------------------------ RightKnob
RightKnob = class(HorizontalKnob)

function RightKnob:init(parent, wx, wy)
    HorizontalKnob.init(self, parent, Image.knobRight, wx, wy)
end


-- Returns the origin coordinates.
function RightKnob:origin()
    return 0, self.image:height()/2
end


function RightKnob:paint(gc)
    HorizontalKnob.paint(self, gc) -- this sets the coordinates of the knob
    if showRightFractionGuide then
        local parent = self.parent
        gc:setFont("sansserif", "r", 7)
        SymbolicDisplay.paintFract(gc, parent.fraction, parent.color, "thin", self.x + self.w + 6, self.y - 18)
    end
end






------------------------------------------------------------------------------ Slider
Slider = class()

function Slider:init()
end


function Slider:setxy(wx, wy)
    self:setValueFromWcoord(wx, wy)
end


-- Returns the nearest slider value of the given window coordinate.
function Slider:wcoordToNearestValue(wmin, wmax, wx)
    local wrange = wmax - wmin
    local vrange = self:max() - self:min()
    local divs = self:divisions()
    if wx < wmin then
        wx = wmin
    elseif wx > wmax then
        wx = wmax
    end
    local portion = math.floor((wx - wmin) / wrange * divs + 0.5)
    return portion/divs * vrange + self:min()
end


-- Returns the window coordinate of the slider's value.
function Slider:wcoord(wmin, wmax)
    local min = self:min()
    local max = self:max()
    return (self:value() - min)/(max - min) * (wmax - wmin) + wmin
end






------------------------------------------------------------------------------ NumeratorSelector
-- A NumeratorSelector is a line with knobs on each end. The student can grab
-- a knob and slide it around to select a new value for the numerator of a
-- fraction.

NumeratorSelector = class(Slider)


function NumeratorSelector:init(model, fraction, wxmin, wymin, wxmax, wymax)
    Slider.init(self)
    self.model          = model
    self.fraction       = fraction
    self.wxmin          = wxmin
    self.wymin          = wymin
    self.wxmax          = wxmax
    self.wymax          = wymax
end


function NumeratorSelector:min()
    return 0
end


function NumeratorSelector:max()
    return self:divisions()
end


function NumeratorSelector:value()
    return self.fraction.numerator
end


function NumeratorSelector:setxy(wx, wy)
    self:setValueFromWcoord(wx, wy)
end


function NumeratorSelector:divisions()
    return self.model:divisions(self.fraction)
end


function NumeratorSelector:paint(gc, wx1, wy1, wx2, wy2)
    self.knob1:paint(gc)
    self.knob2:paint(gc)
    gc:setColorRGB(unpack(self.color))
    gc:setPen("medium", "smooth")
    gc:drawLine(wx1, wy1, wx2, wy2)
end






------------------------------------------------------------------------------ VerticalNumeratorSelector
VerticalNumeratorSelector = class(NumeratorSelector)

function VerticalNumeratorSelector:init(model, wxmin, wymin, wxmax, wymax)
    NumeratorSelector.init(self, model, model.f2, wxmin, wymin, wxmax, wymax)
    self.color = Color.blueFrac

    local wcoord = self:wcoord()
    self.knob1 = TopKnob(self, wcoord, wymin)
    self.knob2 = BottomKnob(self, wcoord, wymax)
end


function VerticalNumeratorSelector:setValueFromWcoord(wx, wy)
    self.fraction.numerator = self:wcoordToNearestValue(self.wxmin, self.wxmax, wx)
end


-- Returns the window coordinate for the value of the selector
function VerticalNumeratorSelector:wcoord()
    return Slider.wcoord(self, self.wxmin, self.wxmax)
end


function VerticalNumeratorSelector:areaWidth()
    return Slider.wcoord(self, self.wxmin, self.wxmax) - self.wxmin
end


function VerticalNumeratorSelector:areaHeight()
    return self.wymax - self.wymin
end


function VerticalNumeratorSelector:areaColor()
    return Color.blueArea
end


function VerticalNumeratorSelector:paint(gc)
    local wx = self:wcoord()
    NumeratorSelector.paint(self, gc, wx, self.wymin, wx, self.wymax)
end






------------------------------------------------------------------------------ HorizontalNumeratorSelector
HorizontalNumeratorSelector = class(NumeratorSelector)

function HorizontalNumeratorSelector:init(model, wxmin, wymin, wxmax, wymax)
    NumeratorSelector.init(self, model, model.f1, wxmin, wymin, wxmax, wymax)
    self.color = Color.redFrac

    local wcoord = self:wcoord()
    self.knob1 = LeftKnob(self, wxmin, wcoord)
    self.knob2 = RightKnob(self, wxmax, wcoord)
end


function HorizontalNumeratorSelector:setValueFromWcoord(wx, wy)
    self.fraction.numerator = self:max() - self:wcoordToNearestValue(self.wymin, self.wymax, wy)
end


function HorizontalNumeratorSelector:wcoord()
    return Slider.wcoord(self, self.wymax, self.wymin)
end


function HorizontalNumeratorSelector:areaWidth()
    return self.wxmax - self.wxmin
end


function HorizontalNumeratorSelector:areaHeight()
    return Slider.wcoord(self, self.wymin, self.wymax) - self.wymin
end


function HorizontalNumeratorSelector:areaColor()
    return Color.redArea
end


function HorizontalNumeratorSelector:paint(gc)
    local wy = self:wcoord()
    NumeratorSelector.paint(self, gc, self.wxmin, wy, self.wxmax, wy)
end





------------------------------------------------------------------------------ DenominatorSelector
DenominatorSelector = class(Slider)


function DenominatorSelector:init(label, model, fraction, image, color, knobImage, wx, wy)
    Slider.init(self)
    self.label          = label
    self.model          = model
    self.fraction       = fraction
    self.image          = image
    self.color          = color
    self.wx             = wx
    self.wy             = wy
    self.slidery        = wy + 30

    self.width          = image:width()
    self.wxmax          = wx + self.width
    self.knob           = SimpleKnob(self, knobImage, self:wcoord(), self.slidery)
end


function DenominatorSelector:min()
    return 1
end


function DenominatorSelector:max()
    return 8
end


function DenominatorSelector:divisions()
    return self:max() - self:min()
end


function DenominatorSelector:value()
    return self.fraction.denominator
end


function DenominatorSelector:wcoord()
    return Slider.wcoord(self, self.wx, self.wxmax)
end


function DenominatorSelector:setxy(wx, wy)
    self:setValueFromWcoord(wx, wy)
end


function DenominatorSelector:setValueFromWcoord(wx, wy)
    local fr = self.fraction
    fr.denominator = self:wcoordToNearestValue(self.wx, self.wxmax, wx)
    fr.numerator = math.min(fr.numerator, self.model:divisions(fr))
end


function DenominatorSelector:paint(gc)
    -- Draw label
    gc:setFont("sansserif", "r", 7)
    gc:setColorRGB(unpack(Color.black))
    gc:drawString(self.label, self.wx, self.wy, "top")

    -- Draw slider well
    gc:drawImage(self.image, self.wx, self.slidery - self.image:height()/2)

    -- Draw knob
    self.knob:paint(gc)

    -- Draw value of denominator
    local value = tostring(self:value())
    local vw    = gc:getStringWidth(value)
    local vh    = gc:getStringHeight(value)
    local x     = self.wxmax + 10
    local y     = self.slidery - vh/2
    local xw    = 40
    gc:setColorRGB(unpack(Color.gray))
    gc:setPen("thin", "smooth")
    gc:drawRect(x, y, xw, vh)
    gc:setColorRGB(unpack(self.color))
    gc:setPen("medium", "smooth")
    gc:drawLine(x + xw + 1, y+1, x + xw + 1, y + vh - 1)

    gc:setColorRGB(unpack(Color.black))
    gc:drawString(value, x + xw - vw - 2, self.slidery, "middle")
end


f1Denom = DenominatorSelector("Denominator for the first fraction", theModel, theModel.f1,
    Image.sliderWellRed, Color.redFrac, Image.beadRed, unpack(Layout.frac1Slider))

f2Denom = DenominatorSelector("Denominator for the second fraction", theModel, theModel.f2,
    Image.sliderWellBlue, Color.blueFrac, Image.beadBlue, unpack(Layout.frac2Slider))







------------------------------------------------------------------------------ Grid
Grid = class()

function Grid:init(model)
    self.model  = model
    self.wx     = Layout.gridMinX
    self.wy     = Layout.gridMinY
    self.ww     = Layout.gridWidth
    self.wh     = Layout.gridHeight

    self.vnsel  = VerticalNumeratorSelector(model, self.wx, self.wy, self.wx + self.ww, self.wy + self.wh)
    self.hnsel  = HorizontalNumeratorSelector(model, self.wx, self.wy, self.wx + self.ww, self.wy + self.wh)
end


function Grid:paint(gc)

    -- Label x axis
    gc:setFont("sansserif", "r", 9)
    local maxv   = self.model.maxv
    local wrange = self.ww
    local wy     = gc:getStringHeight("1") + self.wy + self.wh
    local wx     = self.wx - gc:getStringWidth("00")
    gc:setColorRGB(unpack(Color.black))
    gc:drawString("0", wx, wy, "baseline")
    for x = 1, maxv do
        local label = tostring(x)
        local wx = x/maxv * wrange + self.wx - gc:getStringWidth(label)/2
        gc:drawString(label, wx, wy, "baseline")
    end


    -- Label y axis
    wrange = self.wh
    for y = 1, maxv do
        local wy = self.wy + wrange - y/maxv * wrange
        gc:drawString(tostring(y), wx, wy, "middle")
    end


    -- Fill vertical area of fraction coverage
    local vw = self.vnsel:areaWidth()
    local vh = self.vnsel:areaHeight()
    local hw = self.hnsel:areaWidth()
    local hh = self.hnsel:areaHeight()

    gc:setColorRGB(unpack(self.vnsel:areaColor()))
    gc:fillRect(self.wx, self.wy, vw, vh - hh)


    -- Fill horizontal area of fraction coverage
    gc:setColorRGB(unpack(self.hnsel:areaColor()))
    gc:fillRect(self.wx + vw, self.wy + self.wh - hh, hw - vw, hh)


    -- Fill overlapping area of fraction coverage
    gc:setColorRGB(unpack(Color.redBlueArea))
    gc:fillRect(self.wx, self.wy + self.wh - hh, vw, hh)


    -- Draw vertical grid lines
    wy = self.wy + self.wh
    if self.model.showFractionModel then
        gc:setColorRGB(unpack(Color.minorLine))
        gc:setPen("thin", "dashed")
        local divs = self.vnsel:divisions()
        for x = 1, divs - 1 do
            local wx = x/divs * self.ww + self.wx
            gc:drawLine(wx, self.wy, wx, wy)
        end
    end
    gc:setColorRGB(unpack(Color.majorLine))
    gc:setPen("thin", "smooth")
    for x = 1, maxv - 1 do
        local wx = x/maxv * self.ww + self.wx
        gc:drawLine(wx, self.wy, wx, wy)
    end


    -- Draw horizontal grid lines
    wx = self.wx + self.ww
    if self.model.showFractionModel then
        gc:setColorRGB(unpack(Color.minorLine))
        gc:setPen("thin", "dashed")
        local divs = self.hnsel:divisions()
        for y = 1, divs - 1 do
            local wy = y/divs * self.wh + self.wy
            gc:drawLine(self.wx, wy, wx, wy)
        end
    end
    gc:setColorRGB(unpack(Color.majorLine))
    gc:setPen("thin", "smooth")
    for y = 1, maxv - 1 do
        local wy = y/maxv * self.wh + self.wy
        gc:drawLine(self.wx, wy, wx, wy)
    end


    -- Draw bounding box
    gc:setColorRGB(unpack(Color.majorLine))
    gc:setPen("thin", "smooth")
    gc:drawRect(self.wx, self.wy, self.ww, self.wh)


    -- Draw vertical numerator selector
    self.vnsel:paint(gc)


    -- Draw horizontal numerator selector
    self.hnsel:paint(gc)

end


theGrid = Grid(theModel)






------------------------------------------------------------------------------ Utilities

util = {}

-- Returns a function that tracks mouse movements. Obj is updated to
-- follow the mouse.
function util.mouseTracker(obj, wx, wy)
    return function(wx, wy)
        obj:setxy(wx, wy)
        platform.window:invalidate()
    end
end


function math.gcd(a, b)
    if a == 0 then
        return b
    end
    while b ~= 0 do
        if a > b then
            a = a - b
        else
            b = b - a
        end
    end
    return a
end






------------------------------------------------------------------------------ Event handlers
function on.paint(gc)
    theSymbolicDisplay:paint(gc)
    f1Denom:paint(gc)
    f2Denom:paint(gc)
    btnSFM:paint(gc)
    btnSC:paint(gc)
    theGrid:paint(gc)
end


function on.mouseDown(wx, wy)
    theInteractiveObjects:onMouseDown(wx, wy)
end


function on.save()
    return {
        f1 = theModel.f1,
        f2 = theModel.f2,

        showFractionModel = theModel.showFractionModel,
        showCalculation   = theModel.showCalculation,
    }
end


function on.restore(state)
    theModel.f1.numerator = state.f1.numerator
    theModel.f1.denominator = state.f1.denominator
    theModel.f2.numerator = state.f2.numerator
    theModel.f2.denominator = state.f2.denominator
end
