platform.apilevel = '1.0'
-- All rights reserved
--
-- Periodic Table of the Elements
-- John Powers  2010-05-23

LayoutTheme = 1 -- 1 = white background, 2 = black background

---------------------------------------------------------------------------------------------------- Model


-- Currently highlighted element

currentElement = 1 -- Hydrogen

PLUS_MINUS_SIGN = string.uchar(0xb1)
SUPER_0 = string.uchar(0x2070)
SUPER_1 = string.uchar(0xb9)
SUPER_2 = string.uchar(0xb2)
SUPER_3 = string.uchar(0xb3)
SUPER_4 = string.uchar(0x2074)
SUPER_5 = string.uchar(0x2075)
SUPER_6 = string.uchar(0x2076)
SUPER_7 = string.uchar(0x2077)
SUPER_8 = string.uchar(0x2078)
SUPER_9 = string.uchar(0x2079)
SUB_0   = string.uchar(0x2080)
SUB_1   = string.uchar(0x2081)
SUB_2   = string.uchar(0x2082)
SUB_3   = string.uchar(0x2083)
SUB_4   = string.uchar(0x2084)
SUB_5   = string.uchar(0x2085)
SUB_6   = string.uchar(0x2086)
SUB_7   = string.uchar(0x2087)
SUB_8   = string.uchar(0x2088)
SUB_9   = string.uchar(0x2089)
COPYRIGHT = string.uchar(0xa9)
A_RING  = string.uchar(0xc5)
DEG_C   = string.uchar(0x2103)
beta    = string.uchar(0x3b2)

Superscripts = {
  ["0"] = SUPER_0,
  ["1"] = SUPER_1,
  ["2"] = SUPER_2,
  ["3"] = SUPER_3,
  ["4"] = SUPER_4,
  ["5"] = SUPER_5,
  ["6"] = SUPER_6,
  ["7"] = SUPER_7,
  ["8"] = SUPER_8,
  ["9"] = SUPER_9,
}

Subscripts = {
  ["0"] = SUB_0,
  ["1"] = SUB_1,
  ["2"] = SUB_2,
  ["3"] = SUB_3,
  ["4"] = SUB_4,
  ["5"] = SUB_5,
  ["6"] = SUB_6,
  ["7"] = SUB_7,
  ["8"] = SUB_8,
  ["9"] = SUB_9,
}


-- Returns string with digits enclosed with braces {99} converted to superscripts.
function ss(str)
    return str:
              gsub("{(%d+)}", function(digs) return digs:gsub(".", Superscripts) end):
              gsub("<(%d+)>", function(digs) return digs:gsub(".", Subscripts) end)
end


-- Information about each element indexed by atomic number
Elements = {
-- 1
    { name="Hydrogen", symbol="H", weight="1.00794(7)", protons=1, config="1s",
      radius=0.32, density="0.000082",
      meltingpoint=-259.1, boilingpoint=-252.762, state="Gas", oxstates="+1,-1",
      discovered="1766", group=1, row=0, col=0 },

-- 2
    { name="Helium", symbol="He", weight="4.002602(2)", protons=2, config=ss"1s{2}",
      radius=0.37, density=0.000164,
      boilingpoint=-268.93, state="Gas", oxstates="0",
      discovered="1868", group=2, row=0, col=17 },

-- 3
    { name="Lithium", symbol="Li", weight="6.941(2)", protons=3, config=ss"1s{2} 2s",
      radius="1.30", density=0.534,
      meltingpoint="180.50", boilingpoint=1342, state="Solid", oxstates="+1",
      discovered="1817", group=3, row=1, col=0 },

-- 4
    { name="Beryllium", symbol="Be", weight="9.012182(3)", protons=4, config=ss"1s{2} 2s{2}",
      radius=0.99, density=1.85,
      meltingpoint=1287, boilingpoint=2471, state="Solid", oxstates="+2",
      discovered=1828, group=4, row=1, col=1 },

-- 5
    { name="Boron", symbol="B", weight="10.811(7)", protons=5, config=ss"1s{2} 2s{2} 2p",
      radius=0.84, density=2.34,
      meltingpoint=2075, boilingpoint=4000, state="Solid", oxstates="+3",
      discovered="1808", group=5, row=1, col=12 },

-- 6
    { name="Carbon", symbol="C", weight="12.0107(8)", protons=6, config=ss"1s{2} 2s{2} 2p{2}",
      radius=0.75, density=2.2,
      boilingpoint="sp 3825", state="Solid", oxstates="+2,+4,-4", notes="The graphite form of Carbon is used for the values shown. The 'sp' notation indicates a sublimation point.",
      discovered="Ancient", group=1, row=1, col=13 },

-- 7
    { name="Nitrogen", symbol="N", weight="14.0067(2)", protons=7, config=ss"1s{2} 2s{2} 2p{3}",
      radius=0.71, density=0.001145,
      meltingpoint="-210.0", boilingpoint=-195.798, state="Gas", oxstates=PLUS_MINUS_SIGN .. "1," .. PLUS_MINUS_SIGN .. "2," .. PLUS_MINUS_SIGN .. "3,+4,+5",
      discovered="1772", group=1, row=1, col=14 },

-- 8
    { name="Oxygen", symbol="O", weight="15.9994(3)", protons=8, config=ss"1s{2} 2s{2} 2p{4}",
      radius=0.64, density=0.001308,
      meltingpoint=-218.79, boilingpoint=-182.953, state="Gas", oxstates="-2",
      discovered="1774", group=1, row=1, col=15 },

-- 9
    { name="Fluorine", symbol="F", weight="18.9984032(5)", protons=9, config=ss"1s{2} 2s{2} 2p{5}",
      radius="0.60", density=0.001553,
      boilingpoint=-188.12, state="Gas", oxstates="-1",
      discovered="1886", group=6, row=1, col=16 },

-- 10
    { name="Neon", symbol="Ne", weight="20.1797(6)", protons=10, config=ss"1s{2} 2s{2} 2p{6}",
      radius=0.62, density=0.000825,
      boilingpoint=-246.053, state="Gas", oxstates="0",
      discovered="1898", group=2, row=1, col=17 },

-- 11
    { name="Sodium", symbol="Na", weight="22.98976928(2)", protons=11, config="[Ne] 3s",
      radius="1.60", density=0.97,
      meltingpoint=97.794, boilingpoint=882.94, state="Solid", oxstates="+1",
      discovered="1807", group=3, row=2, col=0  },

-- 12
    { name="Magnesium", symbol="Mg", weight="24.3050(6)", protons=12, config=ss"[Ne] 3s{2}",
      radius="1.40", density=1.74,
      meltingpoint=650, boilingpoint=1090, state="Solid", oxstates="+2",
      discovered="1755", group=4, row=2, col=1  },

-- 13
    { name="Aluminum", symbol="Al", weight="26.9815386(8)", protons ="13", config=ss"[Ne] 3s{2} 3p",
      radius=1.24, density=2.7,
      meltingpoint=660.32, boilingpoint=2519, state="Solid", oxstates="+3",
      discovered="1827", group=8, row=2, col=12 },

-- 14
    { name="Silicon", symbol="Si", weight="28.0855(3)", protons=14, config=ss"[Ne] 3s{2} 3p{2}",
      radius=1.14, density=2.3296,
      meltingpoint=1414, boilingpoint=3265, state="Solid", oxstates="+2,+4,-4",
      discovered="1824",  group=5, row=2, col=13 },

-- 15
    { name="Phosphorus", symbol="P", weight="30.973762(2)", protons=15, config=ss"[Ne] 3s{2} 3p{3}",
      radius=1.09,
      state="Solid", oxstates="+3,+5,-3",
      discovered="1669", group=1, row=2, col=14 },

-- 16
    { name="Sulfur", symbol="S", weight="32.065(5)", protons=16, config=ss"[Ne] 3s{2} 3p{4}",
      radius="1.00", density=2.07, notes="The rhombic form of Sulfur is used for the values shown.",
      meltingpoint=95.3, boilingpoint=444.61, state="Solid", oxstates="+4,+6,-2",
      discovered="Ancient", group=1, row=2, col=15 },

-- 17
    { name="Chlorine", symbol="Cl", weight="35.453(2)", protons=17, config=ss"[Ne] 3s{2} 3p{5}",
      radius="1.00", density=0.002989,
      meltingpoint=-101.5, boilingpoint=-34.04, state="Gas", oxstates="+1,+5,+7,-1",
      discovered="1774", group=6, row=2, col=16 },

-- 18
    { name="Argon", symbol="Ar", weight="39.948(1)", protons=18, config=ss"[Ne] 3s{2} 3p{6}",
      radius=1.01, density=0.001633,
      boilingpoint=-185.847, state="Gas", oxstates="0",
      discovered="1894", group=2, row=2, col=17 },

-- 19
    { name="Potassium", symbol="K", weight="39.0983(1)", protons=19, config="[Ar] 4s",
      radius="2.00", density=0.89,
      meltingpoint=63.5, boilingpoint=759, state="Solid", oxstates="+1",
      discovered="1807", group=3, row=3, col=0},

-- 20
    { name="Calcium", symbol="Ca", weight="40.078(4)", protons=20, config=ss"[Ar] 4s{2}",
      radius=1.74, density=1.54,
      meltingpoint=842, boilingpoint=1484, state="Solid", oxstates="+2",
      discovered="1808", group=4, row=3, col=1  },

-- 21
    { name="Scandium", symbol="Sc", weight="44.955912(6)", protons=21, config=ss"[Ar] 3d 4s{2}",
      radius=1.59, density=2.99,
      meltingpoint=1541, boilingpoint=2836, state="Solid", oxstates="+3",
      discovered="1878",  group=7, row=3, col=2  },

-- 22
    { name="Titanium", symbol="Ti", weight="47.867(1)", protons=22, config=ss"[Ar] 3d{2} 4s{2}",
      radius=1.48, density=4.506,
      meltingpoint=1668, boilingpoint=3287, state="Solid", oxstates="+2,+3,+4",
      discovered="1791", group=7, row=3, col=3  },

-- 23
    { name="Vanadium", symbol="V", weight="50.9415(1)", protons=23, config=ss"[Ar] 3d{3} 4s{2}",
      radius=1.44, density="6.0",
      meltingpoint=1910, boilingpoint=3407, state="Solid", oxstates="+2,+3,+4,+5",
      discovered="1801", group=7, row=3, col=4  },

-- 24
    { name="Chromium", symbol="Cr", weight="51.9961(6)", protons=24, config=ss"[Ar] 3d{5} 4s",
      radius="1.30", density=7.15,
      meltingpoint=1907, boilingpoint=2671, state="Solid", oxstates="+2,+3,+6",
      discovered="1797", group=7, row=3, col=5  },

-- 25
    { name="Manganese", symbol="Mn", weight="54.938045(5)", protons=25, config=ss"[Ar] 3d{5} 4s{2}",
      radius=1.29, density=7.3,
      meltingpoint=1246, boilingpoint=2061, state="Solid", oxstates="+2,+3,+4,+7",
      discovered="1774", group=7, row=3, col=6  },

-- 26
    { name="Iron", symbol="Fe", weight="55.845(2)", protons=26, config=ss"[Ar] 3d{6} 4s{2}",
      radius=1.24, density=7.87,
      meltingpoint=1538, boilingpoint=2861, state="Solid", oxstates="+2,+3",
      discovered="Ancient", group=7, row=3, col=7 },

-- 27
    { name="Cobalt", symbol="Co", weight="58.933195(5)", protons=27, config=ss"[Ar] 3d{7} 4s{2}",
      radius=1.18, density=8.86,
      meltingpoint=1475, boilingpoint=2927, state="Solid", oxstates="+2,+3",
      discovered="1735", group=7, row=3, col=8  },

-- 28
    { name="Nickel", symbol="Ni", weight="58.6934(4)", protons=28, config=ss"[Ar] 3d{8} 4s{2}",
      radius=1.17, density="8.90",
      meltingpoint=1455, boilingpoint=2913, state="Solid", oxstates="+2,+3",
      discovered="1751", group=7, row=3, col=9  },

-- 29
    { name="Copper", symbol="Cu", weight="63.546(3)", protons=29, config=ss"[Ar] 3d{10} 4s",
      radius=1.22, density=8.96,
      meltingpoint=1084.62, boilingpoint=2562, state="Solid", oxstates="+1,+2",
      discovered="Ancient", group=7, row=3, col=10 },

-- 30
    { name="Zinc", symbol="Zn", weight="65.38(2)", protons=30, config=ss"[Ar] 3d{10} 4s{2}",
      radius="1.20", density=7.134,
      meltingpoint=419.53, boilingpoint=907, state="Solid", oxstates="+2",
      discovered=1746, group=7, row=3, col=11 },

-- 31
    { name="Gallium", symbol="Ga", weight="69.723(1)", protons=31, config=ss"[Ar] 3d{10} 4s{2} 4p",
      radius=1.23, density=5.91,
      boilingpoint=2204, state="Solid", oxstates="+3",
      discovered="1875", group=8, row=3, col=12 },

-- 32
    { name="Germanium", symbol="Ge", weight="72.64(1)", protons=32, config=ss"[Ar] 3d{10} 4s{2} 4p{2}",
      radius="1.20", density=5.3234,
      meltingpoint=938.25, boilingpoint=2833, state="Solid", oxstates="+2,+4",
      discovered="1886", group=5, row=3, col=13 },

-- 33
    { name="Arsenic", symbol="As", weight="74.92160(2)", protons=33, config=ss"[Ar] 3d{10} 4s{2} 4p{3}",
      radius="1.20", density=5.75, notes="The gray form of Arsenic is used for the values shown. The 'sp' notation indicates a sublimation point.",
      boilingpoint="sp 616", state="Solid", oxstates="+3,+5,-3",
      discovered="Ancient", group=5, row=3, col=14 },

-- 34
    { name="Selenium", symbol="Se", weight="78.96(3)", protons=34, config=ss"[Ar] 3d{10} 4s{2} 4p{4}",
      radius=1.18, density=4.809, notes="The gray form of Selenium is used for the values shown.",
      meltingpoint=220.8, boilingpoint=685, state="Solid", oxstates="+4,+6,-2",
      discovered="1817", group=1, row=3, col=15 },

-- 35
    { name="Bromine", symbol="Br", weight="79.904(1)", protons=35, config=ss"[Ar] 3d{10} 4s{2} 4p{5}",
      radius=1.17, density=3.1028,
      meltingpoint=-7.2, boilingpoint=58.8, state="Liquid", oxstates="+1,+5,-1",
      discovered="1826", group=6, row=3, col=16 },

-- 36
    { name="Krypton", symbol="Kr", weight="83.798(2)", protons=36, config=ss"[Ar] 3d{10} 4s{2} 4p{6}",
      radius=1.16, density=0.003425,
      boilingpoint=-153.34, state="Gas", oxstates="0",
      discovered="1898", group=2, row=3, col=17 },

-- 37
    { name="Rubidium", symbol="Rb", weight="85.4678(3)", protons=37, config="[Kr] 5s",
      radius=2.15, density=1.53,
      meltingpoint=39.3, boilingpoint=688, state="Solid", oxstates="+1",
      discovered="1861", group=3, row=4, col=0  },

-- 38
    { name="Strontium", symbol="Sr", weight="87.62(1)", protons=38, config=ss"[Kr] 5s{2}",
      radius="1.90", density=2.64,
      meltingpoint=777, boilingpoint=1382, state="Solid", oxstates="+2",
      discovered=1790, group=4, row=4, col=1  },

-- 39
    { name="Yttrium", symbol="Y", weight="88.90585(2)", protons=39, config=ss"[Kr] 4d 5s{2}",
      radius=1.76, density=4.47,
      meltingpoint=1522, boilingpoint=3345, state="Solid", oxstates="+3",
      discovered="1794", group=7, row=4, col=2  },

-- 40
    { name="Zirconium", symbol="Zr", weight="91.224(2)", protons=40, config=ss"[Kr] 4d{2} 5s{2}",
      radius=1.64, density=6.52,
      meltingpoint=1854.7, boilingpoint=4409, state="Solid", oxstates="+4",
      discovered="1789", group=7, row=4, col=3  },

-- 41
    { name="Niobium", symbol="Nb", weight="92.90638(2)", protons=41, config=ss"[Kr] 4d{4} 5s",
      radius=1.56, density=8.57,
      meltingpoint=2477, boilingpoint=4744, state="Solid", oxstates="+3,+5",
      discovered="1801", group=7, row=4, col=4  },

-- 42
    { name="Molybdenum", symbol="Mo", weight="95.96(2)", protons=42, config=ss"[Kr] 4d{5} 5s",
      radius=1.46, density=10.2,
      meltingpoint=2623, boilingpoint=4639, state="Solid", oxstates="+6",
      discovered="1778", group=7, row=4, col=5  },

-- 43
    { name="Technetium", symbol="Tc", weight="[97.9072]", protons=43, config=ss"[Kr] 4d{5} 5s{2}",
      radius=1.38, density=11,
      meltingpoint=2157, boilingpoint=4265, state="Synthetic", oxstates="+4,+6,+7",
      discovered="1937", group=7, row=4, col=6  },

-- 44
    { name="Ruthenium", symbol="Ru", weight="101.07(2)", protons=44, config=ss"[Kr] 4d{7} 5s",
      radius=1.36, density=12.1,
      meltingpoint=2333, boilingpoint=4150, state="Solid", oxstates="+3",
      discovered="1844", group=7, row=4, col=7  },

-- 45
    { name="Rhodium", symbol="Rh", weight="102.90550(2)", protons=45, config=ss"[Kr] 4d{8} 5s",
      radius=1.34, density=12.4,
      meltingpoint=1964, boilingpoint=3695, state="Solid", oxstates="+3",
      discovered="1803", group=7, row=4, col=8  },

-- 46
    { name="Palladium", symbol="Pd", weight="106.42(1)", protons=46, config=ss"[Kr] 4d{10}",
      radius="1.30", density="12.0",
      meltingpoint=1554.8, boilingpoint=2963, state="Solid", oxstates="+2,+3",
      discovered="1803", group=7, row=4, col=9  },

-- 47
    { name="Silver", symbol="Ag", weight="107.8682(2)", protons=47, config=ss"[Kr] 4d{10} 5s",
      radius=1.36, density=10.5,
      meltingpoint=961.78, boilingpoint=2162, state="Solid", oxstates="+1",
      discovered="Ancient", group=7, row=4, col=10 },

-- 48
    { name="Cadmium", symbol="Cd", weight="112.411(8)", protons=48, config=ss"[Kr] 4d{10} 5s{2}",
      radius="1.40", density=8.69,
      meltingpoint=321.069, boilingpoint=767, state="Solid", oxstates="+2",
      discovered="1817", group=7, row=4, col=11 },

-- 49
    { name="Indium", symbol="In", weight="114.818(3)", protons=49, config=ss"[Kr] 4d{10} 5s{2} 5p",
      radius=1.42, density=7.31,
      meltingpoint=156.6, boilingpoint=2072, state="Solid", oxstates="+3",
      discovered="1863", group=8, row=4, col=12 },

-- 50
    { name="Tin", symbol="Sn", weight="118.710(7)", protons=50, config=ss"[Kr] 4d{10} 5s{2} 5p{2}",
      radius="1.40", density=7.287, notes="The white form of Tin is used for the values shown.",
      meltingpoint=231.93, boilingpoint=2602, state="Solid", oxstates="+2,+4",
      discovered="Ancient", group=8, row=4, col=13 },

-- 51
    { name="Antimony", symbol="Sb", weight="121.760(1)", protons=51, config=ss"[Kr] 4d{10} 5s{2} 5p{3}",
      radius="1.40", density=6.68, notes="The gray form of Antimony is used for the values shown.",
      meltingpoint=630.628, boilingpoint=1587, state="Solid", oxstates="+3,+5,-3",
      discovered="Ancient", group=5, row=4, col=14 },

-- 52
    { name="Tellurium", symbol="Te", weight="127.60(3)", protons=52, config=ss"[Kr] 4d{10} 5s{2} 5p{4}",
      radius=1.37, density=6.232,
      meltingpoint=449.51, boilingpoint=988, state="Solid", oxstates="+4,+6,-2",
      discovered="1782", group=5, row=4, col=15 },

--53
    { name="Iodine", symbol="I", weight="126.90447(3)", protons=53, config=ss"[Kr] 4d{10} 5s{2} 5p{5}",
      radius=1.36, density=4.933,
      meltingpoint=113.7, boilingpoint=184.4, state="Solid", oxstates="+1,+5,+7,-1",
      discovered="1811", group=6, row=4, col=16 },

-- 54
    { name="Xenon", symbol="Xe", weight="131.293(6)", protons=54, config=ss"[Kr] 4d{10} 5s{2} 5p{6}",
      radius=1.36, density=0.005366,
      boilingpoint=-108.09, state="Gas", oxstates="0",
      discovered="1898", group=2, row=4, col=17 },

-- 55
    { name="Caesium", symbol="Cs", weight="132.9054519(2)", protons=55, config="[Xe] 6s",
      radius=2.38, density=1.873,
      meltingpoint=28.5, boilingpoint=671, state="Liquid", oxstates="+1",
      discovered="1860", group=3, row=5, col=0  },

-- 56
    { name="Barium", symbol="Ba", weight="137.327(7)", protons=56, config=ss"[Xe] 6s{2}",
      radius=2.06, density=3.62,
      meltingpoint=727, boilingpoint=1897, state="Solid", oxstates="+2",
      discovered="1808", group=4, row=5, col=1  },

-- 57
    { name="Lanthanum", symbol="La", weight="138.90547(7)", protons=57, config=ss"[Xe] 5d 6s{2}",
      radius=1.94, density=6.15,
      meltingpoint=920, boilingpoint=3464, state="Solid", oxstates="+3",
      discovered="1839", group=10, row=8, col=3, up=104  },

-- 58
    { name="Cerium", symbol="Ce", weight="140.116(1)", protons=58, config=ss"[Xe] 4f 5d 6s{2}",
      radius=1.84, density="6.770",
      meltingpoint=799, boilingpoint=3443, state="Solid", oxstates="+3,+4",
      discovered="1803", group=10, row=8, col=4, up=105  },

-- 59
    { name="Praseodymium", symbol="Pr", weight="140.90765(2)", protons=59, config=ss"[Xe] 4f{3} 6s{2}",
      radius="1.90", density=6.77,
      meltingpoint=931, boilingpoint=3520, state="Solid", oxstates="+3",
      discovered="1841", group=10, row=8, col=5, up=106  },

-- 60
    { name="Neodymium", symbol="Nd", weight="144.242(3)", protons=60, config=ss"[Xe] 4f{4} 6s{2}",
      radius=1.88, density=7.01,
      meltingpoint=1016, boilingpoint=3074, state="Solid", oxstates="+3",
      discovered="1885", group=10, row=8, col=6, up=107  },

-- 61
    { name="Promethium", symbol="Pm", weight="[144.9127]", protons=61, config=ss"[Xe] 4f{5} 6s{2}",
      radius=1.86, density=7.26,
      meltingpoint=1042, boilingpoint=3000, state="Synthetic", oxstates="+3",
      discovered="1914", group=10, row=8, col=7, up=108  },

-- 62
    { name="Samarium", symbol="Sm", weight="150.36(2)", protons=62, config=ss"[Xe] 4f{6} 6s{2}",
      radius=1.85, density=7.52,
      meltingpoint=1072, boilingpoint=1794, state="Solid", oxstates="+2,+3",
      discovered="1879", group=10, row=8, col=8, up=109  },

-- 63
    { name="Europium", symbol="Eu", weight="151.964(1)", protons=63, config=ss"[Xe] 4f{7} 6s{2}",
      radius=1.83, density=5.24,
      meltingpoint=822, boilingpoint=1529, state="Solid", oxstates="+3",
      discovered="1901", group=10, row=8, col=9, up=110  },

-- 64
    { name="Gadolinium", symbol="Gd", weight="157.25(3)", protons=64, config=ss"[Xe] 4f{7} 5d 6s{2}",
      radius=1.82, density="7.90",
      meltingpoint=1313, boilingpoint=3273, state="Solid", oxstates="+3",
      discovered="1880", group=10, row=8, col=10, up=111 },

-- 65
    { name="Terbium", symbol="Tb", weight="158.92535(2)", protons=65, config=ss"[Xe] 4f{9} 6s{2}",
      radius=1.81, density=8.23,
      meltingpoint=1359, boilingpoint=3230, state="Solid", oxstates="+3)",
      discovered="1843", group=10, row=8, col=11, up=112 },

-- 66
    { name="Dysprosium", symbol="Dy", weight="162.500(1)", protons=66, config=ss"[Xe] 4f{10} 6s{2}",
      radius="1.80", density=8.55,
      meltingpoint=1412, boilingpoint=2567, state="Solid", oxstates="+3",
      discovered="1886", group=10, row=8, col=12, up=113  },

-- 67
    { name="Holmium", symbol="Ho", weight="164.93032(2)", protons=67, config=ss"[Xe] 4f{11} 6s{2}",
      radius=1.79, density="8.80",
      meltingpoint=1472, boilingpoint=2700, state="Solid", oxstates="+3",
      discovered="1878", group=10, row=8, col=13, up=114  },

-- 68
    { name="Erbium", symbol="Er", weight="167.259(3)", protons=68, config=ss"[Xe] 4f{12} 6s{2}",
      radius=1.77, density=9.07,
      meltingpoint=1529, boilingpoint=2868, state="Solid", oxstates="+3",
      discovered="1842", group=10, row=8, col=14, up=115  },

-- 69
    { name="Thulium", symbol="Tm", weight="168.93421(2)", protons=69, config=ss"[Xe] 4f{13} 6s{2}",
      radius=1.77, density=9.32,
      meltingpoint=1545, boilingpoint=1950, state="Solid", oxstates="+3",
      discovered="1879", group=10, row=8, col=15, up=116  },

-- 70
    { name="Ytterbium", symbol="Yb", weight="173.054(5)", protons=70, config=ss"[Xe] 4f{14} 6s{2}",
      radius=1.78, density="6.90",
      meltingpoint=824, boilingpoint=1196, state="Solid", oxstates="+2,+3",
      discovered="1878", group=10, row=8, col=16, up=117  },

-- 71
    { name="Lutetium", symbol="Lu", weight="174.9668(1)", protons=71, config=ss"[Xe] 4f{14} 5d 6s{2}",
      radius=1.74, density=9.84,
      meltingpoint=1663, boilingpoint=3402, state="Solid", oxstates="+3",
      discovered="1907", group=10, row=8, col=17, up=118  },

-- 72
    { name="Hafnium", symbol="Hf", weight="178.49(2)", protons=72, config=ss"[Xe] 4f{14} 5d{2} 6s{2}",
      radius=1.64, density=13.3,
      meltingpoint=2233, boilingpoint=4603, state="Solid", oxstates="+4",
      discovered="1923", group=7, row=5, col=3  },

-- 73
    { name="Tantalum", symbol="Ta", weight="180.94788(2)", protons=73, config=ss"[Xe] 4f{14} 5d{3} 6s{2}",
      radius=1.58, density=16.4,
      meltingpoint=3017, boilingpoint=5458, state="Solid", oxstates="+5",
      discovered="1802", group=7, row=5, col=4  },

-- 74
    { name="Tungsten", symbol="W", weight="183.84(1)", protons=74, config=ss"[Xe] 4f{14} 5d{4} 6s{2}",
      radius="1.50", density=19.3,
      meltingpoint=3422, boilingpoint=5555, state="Solid", oxstates="+6",
      discovered="1783", group=7, row=5, col=5  },

-- 75
    { name="Rhenium", symbol="Re", weight="186.207(1)", protons=75, config=ss"[Xe] 4f{14} 5d{5} 6s{2}",
      radius=1.41, density=20.8,
      meltingpoint=3185, boilingpoint=5596, state="Solid", oxstates="+4,+6,+7",
      discovered="1925", group=7, row=5, col=6  },

-- 76
    { name="Osmium", symbol="Os", weight="190.23(3)", protons=76, config=ss"[Xe] 4f{14} 5d{6} 6s{2}",
      radius=1.36, density=22.587,
      meltingpoint=3033, boilingpoint=5012, state="Solid", oxstates="+3,+4",
      discovered="1803", group=7, row=5, col=7  },

-- 77
    { name="Iridium", symbol="Ir", weight="192.217(3)", protons=77, config=ss"[Xe] 4f{14} 5d{7} 6s{2}",
      radius=1.32, density=22.562,
      meltingpoint=2446, boilingpoint=4428, state="Solid", oxstates="+3,+4",
      discovered="1803", group=7, row=5, col=8  },

-- 78
    { name="Platinum", symbol="Pt", weight="195.084(9)", protons=78, config=ss"[Xe] 4f{14} 5d{9} 6s",
      radius="1.30", density=21.5,
      meltingpoint=1768.2, boilingpoint=3825, state="Solid", oxstates="+2,+4",
      discovered="1735", group=7, row=5, col=9  },

-- 79
    { name="Gold", symbol="Au", weight="196.966569(4)", protons=79, config=ss"[Xe] 4f{14} 5d{10} 6s",
      radius="1.30", density=19.3,
      meltingpoint=1064.18, boilingpoint=2856, state="Solid", oxstates="+1,+3",
      discovered="Ancient", group=7, row=5, col=10 },

-- 80
    { name="Mercury", symbol="Hg", weight="200.59(2)", protons=80, config=ss"[Xe] 4f{14} 5d{10} 6s{2}",
      radius=1.32, density=13.5336,
      meltingpoint="-38.8290", boilingpoint=356.62, state="Liquid", oxstates="+1,+2",
      discovered="Ancient", group=7, row=5, col=11 },

-- 81
    { name="Thallium", symbol="Tl", weight="204.3833(2)", protons=81, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p",
      radius=1.44, density=11.8,
      meltingpoint=304, boilingpoint=1473, state="Solid", oxstates="+1,+3",
      discovered="1861", group=8, row=5, col=12 },

-- 82
    { name="Lead", symbol="Pb", weight="207.2(1)", protons=82, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p{2}",
      radius=1.45, density=11.3,
      meltingpoint=327.462, boilingpoint=1749, state="Solid", oxstates="+2,+4",
      discovered="Ancient", group=8, row=5, col=13 },

-- 83
    { name="Bismuth", symbol="Bi", weight="208.98040(1)", protons=83, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p{3}",
      radius="1.50", density=9.79,
      meltingpoint=271.406, boilingpoint=1564, state="Solid", oxstates="+3,+5",
      discovered="Ancient", group=8, row=5, col=14 },

-- 84
    { name="Polonium", symbol="Po", weight="[208.9824]", protons=84, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p{4}",
      radius=1.42, density=9.2,
      meltingpoint=254, boilingpoint=962, state="Solid", oxstates="+2,+4",
      discovered="1898", group=5, row=5, col=15 },

-- 85
    { name="Astatine", symbol="At", weight="[209.9871]", protons=85, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p{5}",
      radius=1.48,
      meltingpoint=302, state="Solid", oxstates="---",
      discovered="1940", group=6, row=5, col=16 },

-- 86
    { name="Radon", symbol="Rn", weight="[222.0176]", protons=86, config=ss"[Xe] 4f{14} 5d{10} 6s{2} 6p{6}",
      radius=1.46,density=0.009074,
      meltingpoint=-71, boilingpoint=-61.4, state="Gas", oxstates="0",
      discovered="1900", group=2, row=5, col=17 },

-- 87
    { name="Francium", symbol="Fr", weight="[223.0197]", neurons=138, protons=87, config="[Rn] 7s",
      radius=2.42,
      meltingpoint=27, state="Solid", oxstates="+1",
      discovered="1939", group=3, row=6, col=0, down=57  },

-- 88
    { name="Radium", symbol="Ra", weight="[226.0254]", protons=88, config=ss"[Rn] 7s{2}",
      radius=2.11, density=5,
      meltingpoint=696, state="Solid", oxstates="+2",
      discovered="1898", group=4, row=6, col=1, down=57  },

-- 89
    { name="Actinium", symbol="Ac", weight="[227.0277]", protons=89, config=ss"[Rn] 6d 7s{2}",
      radius=2.01, density=10,
      meltingpoint=1050, boilingpoint=3198, state="Solid", oxstates="+3",
      discovered="1899", group=11,row=9, col=3  },

-- 90
    { name="Thorium", symbol="Th", weight="232.03806(2)", protons=90, config=ss"[Rn] 6d{2} 7s{2}",
      radius="1.90", density=11.7,
      meltingpoint=1750, boilingpoint=4788, state="Solid", oxstates="+4",
      discovered="1828", group=11,row=9, col=4  },

-- 91
    { name="Protactinium", symbol= "Pa", weight="231.03588(2)", protons=91, config=ss"[Rn] 5f{2} ({3}H<4>) 6d 7s{2}",
      radius=1.84, density=15.4,
      meltingpoint=1572, state="Solid", oxstates="+4,+5",
      discovered="1913", group=11,row=9, col=5  },

-- 92
    { name="Uranium", symbol="U", weight="238.02891(3)", protons=92, config=ss"[Rn] 5f{3} ({4}I{0}<9> <2>) 6d 7s{2}",
      radius=1.83, density=19.1,
      meltingpoint=1135, boilingpoint=4131, state="Solid", oxstates="+3,+4,+5,+6",
      discovered="1841", group=11, row=9, col=6  },

-- 93
    { name="Neptunium", symbol="Np", weight="[237.0482]", protons=93, config=ss"[Rn] 5f{4} ({5}I<4>) 6d 7s{2}",
      radius="1.80", density=20.2,
      meltingpoint=644, state="Synthetic", oxstates="+3,+4,+5,+6",
      discovered="1940", group=11,row=9, col=7  },

-- 94
    { name="Plutonium", symbol="Pu", weight="[244.0642]", protons=94, config=ss"[Rn] 5f{6} 7s{2}",
      radius="1.80", density=19.7,
      meltingpoint=640, boilingpoint=3228, state="Synthetic", oxstates="+3,+4,+5,+6",
      discovered="1940", group=11,row=9, col=8  },

-- 95
    { name="Americium", symbol="Am", weight="[243.0614]", protons=95, config=ss"[Rn] 5f{7} 7s{2}",
      radius=1.73, density=12,
      meltingpoint=1176, boilingpoint=2011, state="Synthetic", oxstates="+3,+4,+5,+6",
      discovered="1944", group=11,row=9, col=9  },

-- 96
    { name="Curium", symbol="Cm", weight="[247.0704]", protons=96, config=ss"[Rn] 5f{7} 6d 7s{2}",
      radius=1.68, density=13.51,
      meltingpoint=1345, boilingpoint="~3100", state="Synthetic", oxstates="+3",
      discovered="1944", group=11,row=9, col=10 },

-- 97
    { name="Berkelium", symbol="Bk", weight="[247.0703]", protons=97, config=ss"[Rn] 5f{8} 6d 7s{2}",
      radius=1.68, density=13.25, notes="The " .. beta .. " form of Berkelium is used for the values shown.",
      meltingpoint=986, state="Synthetic", oxstates="+3,+4",
      discovered="1949", group=11,row=9, col=11 },

-- 98
    { name="Californium", symbol="Cf", weight="[251.0796]", protons=98, config=ss"[Rn] 5f{10} 7s{2}",
      radius=1.68, density=15.1,
      state="Synthetic", oxstates="+3",
      discovered="1950", group=11,row=9, col=12 },

-- 99
    { name="Einsteinium", symbol="Es", weight="[252.0830]", protons=99, config=ss"[Rn] 5f{11} 7s{2}",
      radius=1.65,
      state="Synthetic", oxstates="+3",
      discovered="1952", group=11,row=9, col=13 },

-- 100
    { name="Fermium", symbol="Fm", weight="[257.0951]", protons=100, config=ss"[Rn] 5f{12} 7s{2}",
      radius=1.67,
      state="Synthetic", oxstates="+3",
      discovered="1952", group=11,row=9, col=14 },

-- 101
    { name="Mendelevium", symbol="Md", weight="[258.0984]", protons=101, config=ss"[Rn] 5f{13} 7s{2}",
      radius=1.73,
      state="Synthetic", oxstates="+2,+3",
      discovered="1955", group=11,row=9, col=15 },

-- 102
    { name="Nobelium", symbol="No", weight="[259.1010]", protons=102, config=ss"[Rn] 5f{14} 7s{2}",
      radius=1.76,
      state="Synthetic", oxstates="+2,+3",
      discovered="1958", group=11,row=9, col=16 },

-- 103
    { name="Lawrencium", symbol="Lr", weight="[262.1097]", protons=103, config=ss"[Rn] 5f{14} 7s{2} 7p ?",
      radius=1.61,
      state="Synthetic", oxstates="+3",
      discovered="1961", group=11,row=9, col=17 },

-- 104
    { name="Rutherfordium", symbol="Rf", weight="[261.1088]", protons=104, config=ss"[Rn] 5f{14} 6d{2} 7s{2} ?",
      radius=1.57,
      state="Synthetic", oxstates="+4",
      discovered="1964", group=7, row=6, col=3, down=57  },

-- 105
    { name="Dubnium", symbol="Db", weight="[262.1141]", protons=105,
      radius=1.49,
      state="Synthetic",
      discovered="1967", group=7, row=6, col=4, down=58  },

-- 106
    { name="Seaborgium", symbol="Sg", weight="[266.1219]", protons=106,
      radius=1.43,
      state="Synthetic",
      discovered="1974", group=7, row=6, col=5, down=59  },

-- 107
    { name="Bohrium", symbol="Bh", weight="[272]", protons=107,
      radius=1.41,
      state="Synthetic",
      discovered="1981", group=7, row=6, col=6, down=60  },

-- 108
    { name="Hassium", symbol="Hs", weight="[277]", protons=108,
      radius=1.34,
      state="Synthetic",
      discovered="1964", group=7, row=6, col=7, down=61  },

-- 109
    { name="Meitnerium", symbol="Mt", weight="[276]", protons=109,
      radius=1.29,
      state="Synthetic",
      discovered="1982", group=7, row=6, col=8, down=62  },

-- 110
    { name="Darmstadtium",symbol= "Ds", weight="[281]", protons=110,
      radius=1.28,
      state="Synthetic",
      discovered="1987", group=7, row=6, col=9, down=63  },

-- 111
    { name="Roentgenium",symbol="Rg", weight="[280]", protons=111,
      radius=1.21,
      state="Synthetic",
      discovered=1994, group=7, row=6, col=10, down=64 },

-- 112
    { name="Copernicium",symbol="Cn", weight="[285]", protons=112,
      radius=1.22,
      state="Synthetic",
      discovered="1996", group=7, row=6, col=11, down=65 },

-- 113
    { name="Ununtrium", symbol="Uut",                 protons=113,
      radius=1.36,
      state="Synthetic",
      discovered="2004", group=9, row=6, col=12, down=66 },

-- 114
    { name="Ununquadium", symbol="Uuq", weight="[289]", protons=114,
      radius=1.43,
      state="Synthetic",
      discovered="1998", group=9, row=6, col=13, down=67 },

-- 115
    { name="Ununpentium", symbol="Uup",                 protons=115,
      radius=1.62,
      state="Synthetic",
      discovered="2004", group=9, row=6, col=14, down=68 },

-- 116
    { name="Ununhexium", symbol="Uuh", weight="[292]", protons=116,
      radius=1.75,
      state="Synthetic",
               group=9, row=6, col=15, down=69 },

-- 117
    { name="Ununseptium", symbol="Uus",                protons=117,
      radius=1.65,
      state="Synthetic",
               group=9, row=6, col=16, down=70 },

-- 118
    { name="Ununoctium", symbol="Uuo",                 protons=118,
      radius=1.57,
      state="Synthetic",
               group=9, row=6, col=17, down=71 },
}


function asnumber(value)
    if type(value) == "string" then
        return tonumber((value:gsub("[^-+%d.]", "")))
    end
    return value
end


function storeElementDataInSymbolTable()
    -- Initialize empty lists of data
    local db = {}
    for _, attr in ipairs(
        {"name", "symbol", "weight", "protons",
         "config", "radius", "density", "meltingpoint",
         "boilingpoint", "state", "oxstates", "discovered", "notes"}) do
        db[attr] = {}
    end

    -- Pack all the information into lists
    for an, info in ipairs(Elements) do
        for attr in pairs(db) do
            local datum = info[attr] or ""
            if attr == "weight" or attr == "protons" or attr == "radius" or
               attr == "density" or attr == "meltingpoint" or attr == "boilingpoint" or
               attr == "discovered" then
                    datum = asnumber(datum) or datum
            end
            db[attr][an] = datum
        end
    end

    -- Store the lists in the symbol table
    for attr, values in pairs(db) do
        local varname = "element." .. attr
        math.eval("Unlock " .. varname)
        var.store(varname, values)
        math.eval("Lock " .. varname)
    end

end



---------------------------------------------------------------------------------------------------- Layout


-- Screen layout parameters
PTLayout = {
    columns       = 18, rows         = 10,
    cellwidth     = 17, cellheight   = 21,
    leftmargin    = 7, topmargin     =  2,
    atomicNumberX = 1, atomicNumberY = -2, atomicNumberFontSize = 7, atomicNumberFontStyle = "r",
    nameX         = 1, nameY         = 12, nameFontSize         = 7, nameFontStyle         = "r",
}


PTThemeColors = {
    -- 1  white background
    {
        backgroundColor       = {255, 255, 255},  -- white
        borderColor           = {255, 255, 255},
        highlightBorderColor  = {0, 0, 0},
        textColor             = {0, 0, 0},        -- black
        popupBackgroundColor  = {0xF3, 0xF7, 0xFA},
        popupBorderColor      = {0, 0, 0},
        popupTextColor        = {0, 0, 0},
    },

    -- 2  black background
    {
        backgroundColor       = {0, 0, 0},
        borderColor           = {0, 0, 0},
        highlightBorderColor  = {255, 255, 255},
        textColor             = {255, 255, 255},
        popupBackgroundColor  = {0xF3, 0xF7, 0xFA},
        popupBorderColor      = {0, 0, 0},
        popupTextColor        = {0, 0, 0},
    },
}


-- Returns a font size that allows the text to fit in the maxwidth of pixels.
-- Leaves the font set at the size and style that fit.
function PTLayout:fitFont(gc, style, size, text, maxwidth)
    for fontsize = size, 6, -1 do
        -- Try a font on for size
        gc:setFont("sansserif", style, size)
        if gc:getStringWidth(text) < maxwidth then
            return fontsize
        end
    end

    return 6
end


function PTLayout:setWindow(gc)
    local w = platform.window:width()
    local h = platform.window:height()

    if platform.isDeviceModeRendering() then
        -- Small screen
        self.cellwidth = 17
        self.cellheight = 21

        self.leftmargin = 7
        self.topmargin = 1

        self.atomicNumberX = 1
        self.atomicNumberY = -2
        self.atomicNumberFontSize = 7
        self.atomicNumberFontStyle = "r"

        self.nameX = 1
        self.nameY = 12
        self.nameFontSize = 7
        self.nameFontStyle = "r"

        self.infoNameFontSize = 9
        self.infoNameFontStyle = "b"
        self.infoFontSize = 7
        self.infoFontStyle = "r"

        self.largescreen = false
    else
        -- Large screen
        self.cellwidth = math.floor(w / (self.columns + 1))
        self.cellheight = math.floor(h / (self.rows + 1))

        self.leftmargin = (w - self.cellwidth * self.columns) / 2
        self.topmargin = (h - self.cellheight * self.rows) / 2

        self.nameX = 2
        self.nameY = 15
        self.nameFontStyle = "b"
        self.nameFontSize = self:fitFont(gc, self.nameFontStyle, 12, "Uup", self.cellwidth) -- 12         jpp 2010-11-18

        self.atomicNumberX = 2
        self.atomicNumberY = 0
        self.atomicNumberFontStyle = "r"
        self.atomicNumberFontSize = self:fitFont(gc, self.atomicNumberFontStyle, self.nameFontSize-2, "118", self.cellwidth) -- 9   jpp 2010-11-18

        self.infoFontStyle = "r"
        self.infoFontSize = self:fitFont(gc, self.infoFontStyle, 11, InfoFieldName.config .. ": " .. Elements[92].config, 9*self.cellwidth) -- 11      jpp 2010-11-18
        self.infoNameFontStyle = "b"
        self.infoNameFontSize = self.infoFontSize + 2   -- 13      jpp 2010-11-18

        self.largescreen = true

        self.legendY = self.cellheight * 7.25 + self.topmargin
    end

    self.infoX = self.cellwidth * 2.5 + self.leftmargin
    self.infoY = self.topmargin
    self.infoBottomY = self.topmargin + 3 * self.cellheight
end




-- Background colors for groups of elements
GroupBackground = {
    { r =   0, g = 238, b =   0 },  -- 1: Other nonmetals
    { r = 136, g = 170, b = 255 },  -- 2: Noble gases
    { r = 255, g = 170, b =   0 },  -- 3: Alkali metals
    { r = 243, g = 243, b =   0 },  -- 4: Alkaline earth metals
    { r =  85, g = 204, b = 136 },  -- 5: Metalloids
    { r =   0, g = 221, b = 187 },  -- 6: Halogens
    { r = 221, g = 153, b = 153 },  -- 7: Transition metals
    { r = 153, g = 187, b = 170 },  -- 8: Other metals
    { r = 240, g = 240, b = 240 },  -- 9: Unknown
    { r = 255, g = 170, b = 136 },  -- 10: Lanthanides
    { r = 221, g = 170, b = 204 }   -- 11: Actinides
}


-- Returns the RGB background color for a given group of elements
function groupBkgColor(group)
    local color = GroupBackground[group]
    return color.r, color.g, color.b
end






-------------------------------------------------------------------------- View management


local theViews = {}

function addView(view)
    table.insert(theViews, view)
    platform.window:invalidate()
end


function closeTopView()
    if #theViews > 1 then
        table.remove(theViews)
        platform.window:invalidate()
    end
end


function paintViews(gc)
    for _, view in ipairs(theViews) do
        view:paint(gc)
    end
end


function currentView()
    return theViews[#theViews]
end






-------------------------------------------------------------------------- View
View = class()

function View:paint(gc)
    -- Implemented in subclass
end


function View:arrowKey(key)
    -- Implemented in subclass
end


function View:mouseDown(x, y)
    -- Implemented in subclass
end


function View:help()
    -- Implemented in subclass
end


function View:escapeKey()
    -- Implemented in subclass
end


function View:charIn(ch)
    -- Implemented in subclass
end


function View:enterKey()
    -- Implemented in subclass
end






-------------------------------------------------------------------------- ElementView
ElementView = class(View)

function ElementView:paint(gc)
    PTLayout:setWindow(gc) -- Calculate table layout based on screen real estate

    -- Clear window
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].backgroundColor))
    gc:fillRect(0, 0, platform.window:width(), platform.window:height())

    -- Draw cells of periodic table
    for an, elem in ipairs(Elements) do
        paintCell(gc, an, elem)
    end

    paintHighlight(gc, currentElement)
end


function ElementView:arrowKey(key)
    local elem = Elements[currentElement]
    local nextElement = elem[key]
    if nextElement then
        -- Use overridden next element

    elseif key == "left" then
        if currentElement > 1 then
            nextElement = currentElement - 1
        else
            nextElement = currentElement
        end
        
    elseif key == "right" then
        if currentElement < #Elements then
            nextElement = currentElement + 1
        else
            nextElement = currentElement
        end
        
    else
        local row = elem.row
        local col = elem.col
        nextElement = currentElement
        
        if key == "up" then
            if row > 0 then
                nextElement = findElementNear(row - 1, col)
            end
        else
            if row < PTLayout.rows - 1 then
                nextElement = findElementNear(row + 1, col)
            end
        end
    end

    if currentElement ~= nextElement then
        currentElement = nextElement
        platform.window:invalidate() -- repaint the window
    end
end


function ElementView:mouseDown(x, y)
    if doubleClick then
        addView(DetailView())
    else
        local clickedElement = findElementAt(x, y)
        if clickedElement and currentElement ~= clickedElement then
            currentElement = clickedElement
            platform.window:invalidate() -- repaint the window
        end
    end
end


function ElementView:enterKey()
    addView(DetailView())
end


function ElementView:help()
    addView(AboutView())
    platform.window:invalidate()
end


function ElementView:charIn(ch)
    if ch == "s" then
        storeElementDataInSymbolTable()
    elseif ch == "w" then
        LayoutTheme = 1
        platform.window:invalidate()
    elseif ch == "b" then
        LayoutTheme = 2
        platform.window:invalidate()
    end
end


addView(ElementView())






-------------------------------------------------------------------------- ViewWithCloseButton

ViewWithCloseButton = class(View)


function ViewWithCloseButton:setup(width, height)
    local w = platform.window:width()
    local h = platform.window:height()

    self.x = (w - width)/2
    if self.x < 0 then
        self.x = 0
    end
    self.y = (h - height)/2
    if self.y < 0 then
        self.y = 0
    end
    self.w = width
    self.h = height
end


function ViewWithCloseButton:setFontInfo(gc)
    if platform.isDeviceModeRendering() then
        self.titleFontSize = 9
        self.infoFontSize = 7
    else
        self.titleFontSize = 12
        self.infoFontSize = 11
    end

    gc:setFont("sansserif", "r", self.titleFontSize)
    self.titleHeight = gc:getStringHeight("r")
    gc:setFont("sansserif", "r", self.infoFontSize)
    self.infoHeight = gc:getStringHeight("r")
end


function ViewWithCloseButton:paint(gc)
    -- Clear interior
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].popupBackgroundColor))
    gc:fillRect(self.x, self.y, self.w, self.h)

    -- Paint border
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].popupBorderColor))
    gc:setPen("medium", "smooth")
    gc:drawRect(self.x, self.y, self.w, self.h)

    -- Paint close box
    local closeButton
    if platform.isDeviceModeRendering() then
        closeButton = theHHCloseButton
    else
        closeButton = theCloseButton
    end
    gc:drawImage(closeButton, self.x + self.w - closeButton:width(),
                              self.y - closeButton:height()/2)
    
end


function ViewWithCloseButton:mouseDown(x, y)
    self:escapeKey()
end


function ViewWithCloseButton:escapeKey()
    closeTopView()
    platform.window:invalidate()
end





-------------------------------------------------------------------------- DetailView
DetailView = class(ViewWithCloseButton)


function DetailView:setup(gc)
    self:setFontInfo(gc)
    if platform.window:width() <= 320 then
        self.width = 200
    else
        self.width = 300
    end
    local height = self.titleHeight + (#InfoLayout)*self.infoHeight + 24
    ViewWithCloseButton.setup(self, self.width, height)
end


function DetailView:paint(gc)
    self:setup(gc)
    ViewWithCloseButton.paint(self, gc)
    local marginx = self.x + 10
    local texty   = self.y + 2
    local info  = Elements[currentElement]

    -- Display name of element
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].popupTextColor))
    gc:setFont("sansserif", "b", self.titleFontSize)
    gc:drawString(info.name, marginx, texty, "top")
    texty = texty + self.titleHeight

    -- Display other information about the element
    gc:setFont("sansserif", "r", self.infoFontSize)
    for i = 2, #InfoLayout do
        local attrib = InfoLayout[i]
        local attribname = InfoFieldName[attrib] or attrib
        if attrib == "config" then
            local x = gc:drawString(attribname .. ": ", marginx, texty, "top")
            drawConfig(gc, info[attrib] or "---", x, texty, "r", "i", self.infoFontSize)
            gc:setFont("sansserif", "r", self.infoFontSize)
        elseif attrib == "notes" then
            local note = info[attrib]
            if note ~= nil then
                local maxx = marginx + self.width - 10
                local x = marginx
                for _, word in ipairs(note:split()) do
                    if x + gc:getStringWidth(word) > maxx then
                        x = marginx
                        texty = texty + self.infoHeight
                    end
                    x = gc:drawString(word .. " ", x, texty, "top")
                end
            end
        else
            local value = info[attrib] or "---"
            local units = Units[attrib]
            local disp = attribname .. ": " .. value
            if units then
                disp = disp .. " " .. units
            end
            gc:drawString(disp, marginx, texty, "top")
        end
        texty = texty + self.infoHeight
    end
end






-------------------------------------------------------------------------- AboutView
AboutView = class(ViewWithCloseButton)


function AboutView:setup(gc)
    self:setFontInfo(gc)
    if platform.window:width() <= 320 then
        self.width = 250
        self.legendWidth = 80
    else
        self.width = 400
        self.legendWidth = 150
    end

    local height = self.titleHeight + 13*self.infoHeight + 10
    ViewWithCloseButton.setup(self, self.width, height)
end


function AboutView:paint(gc)
    self:setup(gc)
    local marginx = self.x + 10
    local marginy = self.y + 5
    local texty   = marginy

    ViewWithCloseButton.paint(self, gc)

    -- Title
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].popupTextColor))
    local title = "Periodic Table of the Elements"
    gc:setFont("sansserif", "b", self.titleFontSize)
    gc:drawString(title, marginx, texty, "top")
    texty = texty + self.titleHeight

    -- Version
    title     = "Version 1.0"
    gc:setFont("sansserif", "r", self.infoFontSize)
    gc:drawString(title, marginx, texty, "top")
    texty = texty + self.infoHeight

    -- Copyright
    title     = "Copyright " .. COPYRIGHT .. " 2010, Texas Instruments"
    gc:drawString(title, marginx, texty, "top")
    texty = texty + 1.5*self.infoHeight

    -- Help
    gc:drawString("Press 's' to export properties to the symbol table", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Press 'w' for a white background", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Press 'b' for a black background", marginx, texty, "top")
    texty = texty + 1.5*self.infoHeight

    gc:setPen("thin", "smooth")
    for i, name in ipairs(AboutView.Legend) do
        local row, col = math.modf((i-1)/2)
        local x = 2*col*self.legendWidth + marginx + 10*col
        local y = row * (self.infoHeight + 2) + texty

        -- Paint group color
        gc:setColorRGB(groupBkgColor(i))
        gc:fillRect(x-2, y-1, self.legendWidth+2, self.infoHeight+2)
        gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].popupTextColor))
        gc:drawRect(x-2, y-1, self.legendWidth+2, self.infoHeight+2)
        gc:drawString(name, x, y, "top")
    end
end


AboutView.Legend = {
    "Other Non-Metals",
    "Noble Gases",
    "Alkali Metals",
    "Alkali Earth Metals",
    "Metalloids",
    "Halogens",
    "Transition Metals",
    "Other Metals",
    "Unknown",
    "Lanthanides",
    "Actinides"
}







-------------------------------- Event Handlers ------------------------------------


function on.paint(gc)
    paintViews(gc)
end


function on.arrowKey(key)
    currentView():arrowKey(key)
end


doubleClick = false
function on.mouseDown(x, y)
    currentView():mouseDown(x, y)
    doubleClick = true
    timer.start(0.5)
end


function on.help()
    currentView():help()
end


function on.escapeKey()
    currentView():escapeKey()
end


function on.timer()
    timer.stop()
    doubleClick = false
end


function on.enterKey()
    currentView():enterKey()
end


function on.charIn(ch)
    currentView():charIn(ch)
end


function on.resize()
    closeTopView()
end






----------- Utility functions


-- Paint cell for an element given its atomic number
function paintCell(gc, an, location)
    local w     = PTLayout.cellwidth
    local h     = PTLayout.cellheight
    local left  = PTLayout.leftmargin + location.col * w
    local top   = PTLayout.topmargin + location.row * h
   
    -- Draw background
    gc:setColorRGB(groupBkgColor(location.group))
    gc:fillRect(left, top, w, h)

    -- Draw border
    gc:setPen("thin", "smooth")
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].borderColor))
    gc:drawRect(left, top, w, h)
    
    -- Draw atomic number
    gc:setColorRGB(0, 0, 0)
    gc:setFont("sansserif", PTLayout.atomicNumberFontStyle, PTLayout.atomicNumberFontSize)
    gc:drawString(tostring(an), left + PTLayout.atomicNumberX, top + PTLayout.atomicNumberY, "top")

    -- Draw symbol name of element
    gc:setFont("sansserif", PTLayout.nameFontStyle, PTLayout.nameFontSize)
    gc:drawString(Elements[an].symbol, left + PTLayout.nameX, top + h, "bottom")
end


InfoFieldName = {
    name         = "Name",
    weight       = "Atomic Weight",
    config       = "Electron Config",
    radius       = "Atomic Radius",
    state        = "State",
    density      = "Density",
    discovered   = "Discovered",
    meltingpoint = "Melting Point",
    boilingpoint = "Boiling Point",
    oxstates     = "Oxidation States",
    protons      = "Protons",
}

InfoLayout = {"name", "weight", "config", "radius", "state", "density",
              "meltingpoint", "boilingpoint", "oxstates", "discovered", "notes" }

Units = {
    weight = "u",
    radius = A_RING,
    density = "g/cm" .. SUPER_3,
    meltingpoint = DEG_C,
    boilingpoint = DEG_C,
}

-- Highlight information about selected element
function paintHighlight(gc, an)
    local info  = Elements[an]
    local w     = PTLayout.cellwidth
    local h     = PTLayout.cellheight
    local left  = PTLayout.leftmargin + info.col * w
    local top   = PTLayout.topmargin + info.row * h
    local y     = PTLayout.infoY

    -- Draw thick border around selected element
    gc:setPen("medium", "smooth")
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].highlightBorderColor))
    gc:drawRect(left, top, w, h)

    -- Display name of element
    gc:setColorRGB(unpack(PTThemeColors[LayoutTheme].textColor))
    gc:setFont("sansserif", PTLayout.infoNameFontStyle, PTLayout.infoNameFontSize)
    gc:drawString(info.name, PTLayout.infoX, y, "top")
    y = y + gc:getStringHeight(info.name)*1.1

    -- Display other information about the element
    local x = PTLayout.infoX
    gc:setFont("sansserif", PTLayout.infoFontStyle, PTLayout.infoFontSize)
    local charheight = gc:getStringHeight(info.name) - 4
    for i = 2, 5 do
        local attrib = InfoLayout[i]
        local attribname = InfoFieldName[attrib] or attrib
        if attrib == "config" then
            local x = gc:drawString(attribname .. ": ", x, y, "top")
            drawConfig(gc, info[attrib] or "---", x, y, "r", "i", PTLayout.infoFontSize)
            gc:setFont("sansserif", PTLayout.infoFontStyle, PTLayout.infoFontSize)
        else
            local value = info[attrib] or "---"
            local units = Units[attrib]
            local disp = attribname .. ": " .. value
            if units then
                disp = disp .. " " .. units
            end
            gc:drawString(disp, x, y, "top")
        end
        y = y + charheight
    end
end


-- Returns the atomic number of the element on _row_ nearest the _col_ number
function findElementNear(row, col)
    local nearAn
    local nearCol

    for an, loc in ipairs(Elements) do
        if loc.row == row then
            if loc.col == col then
                return an
            end
            if loc.col < col then
                nearCol = loc.col
                nearAn  = an
            elseif loc.col > col then
                if loc.col - col < col - nearCol then
                    return an
                end
                return nearAn
            end
        end
    end
end

-- Returns the atomic number of the element at widget coordinates (x, y)
function findElementAt(x, y)
    for an, loc in ipairs(Elements) do
        local x0 = loc.col * PTLayout.cellwidth + PTLayout.leftmargin
        local x1 = x0 + PTLayout.cellwidth
        local y0 = loc.row * PTLayout.cellheight + PTLayout.topmargin
        local y1 = y0 + PTLayout.cellheight

        if x0 <= x and x < x1 and y0 <= y and y < y1 then
            return an
        end
    end
    return nil
end


-- Draws the Electron Configuration info string of an element.
function drawConfig(gc, config, x, y, normal, italic, fontsize)

    -- Display the parent element and brackets with the normal font
    local scan
    local parent, parent_end = config:find("^%b[]")
    if parent then
        gc:setFont("sansserif", normal, fontsize)
        x = gc:drawString(config:sub(parent, parent_end), x, y, "top")
        scan = parent_end + 1
    else
        scan = 1
    end

    -- Display shell letters in italics
    while true do
        local shell = config:find("%l", scan)
        if shell then
            -- Display everything before shell letter
            if shell ~= scan then
                gc:setFont("sansserif", normal, fontsize)
                x = gc:drawString(config:sub(scan, shell-1), x, y, "top")
            end

            -- Display shell letter in italic font
            gc:setFont("sansserif", italic, fontsize)
            x = gc:drawString(config:sub(shell, shell), x, y, "top")
            scan = shell + 1
        else
            -- Display everything after last shell letter
            if scan <= #config then
                gc:setFont("sansserif", normal, fontsize)
                gc:drawString(config:sub(scan), x, y, "top")
            end
            break
        end
    end

end





------------------------------------------------------------------------- Images

closeButton = ".\000\000\000\018\000\000\000\000\000\000\000\092\000\000\000\016\000\001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\2558\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\024\2398\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\023\239\024\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\247\238\024\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\240\185\240\185\240\185j\217j\217j\217\240\185\240\185\240\185i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\247\238\023\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213j\217\240\185\255\255\255\255\255\255\240\185\138\217\240\185\255\255\255\255\255\255\240\185j\217i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\246\238\246\234i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213j\213j\217\138\217\138\217\240\185\255\255\255\255\255\255\240\185\255\255\255\255\255\255\240\185\138\217\138\217j\217j\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\213\234\180\230\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\196\204\196\204\196\204\196\204\229\204\229\208\229\208\005\209L\169\255\255\255\255\255\255\255\255\255\255L\169\005\209\229\208\229\208\229\204\196\204\196\204\196\204\196\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200r\226\147\230\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\196\204\196\204\196\204\228\204\229\204\229\208\005\209\005\209\006\209\006\209L\169\222\251\222\251\222\251L\169\006\209\006\209\005\209\005\209\229\208\229\204\228\204\196\204\196\204\196\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200r\2260\222\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\195\204\196\204\196\204\196\204\229\204\229\208\005\209\006\209\006\209&\213&\213L\169\189\247\189\247\189\247L\169&\213&\213\006\209\006\209\005\209\229\208\229\204\196\204\196\204\196\204\195\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\205\217\015\222\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\228\204\229\204\229\208\229\208\005\209\006\209&\213&\213'\213G\213L\169\156\243\156\243\156\243\156\243\156\243L\169G\213'\213&\213&\213\006\209\005\209\229\208\229\208\229\204\228\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\205\217\016\222\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\006\209\006\213&\213'\213'\213G\213G\217H\217L\169{\239{\239{\239L\169{\239{\239{\239L\169H\217G\217G\213'\213'\213&\213\006\213\006\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\205\2170\222'\213'\213'\213'\213'\213'\213'\213'\213G\213G\213G\213G\217H\217h\217h\217h\217L\169{\239{\239{\239L\169\137\221L\169{\239{\239{\239L\169h\217h\217h\217H\217G\217G\213G\213G\213'\213'\213'\213'\213'\213'\213'\213'\213'\213\237\217Q\222h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217\137\221\137\221\137\221\137\221\137\221L\169L\169L\169\169\221\169\221\169\221L\169L\169L\169\137\221\137\221\137\221\137\221\137\221h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217\015\218Q\222\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\169\221\169\221\169\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\169\221\169\221\169\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\2210\222r\226\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225r\226\147\226\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\147\226\247\234r\226r\226r\226r\230r\230r\230r\230r\230r\230\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\230r\230r\230\146\234\146\234\146\230r\230r\230\247\234"
theCloseButton = image.new(closeButton)

hhCloseButton = "\016\000\000\000\016\000\000\000\000\000\000\000 \000\000\000\016\000\001\0009\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\231\025\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\227\025\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\248\222\248\222\248\222\017\230m\233\248\222\248\222\248\222\248\222\248\222\248\222\142\233t\226\248\222\248\222\248\222\247\218\247\218\247\218M\229g\236m\233\247\218\247\218\247\218\247\218m\233g\236\142\229\247\218\247\218\247\218\214\218\214\218\214\218\214\218L\229g\236M\229\214\218\214\218M\229g\236L\229\214\218\214\218\214\218\214\218\182\214\182\214\182\214\182\214\182\214L\229g\236L\229L\229g\236L\229\182\214\182\214\182\214\182\214\182\214\181\214\181\214\181\214\181\214\181\214\181\214L\229g\236g\236L\229\181\214\181\214\181\214\181\214\181\214\181\214\148\210\148\210\148\210\148\210\148\210\148\210L\229g\236g\236L\229\148\210\148\210\148\210\148\210\148\210\148\210t\206t\206t\206t\206t\206,\225g\236,\225,\225g\236,\225t\206t\206t\206t\206t\206s\206s\206s\206s\206+\225g\236+\225s\206s\206+\225g\236+\225s\206s\206s\206s\206S\202S\202S\202L\221g\236+\225S\202S\202S\202S\202+\225g\236+\225S\202S\202S\202R\202R\202R\202\240\209L\221R\202R\202R\202R\202R\202R\202+\225\142\213R\202R\202R\2022\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\198\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194"
theHHCloseButton = image.new(hhCloseButton)
