-- Graphical representation of area under a curve using Riemann integration
-- Copyright (C) 2010 Texas Instruments
-- All rights reserved
--
-- John Powers  2010-11-18


Color = {
    black           = {0, 0, 0},
    lightred        = {0xFF, 0x55, 0x55},
    darkred         = {0xAA, 0x00, 0x00},
    lightgreen      = {0x55, 0xFF, 0x55},
    darkgreen       = {0x00, 0xAA, 0x00},
    purple          = {0x80, 0x00, 0x80},
    lightgray       = {0xAA, 0xAA, 0xAA},
    lightblue       = {0x55, 0x55, 0xFF},
    darkblue        = {0x00, 0x00, 0xAA},
    lightyellow     = {0xFF, 0xFF, 0x55},
    darkyellow      = {0xAA, 0xAA, 0x00},
    white           = {0xFF, 0xFF, 0xFF},
}


Theme = {
    axiscolor   = Color.black,
    offscreenaxiscolor = Color.lightgray,
    tickpixels  = 24,               -- pixels between tick marks
    ticklength  = 5,                -- pixel length of tick from axis
    labeldist   = 5,                -- pixel distance from tick mark to label

    gridcolor   = {0xCC, 0xFF, 0xFF}, -- light turquoise

    functioncolor = Color.purple,


    handleColor         = Color.black,
    handleSelectedColor = Color.black,
    handleValueFontSize = 7,
    handleValueFontStyle = "r",
    handleValueOffset    = 12,

    labelX              = 20,
    labelbuttonX        = 9,
    leftlabelY          = 20,
    rightlabelY         = 38,
    midlabelY           = 56,
    traplabelY          = 74,
}

theCursor = "pointer"


COPYRIGHT = string.uchar(0xa9)
MULTIPLY  = string.uchar(0xd7)
EE = string.uchar(0xf000)


-- Converts _range_ to a nice number for graph labels.
-- Returns the tick mark scale and major tick mark count.
function nicenum(range)
    if range == 0 then
        return 0
    end

    local sign = 1
    if range < 0 then
        sign = -1
        range = -range
    end

    local expv = math.floor(math.log10(range))
    local scale = 10^expv
    local f = range / scale
    local nf
    local major = 5
    if f <= 1 then
        nf = 1
    elseif f <= 2 then
        nf = 2
    elseif f <= 2.5 then
        nf = 2.5
        major = 4
    elseif f <= 5 then
        nf = 5
        major = 4
    else
        nf = 10
    end
    return sign * nf * scale, major
end





---------------------------------------------------------------------- GraphPaper
-- This class fills the window with graph paper

GraphPaper = class()


function GraphPaper:init(graph)
    self.graph = graph
end


function GraphPaper:paint(gc)
    local w  = platform.window:width()
    local h  = platform.window:height()
    local ox = self.graph.yaxis.origin
    local oy = self.graph.xaxis.origin
    local tickpixels = Theme.tickpixels

    gc:setColorRGB(unpack(Theme.gridcolor))
    gc:setPen("thin", "smooth")

    -- Draw horizontal grid lines
    local py = h - oy
    while true do
        py = py - tickpixels
        if py < 0 then break end
        gc:drawLine(0, py, w-1, py)
    end

    py = h - oy
    while true do
        py = py + tickpixels
        if py > h then break end
        gc:drawLine(0, py, w-1, py)
    end

    -- Draw vertical grid lines
    local px = ox
    while true do
        px = px - tickpixels
        if px < 0 then break end
        gc:drawLine(px, 0, px, h-1)
    end

    px = ox
    while true do
        px = px + tickpixels
        if px > w then break end
        gc:drawLine(px, 0, px, h-1)
    end
end






---------------------------------------------------------------------- Axis

Axis = class()

function Axis:init(graph)
    self.graph = graph
end


function Axis:setOrigin(o)
    self.origin = math.floor(o + 0.5)
    document.markChanged()
end


function Axis:checkAxisLimits(value, limit)
    local linestyle = "smooth"
    local fontsize = 9
    local axiscolor = Theme.axiscolor

    if value < 0  or value >= limit then
        linestyle = "dotted"
        fontsize = 7
        axiscolor = Theme.offscreenaxiscolor
        if value < 0 then
            value = 30
        elseif value >= limit then
            value = limit - 30
        end
    end

    return value, linestyle, fontsize, axiscolor
end







---------------------------------------------------------------------- HorizontalAxis

HorizontalAxis = class(Axis)


function HorizontalAxis:paint(gc)
    local w = platform.window:width()
    local h = platform.window:height()

    local py, linestyle, fontsize, axiscolor = self:checkAxisLimits(h - self.origin, h)

    gc:setColorRGB(unpack(axiscolor))
    gc:setPen("thin", linestyle)
    gc:drawLine(0, py, w, py)

    local ticklength = Theme.ticklength
    local labeldist = Theme.labeldist

    local ptick = Theme.tickpixels
    local majortick = self.graph.majortick
    local pmajortick = majortick * ptick

    local xinc = self.graph.inc * majortick

    gc:setFont("sansserif", "r", fontsize)

    -- Plot ticks to the left of axis
    local px = self.graph.yaxis.origin
    local step = 0
    while true do
        px = px - pmajortick
        if px < 0 then break end

        -- Draw tick mark
        gc:drawLine(px, py-ticklength, px, py+ticklength)

        -- Label tick mark
        step = step - 1
        local value = tostring(step * xinc)
        local vwidth = gc:getStringWidth(value)
        gc:drawString(value, px-vwidth/2, py+ticklength+labeldist, "top")
    end

    -- Plot ticks to the right of axis
    px = self.graph.yaxis.origin
    step = 0
    while true do
        px = px + pmajortick
        if px > w then break end

        -- Draw tick mark
        gc:drawLine(px, py-ticklength, px, py+ticklength)

        -- Label tick mark
        step = step + 1
        local value = tostring(step * xinc)
        local vwidth = gc:getStringWidth(value)
        gc:drawString(value, px-vwidth/2, py+ticklength+labeldist, "top")
    end
end







---------------------------------------------------------------------- VerticalAxis

VerticalAxis = class(Axis)

function VerticalAxis:paint(gc)
    local w = platform.window:width()
    local h = platform.window:height()

    local px, linestyle, fontsize, axiscolor = self:checkAxisLimits(self.origin, w)

    gc:setColorRGB(unpack(axiscolor))
    gc:setPen("thin", linestyle)
    gc:drawLine(px, 0, px, h)

    local ticklength = Theme.ticklength
    local labeldist = Theme.labeldist

    local ptick = Theme.tickpixels
    local majortick = self.graph.majortick
    local pmajortick = majortick * ptick

    local yinc = self.graph.inc * majortick

    gc:setFont("sansserif", "r", fontsize)
    local vheight = gc:getStringHeight("1")/2

    -- Plot ticks above the axis
    local py = h - self.graph.xaxis.origin
    local step = 0
    while true do
        py = py - pmajortick
        if py < 0 then break end

        -- Draw tick mark
        gc:drawLine(px-ticklength, py, px+ticklength, py)

        -- Label tick mark
        step = step + 1
        local value = tostring(step * yinc)
        local vwidth = gc:getStringWidth(value)
        gc:drawString(value, px-ticklength-vwidth-labeldist, py - vheight, "top")
    end

    -- Plot ticks below axis
    py = h - self.graph.xaxis.origin
    step = 0
    while true do
        py = py + pmajortick
        if py > h then break end

        -- Draw tick mark
        gc:drawLine(px-ticklength, py, px+ticklength, py)

        -- Label tick mark
        step = step - 1
        local value = tostring(step * yinc)
        local vwidth = gc:getStringWidth(value)
        gc:drawString(value, px-ticklength-vwidth-labeldist, py - vheight, "top")
    end

end


function VerticalAxis:setOrigin(o)
    self.origin = math.floor(o + 0.5)
    document.markChanged()
end






---------------------------------------------------------------------- Function

Function = class()


function Function:init(graph)
    self.graph = graph
    self.color = Theme.functioncolor
    self.cache = {}
end


function Function:value(x)
    local cached_value = self.cache[x]
    if cached_value then
        return cached_value
    end
    local mx = tostring(x):gsub("e", EE)
    local y = math.eval("f1(" .. mx .. ")")
    self.cache[x] = y
    return y
end


function Function:flushCache()
    self.cache = {}
end


function Function:paint(gc)
    local prevx
    local prevy

    gc:setColorRGB(unpack(self.color))
    gc:setPen("medium", "smooth")

    for px = 0, platform.window:width() - 1 do
        local x = self.graph:tox(px)
        local y = self:value(x)
        if y then
            local py = self.graph:topy(y)
            if py > 30000 or py < -30000 then
                py = nil
                prevy = nil
            end
            if prevy then
                gc:drawLine(prevx, prevy, px, py)
            end
            prevx = px
            prevy = py
        else
            prevy = nil
        end
    end
end






---------------------------------------------------------------------- Rectangle

Rectangle = class()


function Rectangle:init(graph, color, bordercolor, x, width)
    self.graph  = graph
    self.color  = color
    self.bordercolor = bordercolor
    self.x      = x
    self.width  = width
end


function Rectangle:value(x)
    return self.graph.func:value(x)
end


function Rectangle:paint(gc)
    local wx = self.graph:topx(self.x)
    local wy = self:top()
    local ww = self.width / self.graph.dx
    local wh = platform.window:height() - self.graph.xaxis.origin - wy
    gc:setColorRGB(unpack(self.color))
    if wh < 0 then
        wh = -wh
        wy = wy - wh
    end
    gc:fillRect(wx, wy, ww, wh)
    gc:setColorRGB(unpack(self.bordercolor))
    gc:setPen("thin", "smooth")
    gc:drawRect(wx, wy, ww, wh)
end


function Rectangle.paintArea(gc, label, line, color, value)
    gc:setFont("sansserif", "b", 10)
    gc:setColorRGB(unpack(color))
    gc:drawString(label .. " = " .. string.format("%.5g", value), Theme.labelX, line, "middle")
end






---------------------------------------------------------------------- LeftRectangle

LeftRectangle = class(Rectangle)

function LeftRectangle:init(graph, x, width)
    Rectangle.init(self, graph, Color.lightgreen, Color.darkgreen, x, width)
end


function LeftRectangle:top()
    return self.graph:topy(self:value(self.x) or 0)
end


function LeftRectangle.calcArea(f, x, width)
    return width * (f:value(x) or 0)
end


function LeftRectangle.paintArea(gc, area)
    Rectangle.paintArea(gc, "Left", Theme.leftlabelY, Color.darkgreen, area)
end






---------------------------------------------------------------------- RightRectangle

RightRectangle = class(Rectangle)

function RightRectangle:init(graph, x, width)
    Rectangle.init(self, graph, Color.lightblue, Color.darkblue, x, width)
end


function RightRectangle:top()
    return self.graph:topy(self:value(self.x + self.width) or 0)
end


function RightRectangle.calcArea(f, x, width)
    return width * (f:value(x + width) or 0)
end


function RightRectangle.paintArea(gc, area)
    Rectangle.paintArea(gc, "Right", Theme.rightlabelY, Color.darkblue, area)
end






---------------------------------------------------------------------- MidRectangle

MidRectangle = class(Rectangle)

function MidRectangle:init(graph, x, width)
    Rectangle.init(self, graph, Color.lightred, Color.darkred, x, width)
end


function MidRectangle:top()
    return self.graph:topy(self:value(self.x + self.width/2) or 0)
end


function MidRectangle.calcArea(f, x, width)
    return width * (f:value(x + width/2) or 0)
end


function MidRectangle.paintArea(gc, area)
    Rectangle.paintArea(gc, "Mid", Theme.midlabelY, Color.darkred, area)
end






---------------------------------------------------------------------- Trapezoid

Trapezoid = class(Rectangle)

function Trapezoid:init(graph, x, width)
    Rectangle.init(self, graph, Color.lightyellow, Color.darkyellow, x, width)
end


function Trapezoid:paint(gc)
    local g = self.graph
    local function xandy(x)
        return g:topx(x), g:topy(self:value(x) or 0)
    end

    local verts = {}

    verts[1], verts[2] = xandy(self.x)
    verts[3], verts[4] = xandy(self.x + self.width)
    verts[5], verts[6] = verts[3], g:topy(0)
    verts[7], verts[8] = verts[1], verts[6]
    verts[9], verts[10] = verts[1], verts[2]

    gc:setColorRGB(unpack(self.color))
    gc:fillPolygon(verts)
    gc:setPen("thin", "smooth")
    gc:setColorRGB(unpack(self.bordercolor))
    gc:drawPolyLine(verts)
end


function Trapezoid.calcArea(f, x, width)
    return ((f:value(x) or 0) + (f:value(x + width) or 0)) * width/2
end


function Trapezoid.paintArea(gc, area)
    Rectangle.paintArea(gc, "Trapezoid", Theme.traplabelY, Color.darkyellow, area)
end






---------------------------------------------------------------------- Riemann
-- This is a collection of rectangles dividing the space between two
-- handles which define the left and right ends of the domain over
-- which to integrate a function.

Riemann = class()

function Riemann:init(graph, divs)
    self.graph = graph
    self.divs  = divs or 4
    self.rectype = LeftRectangle
end


function Riemann:increaseDivs()
    self.divs = self.divs + 1
    document.markChanged()
    platform.window:invalidate()
end


function Riemann:decreaseDivs()
    if self.divs > 1 then
        self.divs = self.divs - 1
        document.markChanged()
        platform.window:invalidate()
    end
end


function Riemann:setRectType(rectype)
    self.rectype = rectype
    platform.window:invalidate()
end


function Riemann:setLeft()
    self:setRectType(LeftRectangle)
end


function Riemann:setRight()
    self:setRectType(RightRectangle)
end


function Riemann:setMid()
    self:setRectType(MidRectangle)
end


function Riemann:setTrap()
    self:setRectType(Trapezoid)
end


function Riemann:buildRects()
    local rects = {}
    local left = self.graph.lhandle.x
    local right = self.graph.rhandle.x
    local width = (right - left)/self.divs

    for i = 1, self.divs do
        table.insert(rects, self.rectype(self.graph, left, width))
        left = left + width
    end

    return rects
end


function Riemann:paintAreas(gc)
    local left = self.graph.lhandle.x
    local right = self.graph.rhandle.x
    local width = (right - left)/self.divs
    local f = self.graph.func

    local calcAreas = function(rectype)
        local area = 0
        local x = left
        for i = 1, self.divs do
            area = area + rectype.calcArea(f, x, width)
            x = x + width
        end
        return area
    end

    local leftArea  = calcAreas(LeftRectangle)
    local rightArea = calcAreas(RightRectangle)
    local midArea   = calcAreas(MidRectangle)
    local trapArea  = calcAreas(Trapezoid)

    LeftRectangle.paintArea(gc, leftArea)
    RightRectangle.paintArea(gc, rightArea)
    MidRectangle.paintArea(gc, midArea)
    Trapezoid.paintArea(gc, trapArea)
end


function Riemann:paint(gc)
    for _, r in ipairs(self:buildRects()) do
        r:paint(gc)
    end

    self:paintAreas(gc)
end






---------------------------------------------------------------------- Handle
-- The user can grab a handle to set the upper or lower end of the
-- domain to analyze.
Handle = class()

function Handle:init(graph, x)
    self.graph = graph
    self.x = x
    MouseHit.register(self)
    self.tracking = false
end


function Handle:setTracking(t)
    self.tracking = t
end


function Handle:grab()
    theCursor = "drag grab"
    cursor.set(theCursor)
end


function Handle:getpx()
    return self.px
end


function Handle:getpy()
    return platform.window:height() - self.graph.xaxis.origin
end


function Handle:calcx()
    return self.graph:tox(self.px)
end


function Handle:paint(gc)
    self.px = self.graph:topx(self.x)
    local color = Theme.handleColor
    if self.tracking then
        color = Theme.handleSelectedColor
    end
    gc:setColorRGB(unpack(color))
    gc:setPen("medium", "smooth")
    gc:drawPolyLine(self:segments())
    if self.tracking then
        gc:setFont("sansserif", Theme.handleValueFontStyle, Theme.handleValueFontSize)
        local value = string.format("%.4g", self:calcx())
        local vw    = gc:getStringWidth(value)
        local vh    = gc:getStringHeight(value)
        gc:drawString(value, self:getpx() - vw/2, self:getpy() - Theme.handleValueOffset, "bottom")
    end
end


Handle.Rise = 10
Handle.Run  = 5






---------------------------------------------------------------------- LeftHandle
LeftHandle = class(Handle)


function LeftHandle:init(graph, x)
    Handle.init(self, graph, x)
    self.minusbutton = CommandButton(Image.minusButton, function() graph:decr() end)
    self.plusbutton = CommandButton(Image.plusButton, function() graph:incr() end)
end


function LeftHandle:segments()
    local px = self.px
    local py = platform.window:height() - self.graph.xaxis.origin
    local rise = self.Rise
    local run  = self.Run

    return {px+run, py-rise, px, py-rise, px, py+rise, px+run, py+rise}
end


function LeftHandle:trackMouse(wx, wy)
    if wx < 10 then
        wx = 10
    end
    local right = self.graph.rhandle.px - self.Run*3
    if wx > right then
        wx = right
    end
    self.x = self.graph:tox(wx)
    document.markChanged()
    platform.window:invalidate()
end


function LeftHandle:contains(px, py)
    local left = self.px
    local top  = platform.window:height() - self.graph.xaxis.origin - self.Rise
    local width = self.Run
    local height = 2*self.Rise

    return px >= left and px <= left+width and
           py >= top  and py <= top+height
end


function LeftHandle:paint(gc)
    Handle.paint(self, gc)
    local px = self:getpx()
    local py = self:getpy()

    self.minusbutton:setxy(px-10, py+44)
    self.plusbutton:setxy(px+34, py+44)
    self.minusbutton:paint(gc)
    self.plusbutton:paint(gc)
end






---------------------------------------------------------------------- RightHandle
RightHandle = class(Handle)


function RightHandle:segments()
    local px = self.px
    local py = platform.window:height() - self.graph.xaxis.origin
    local rise = self.Rise
    local run  = self.Run

    return {px-run, py-rise, px, py-rise, px, py+rise, px-run, py+rise}
end


function RightHandle:trackMouse(wx, wy)
    local right = platform.window:width() - 10
    if wx >= right then
        wx = right
    end
    local left = self.graph.lhandle.px + self.Run*3
    if wx < left then
        wx = left
    end
    self.x = self.graph:tox(wx)
    document.markChanged()
    platform.window:invalidate()
end


function RightHandle:contains(px, py)
    local left = self.px - self.Run
    local top  = platform.window:height() - self.graph.xaxis.origin - self.Rise
    local width = self.Run
    local height = 2*self.Rise

    return px >= left and px <= left+width and
           py >= top  and py <= top+height
end






---------------------------------------------------------------------- Image

Image = {

    radioButtonSelected = image.new("\011\000\000\000\011\000\000\000\000\000\000\000\022\000\000\000\016\000\001\000\255\127\255\1279\231\016\194)\165\165\148)\165\016\1949\231\255\127\255\127\255\127\214\218\206\185s\206Z\235\255\255Z\235s\206\206\185\214\218\255\1279\231\206\185{\2399\231\148\210\239\189s\206\024\227{\239\206\1859\231\016\194s\2069\231\239\189\206\185\206\185\140\177k\173\247\222s\206\016\194)\165Z\2351\198\140\177\140\177k\173)\165\008\161\206\185Z\235)\165\165\148\255\255J\169)\165\008\161\231\156\231\156\198\152\231\156\255\255\165\148)\165Z\235\206\185\231\156\231\156\198\152\198\152\165\148\173\181Z\235)\165\016\194s\206\214\218\231\156\165\148\165\148\165\148\165\148\181\214s\206\016\1949\231\206\185{\239\214\218\140\177\198\152\140\177\181\214{\239\206\1859\231\255\127\214\218\206\185s\206Z\235\255\255Z\235s\206\206\185\214\218\255\127\255\127\255\1279\231\016\194)\165\165\148)\165\016\1949\231\255\127\255\127"),
    radioButtonUnselected = image.new("\011\000\000\000\011\000\000\000\000\000\000\000\022\000\000\000\016\000\001\000\255\127\255\1279\231\016\194)\165\165\148)\165\016\1949\231\255\127\255\127\255\127\214\218\206\185s\206Z\235\255\255Z\235s\206\206\185\214\218\255\1279\231\206\185{\239\255\255\222\251\189\247\222\251\222\251{\239\206\1859\231\016\194s\206\255\255\189\247\189\247\189\247\189\247\189\247\222\251s\206\016\194)\165Z\235\222\251\189\247\189\247\189\247\156\243\156\243\156\243Z\235)\165\165\148\255\255\189\247\156\243\156\243\156\243Z\235\024\227\024\227\255\255\165\148)\165Z\235\189\247Z\235\024\227\247\222\181\214\148\210\024\227Z\235)\165\016\194s\206\189\247\247\222\181\214s\206R\202R\202{\239s\206\016\1949\231\206\185{\239\156\243\214\218R\202\181\214Z\235{\239\206\1859\231\255\127\214\218\206\185s\206Z\235\255\255Z\235s\206\206\185\214\218\255\127\255\127\255\1279\231\016\194)\165\165\148)\165\016\1949\231\255\127\255\127"),

    minusButton = image.new("\023\000\000\000\024\000\000\000\000\000\000\000.\000\000\000\016\000\001\000\223w\223w\223w\223w\255\127\255\127\255\127\189\247\157\243\156\243\156\243\156\243\156\243\156\243\156\243\189\243\189\247\255\127\255\127\223w\223w\223w\223w\223w\223w\223w\223w\255\127\189\243\156\243\189\247\222\251\222\251\255\255\255\255\255\255\222\251\222\251\222\247\189\247\157\243\189\247\255\127\223w\223w\223w\223w\223w\255\127\190\247\189\247\189\247\222\251\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247\189\247\189\247\255\127\223w\223w\223w\255\127\255\127\189\243\189\247\222\251\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\247\189\247\189\247\255\127\222w\223w\255\127\189\243\189\247\222\251\255\255\255\255\255\255\222\251\222\251\222\251\189\247\222\251\222\251\222\251\255\255\255\255\255\255\255\255\222\251\189\243\189\243\255\127\255\127\189\247\189\243\222\251\255\255\255\255\222\251\222\251\222\251\189\247\189\247\189\247\189\247\189\247\189\247\222\251\222\251\255\255\255\255\255\255\189\247\156\243\255\127\255\127\189\247\189\247\255\255\255\255\222\251\222\251\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\222\251\255\255\255\255\222\251\189\247\189\247\189\243\189\247\222\251\255\255\222\251\222\251\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\222\251\255\255\255\255\222\247\156\243\156\243\189\247\255\255\255\255\222\251\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\222\251\255\255\222\251\156\243\156\243\189\247\255\255\222\251\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\189\247\222\251\222\251\255\255\156\243\156\243\189\247\255\255\222\251\189\247\189\247\189\247\189\247\024\227\181\214\181\214\181\214\181\214\214\218Z\235\156\243\189\247\189\247\189\247\189\247\222\251\255\255\156\243\156\243\189\247\255\255\189\247\189\247\189\247\189\247\156\2431\198)\165)\165)\165)\165\173\181\247\222{\239\156\243\189\247\189\247\189\247\222\251\255\255\156\243\156\243\189\247\255\255\189\247\189\247\156\243\156\243\156\243\239\189\165\148\165\148\165\148\165\148)\165\214\218Z\235\156\243\156\243\156\243\189\247\222\251\255\255\156\243\156\243\189\247\255\255\189\247\189\247\156\243\156\243\156\243\247\222s\206s\206s\206s\206\181\214Z\235{\239\156\243\156\243\156\243\189\247\222\251\222\251\156\243\156\243\156\243\222\251\189\247\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\189\247\222\251{\239\156\239\156\243\189\247\189\247\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\189\247\189\247{\239\157\243\156\239\156\243\189\247\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\189\247\156\243\156\239\255\127\156\239{\239\189\247\189\247\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\189\247\156\243{\239\157\243\255\127\156\243{\235{\239\156\243\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\156\243{\239{\235\255\127\255\127\255\127|\239:\231{\239\156\243\189\247\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\189\247\156\243Z\235[\235\255\127\255\127\255\127\255\127\255\127{\239Z\231Z\235\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\239:\231[\235\156\239\255\127\255\127\223w\255\127\255\127\255\127{\239Z\235Z\235Z\235{\235{\239{\239\156\243\156\243{\239{\239Z\235Z\235Z\235Z\235\156\239\255\127\255\127\223w\223w\223w\223w\255\127\255\127\255\127Z\2319\2319\231Z\235Z\235{\239{\235Z\235:\2319\2319\231[\235\255\127\255\127\222w\223w\223w\223w\223w\223w\223w\255\127\255\127\255\127\255\127{\235Z\231:\231:\231:\231Z\231[\235|\239\255\127\255\127\255\127\255\127\223w\223w\223w"),
    plusButton = image.new("\023\000\000\000\024\000\000\000\000\000\000\000.\000\000\000\016\000\001\000\223w\223w\223w\223w\223w\255\127\222\247\189\247\189\243\157\243\156\243\156\243\156\243\156\243\157\243\189\247\255\127\255\127\223w\223w\223w\223w\223w\223w\223w\223w\223w\222\247\190\247\189\247\189\247\189\247\189\247\189\247\190\247\190\247\189\247\189\247\189\247\189\247\189\247\190\247\255\127\223w\223w\223w\223w\223w\255\127\222\247\189\247\189\247\189\247\222\251\222\251\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247\189\247\189\243\190\247\255\127\223w\223w\223w\223w\222\247\189\243\189\243\189\247\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\222\251\189\247\157\243\189\247\223w\223w\223w\222\247\189\247\189\243\222\251\255\255\255\255\255\255\255\255\222\251\222\251\222\251\222\251\222\251\222\251\255\255\255\255\255\255\255\255\189\247\189\247\190\247\255\127\222\247\190\247\189\243\222\251\255\255\255\255\255\255\222\251\222\251\189\247\189\247\189\247\189\247\189\247\189\247\222\251\222\251\255\255\255\255\222\251\189\247\189\243\222\247\222\247\189\247\189\247\255\255\255\255\255\255\222\251\189\247\189\247\156\243{\2399\231Z\235\156\243\189\247\189\247\189\247\222\251\255\255\255\255\222\251\156\243\190\247\189\243\189\247\222\251\255\255\222\251\222\251\189\247\189\247\189\247{\239\214\218)\165\239\189\024\227\189\247\189\247\189\247\189\247\222\251\255\255\255\255\189\247\157\243\157\243\189\247\222\251\255\255\222\251\189\247\189\247\189\247\189\247Z\235\181\214\008\161\173\181\024\227\156\243\189\247\189\247\189\247\189\247\222\251\255\255\222\251\156\243\156\243\189\247\255\255\222\251\189\247\189\247\156\243Z\235Z\235\024\227s\206\231\156\140\177\181\214Z\235Z\235{\239\189\247\189\247\222\251\222\251\222\251\156\243\156\243\189\247\255\255\222\251\189\247\156\243{\239\181\214s\206R\202\239\189\198\152J\169\016\194s\206s\206\024\227\156\243\189\247\189\247\222\251\255\255\156\243\156\243\189\247\255\255\189\247\189\247\156\243Z\235J\169\198\152\198\152\198\152\132\144\165\148\198\152\198\152\198\152\016\194\156\243\156\243\189\247\222\251\255\255\156\243\156\243\189\247\255\255\189\247\156\243\156\243{\2391\198\239\189\206\185k\173\198\152\008\161\140\177\206\185\239\189\181\214\156\243\156\243\189\247\222\251\255\255\156\243\156\243\189\247\222\251\189\247\156\243\156\243{\239\024\227\247\222\214\218R\202\231\156k\173\148\210\247\222\247\2229\231\156\243\156\243\156\243\189\247\222\251\156\243\156\243\189\243\222\251\189\247\156\243\156\243\156\243\156\243\156\243Z\235\148\210\008\161\173\181\247\222{\239\156\243\156\243\156\243\156\243\156\243\189\247\222\251|\239\156\239\156\243\189\247\189\247\156\243\156\243\156\243\156\243\156\243Z\235\148\210\008\161\173\181\247\222{\239\156\243\156\243\156\243\156\243\189\247\189\247\189\247{\239\156\243\156\243\189\247\189\247\156\243\156\243\156\243\156\243\156\243{\239\024\227R\202\181\2149\231{\239\156\243\156\243\156\243\156\243\189\247\189\247\156\243|\239\189\243\156\239\156\243\189\247\189\247\156\243\156\243\156\243\156\243{\239{\239{\239{\239{\239{\239\156\243\156\243\156\243\156\243\189\247\156\243|\239\156\239\255\127\156\239Z\235\156\243\189\247\156\243\156\243\156\243\156\243\156\243{\239{\239{\239{\239\156\243\156\243\156\243\156\243\189\247\189\247|\239Z\235\255\127\255\127\255\127{\239Z\235\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243{\239[\235\156\239\255\127\223w\255\127\255\127[\235[\235{\239\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243\156\243{\235Z\235|\239\255\127\255\127\223w\255\127\255\127\255\127Z\235Z\235{\239{\239{\239{\239\156\243\156\243\156\243\156\243{\239{\239{\239{\2399\231{\235\255\127\255\127\223w\223w\223w\255\127\255\127\255\127{\2359\227\024\2279\231Z\235{\239{\239{\239Z\2359\231\025\227\025\227:\231\156\239\255\127\255\127\223w\223w\223w\223w\223w\255\127\255\127\255\127|\239{\235Z\235Z\231Z\231Z\231Z\231Z\231Z\231[\235{\235\156\239\255\127\255\127\255\127\223w\223w"),

    zoomInButton = image.new("\024\000\000\000\024\000\000\000\000\000\000\0000\000\000\000\016\000\001\000\000\000\000\000\000\000\000\000\000\000\000\128\132\144J\165\141\177\141\173*\161c\140\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\231\152\149\202;\227\092\227;\223\026\219\249\214\183\206\241\181\165\144\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\174\177}\231;\227\026\219\027\219<\223\027\219\250\214\183\206\150\198U\194\010\157\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\207\181}\231\026\219]\231\159\239\191\243\191\243\191\243\159\239~\231\250\2104\186U\194\010\157\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\231\152\092\227\026\219\158\239\223\243\191\243\191\243\191\243\191\239\191\243\191\243\159\239\219\2064\186T\194\133\144\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\149\202\026\219}\235\191\243\191\239\191\243\191\239<\182\027\178\159\235\191\243\191\243\127\231\186\202U\190\175\177\000\128\000\000\000\000\000\000\000\000\000\000\000\000\133\144\249\214\026\219\223\247\191\243\191\239\191\239\159\235\251\165\186\161\159\239\223\247\191\243\191\239>\223V\1944\190\132\140\000\000\000\000\000\000\000\000\000\000\000\000K\165\217\210\092\231\223\247\223\247\191\243\191\243\159\235\218\165\185\161~\231\191\239\191\239\191\239\127\231\185\202T\190\009\157\000\000\000\000\000\000\000\000\000\000\000\000l\169\183\206}\235\223\251\191\243<\182\251\169\250\169\185\161\152\157\184\161\150\157\215\169\159\235\159\235\252\214U\190K\161\000\000\000\000\000\000\000\000\000\000\000\000L\165v\198\157\239\255\251\191\243\251\169\186\161\185\157\152\157w\153v\1535\149u\157\127\235\159\235\028\219U\194L\165\000\000\000\000\000\000\000\000\000\000\000\000\009\157T\190|\235\255\255\223\251\159\239\159\239~\235\151\161v\153]\227~\235~\231\159\235\159\239\028\219v\198*\161\000\000\000\000\000\000\000\000\000\000\000\000\132\1404\190\249\218\255\255\255\255\255\255\255\255\223\251\150\157U\153\190\243\255\251\223\247\159\239\159\239\217\210\151\202\166\148\000\000\000\000\000\000\000\000\000\000\000\000\000\128\209\177U\194\222\247\255\255\255\255\255\255\223\247\150\161u\157\158\243\223\251\191\243\191\239~\231\216\210T\194\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\198\148T\190\183\210\255\255\255\255\255\255\255\255\190\243\158\239\223\251\223\247\191\243\191\239\250\214\249\210\008\157\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000l\165U\190\215\210\223\251\255\255\255\255\255\255\255\255\223\251\223\247\159\239\026\219\249\214S\194!\132\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128l\165T\190u\198:\227\157\243\222\247\191\247\157\239;\227\026\219\026\219\241\181\182\206\240\185\132\144\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\199\148\241\181u\194\151\202\216\210\249\214\249\214\026\219\182\206*\161B\132\019\186R\198\008\161\231\156\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\198\148l\165\175\177\207\181\141\173\232\152!\132\000\000\000\000\016\194\148\210)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\165\148\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165c\140\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\140\177\008\161\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\000\128\000\000\000\000"),    zoomOutButton = image.new("\024\000\000\000\024\000\000\000\000\000\000\0000\000\000\000\016\000\001\000\000\000\000\000\000\000\000\000\000\000\000\128\132\144J\165\141\177\141\173*\161c\140\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\231\152\149\202;\227\092\227;\223\026\219\249\214\183\206\241\181\165\144\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\174\177}\231;\227\026\219\027\219<\223\027\219\250\214\183\206\150\198U\194\010\157\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\207\181}\231\026\219]\231\159\239\191\243\191\243\191\243\159\239~\231\250\2104\186U\194\010\157\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\231\152\092\227\026\219\158\239\223\243\191\243\191\243\191\243\191\243\191\243\191\243\159\239\219\2064\186T\194\133\144\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\149\202\026\219}\235\191\243\191\239\191\243\191\243\191\243\191\243\191\243\191\243\191\243\127\231\186\202U\190\175\177\000\128\000\000\000\000\000\000\000\000\000\000\000\000\133\144\249\214\026\219\223\247\191\243\191\239\191\239\191\239\191\243\191\243\223\243\191\243\191\243\191\239>\223V\1944\190\132\140\000\000\000\000\000\000\000\000\000\000\000\000K\165\217\210\092\231\223\247\223\247\191\243\191\243\159\239\159\235\159\235\159\235\159\239\191\239\191\239\127\231\185\202T\190\009\157\000\000\000\000\000\000\000\000\000\000\000\000l\169\183\206}\235\223\251\191\243<\182\251\169\250\169\217\169\216\165\184\161\150\157\215\169\159\235\159\235\252\214U\190K\161\000\000\000\000\000\000\000\000\000\000\000\000L\165v\198\157\239\255\251\191\243\251\169\186\161\185\157\152\157w\153v\1535\149u\157\127\235\159\235\028\219U\194L\165\000\000\000\000\000\000\000\000\000\000\000\000\009\157T\190|\235\255\255\223\251\159\239\159\239\158\239\158\239\158\235~\235~\235~\231\159\235\159\239\028\219v\198*\161\000\000\000\000\000\000\000\000\000\000\000\000\132\1404\190\249\218\255\255\255\255\255\255\255\255\255\255\255\255\255\251\255\251\223\251\223\247\159\239\159\239\217\210\151\202\166\148\000\000\000\000\000\000\000\000\000\000\000\000\000\128\209\177U\194\222\247\255\255\255\255\255\255\255\255\255\251\223\251\223\251\223\247\191\243\191\239~\231\216\210T\194\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\198\148T\190\183\210\255\255\255\255\255\255\255\255\255\255\255\251\223\251\191\247\191\243\191\239\250\214\249\210\008\157\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000l\165U\190\215\210\223\251\255\255\255\255\255\255\255\251\223\251\223\247\159\239\026\219\249\214S\194!\132\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128l\165T\190u\198:\227\157\243\222\247\191\247\157\239;\227\026\219\026\219\241\181\182\206\240\185\132\144\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\199\148\241\181u\194\151\202\216\210\249\214\249\214\026\219\182\206*\161B\132\018\186R\202\008\161\231\156\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\198\148l\165\175\177\207\181\141\173\232\152!\132\000\000\000\000\016\194\148\210)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\231\156\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165\008\161\165\148\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\239\189\181\214)\165c\140\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000!\132\140\177\008\161\000\128\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\000\128\000\000\000\000"),
    hhCloseButton = image.new("\016\000\000\000\016\000\000\000\000\000\000\000 \000\000\000\016\000\001\0009\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\2319\231\025\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\2279\227\025\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\024\227\248\222\248\222\248\222\017\230m\233\248\222\248\222\248\222\248\222\248\222\248\222\142\233t\226\248\222\248\222\248\222\247\218\247\218\247\218M\229g\236m\233\247\218\247\218\247\218\247\218m\233g\236\142\229\247\218\247\218\247\218\214\218\214\218\214\218\214\218L\229g\236M\229\214\218\214\218M\229g\236L\229\214\218\214\218\214\218\214\218\182\214\182\214\182\214\182\214\182\214L\229g\236L\229L\229g\236L\229\182\214\182\214\182\214\182\214\182\214\181\214\181\214\181\214\181\214\181\214\181\214L\229g\236g\236L\229\181\214\181\214\181\214\181\214\181\214\181\214\148\210\148\210\148\210\148\210\148\210\148\210L\229g\236g\236L\229\148\210\148\210\148\210\148\210\148\210\148\210t\206t\206t\206t\206t\206,\225g\236,\225,\225g\236,\225t\206t\206t\206t\206t\206s\206s\206s\206s\206+\225g\236+\225s\206s\206+\225g\236+\225s\206s\206s\206s\206S\202S\202S\202L\221g\236+\225S\202S\202S\202S\202+\225g\236+\225S\202S\202S\202R\202R\202R\202\240\209L\221R\202R\202R\202R\202R\202R\202+\225\142\213R\202R\202R\2022\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1982\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\1981\198\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194\017\194"),
    closeButton = image.new(".\000\000\000\018\000\000\000\000\000\000\000\092\000\000\000\016\000\001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\2558\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\024\2398\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\023\239\024\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\247\238\024\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\240\185\240\185\240\185j\217j\217j\217\240\185\240\185\240\185i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\247\238\023\239i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213j\217\240\185\255\255\255\255\255\255\240\185\138\217\240\185\255\255\255\255\255\255\240\185j\217i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\246\238\246\234i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213j\213j\217\138\217\138\217\240\185\255\255\255\255\255\255\240\185\255\255\255\255\255\255\240\185\138\217\138\217j\217j\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213i\213\213\234\180\230\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\196\204\196\204\196\204\196\204\229\204\229\208\229\208\005\209L\169\255\255\255\255\255\255\255\255\255\255L\169\005\209\229\208\229\208\229\204\196\204\196\204\196\204\196\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200r\226\147\230\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\196\204\196\204\196\204\228\204\229\204\229\208\005\209\005\209\006\209\006\209L\169\222\251\222\251\222\251L\169\006\209\006\209\005\209\005\209\229\208\229\204\228\204\196\204\196\204\196\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200r\2260\222\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\195\204\196\204\196\204\196\204\229\204\229\208\005\209\006\209\006\209&\213&\213L\169\189\247\189\247\189\247L\169&\213&\213\006\209\006\209\005\209\229\208\229\204\196\204\196\204\196\204\195\204\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\163\200\205\217\015\222\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\228\204\229\204\229\208\229\208\005\209\006\209&\213&\213'\213G\213L\169\156\243\156\243\156\243\156\243\156\243L\169G\213'\213&\213&\213\006\209\005\209\229\208\229\208\229\204\228\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\196\204\205\217\016\222\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\006\209\006\213&\213'\213'\213G\213G\217H\217L\169{\239{\239{\239L\169{\239{\239{\239L\169H\217G\217G\213'\213'\213&\213\006\213\006\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\005\209\205\2170\222'\213'\213'\213'\213'\213'\213'\213'\213G\213G\213G\213G\217H\217h\217h\217h\217L\169{\239{\239{\239L\169\137\221L\169{\239{\239{\239L\169h\217h\217h\217H\217G\217G\213G\213G\213'\213'\213'\213'\213'\213'\213'\213'\213'\213\237\217Q\222h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217\137\221\137\221\137\221\137\221\137\221L\169L\169L\169\169\221\169\221\169\221L\169L\169L\169\137\221\137\221\137\221\137\221\137\221h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217h\217\015\218Q\222\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\169\221\169\221\169\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\170\221\169\221\169\221\169\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\221\137\2210\222r\226\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225r\226\147\226\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\170\225\147\226\247\234r\226r\226r\226r\230r\230r\230r\230r\230r\230\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\234\146\230r\230r\230\146\234\146\234\146\230r\230r\230\247\234"),
}






---------------------------------------------------------------------- MouseHit

MouseHit = {}
MouseHit.list = {}

function MouseHit.register(obj)
    table.insert(MouseHit.list, obj)
end


function MouseHit.findHit(px, py)
    for _, obj in ipairs(MouseHit.list) do
        if obj:contains(px, py) then
            return obj
        end
    end
    return nil
end






---------------------------------------------------------------------- Button

Button = class()

function Button:setTracking(t)
    if t then
        self:select()
    end
end


function Button:trackMouse(x, y)
end


function Button:grab()
end


function Button:getpx()
    return self.x
end


function Button:getpy()
    return self.y
end


function Button:contains(wx, wy)
    return math.sqrt((wx - self.x)^2 + (wy - self.y)^2) <= self.radius
end







---------------------------------------------------------------------- CommandButton

CommandButton = class(Button)

function CommandButton:init(image, cmd)
    self.image      = image
    self.cmd        = cmd
    self.radius     = image:width()/2
    MouseHit.register(self)
end


function CommandButton:select()
    self.cmd()
    platform.window:invalidate()
end


function CommandButton:paint(gc)
    gc:drawImage(self.image, self.x-self.radius, self.y-self.radius)
end


function CommandButton:setxy(x, y)
    self.x      = x
    self.y      = y
end






---------------------------------------------------------------------- RadioButtonGroup

RadioButtonGroup = class()

function RadioButtonGroup:init()
    self.members = {}
end


function RadioButtonGroup:add(member)
    table.insert(self.members, member)
end


function RadioButtonGroup:unselectOthers(member)
    for _, button in ipairs(self.members) do
        if button ~= member then
            button:unselect()
        end
    end
end


function RadioButtonGroup:paint(gc)
    for _, btn in ipairs(self.members) do
        btn:paint(gc)
    end
end






---------------------------------------------------------------------- RadioButton

RadioButton = class(Button)

function RadioButton:init(x, y, group, selected, cmd)
    self.x          = x
    self.y          = y
    self.group      = group
    self.cmd        = cmd
    self.selected   = selected
    self.radius     = Image.radioButtonUnselected:width() / 2
    group:add(self)
    MouseHit.register(self)
end


function RadioButton:select()
    self.group:unselectOthers(self)
    self.selected = true
    self.cmd()
    platform.window:invalidate()
end


function RadioButton:unselect()
    self.selected = false
    platform.window:invalidate()
end


function RadioButton:paint(gc)
    local image
    if self.selected then
        image = Image.radioButtonSelected
    else
        image = Image.radioButtonUnselected
    end
    gc:drawImage(image, self.x - self.radius, self.y - self.radius)
end


selectRectButtonGroup = RadioButtonGroup()

theLeftButton = RadioButton(Theme.labelbuttonX, Theme.leftlabelY, selectRectButtonGroup, true,
    function() theGraph:setLeft() end
)
theRightButton = RadioButton(Theme.labelbuttonX, Theme.rightlabelY, selectRectButtonGroup, false,
    function() theGraph:setRight() end
)
theMidButton = RadioButton(Theme.labelbuttonX, Theme.midlabelY, selectRectButtonGroup, false,
    function() theGraph:setMid() end
)
theTrapButton = RadioButton(Theme.labelbuttonX, Theme.traplabelY, selectRectButtonGroup, false,
    function() theGraph:setTrap() end
)







-------------------------------------------------------------------------- View management


local theViews = {}

function addView(view)
    table.insert(theViews, view)
    platform.window:invalidate()
end


function closeTopView()
    if #theViews > 1 then
        table.remove(theViews)
        platform.window:invalidate()
    end
end


function paintViews(gc)
    for _, view in ipairs(theViews) do
        view:paint(gc)
    end
end


function currentView()
    return theViews[#theViews]
end






-------------------------------------------------------------------------- View
View = class()

function View:paint(gc)
    -- Implemented in subclass
end


function View:help()
    -- Implemented in subclass
end


function View:escapeKey()
    -- Implemented in subclass
end


function View:charIn(ch)
    -- Implemented in subclass
end






-------------------------------------------------------------------------- ViewWithCloseButton

ViewWithCloseButton = class(View)


function ViewWithCloseButton:setup(width, height)
    local w = platform.window:width()
    local h = platform.window:height()

    self.x = (w - width)/2
    if self.x < 0 then
        self.x = 0
    end
    self.y = (h - height)/2
    if self.y < 0 then
        self.y = 0
    end
    self.w = width
    self.h = height
end


function ViewWithCloseButton:setFontInfo(gc)
    if platform.isDeviceModeRendering() then
        self.titleFontSize = 9
        self.infoFontSize = 7
    else
        self.titleFontSize = 12
        self.infoFontSize = 11
    end

    gc:setFont("sansserif", "r", self.titleFontSize)
    self.titleHeight = gc:getStringHeight("r")
    gc:setFont("sansserif", "r", self.infoFontSize)
    self.infoHeight = gc:getStringHeight("r")
end


function ViewWithCloseButton:paint(gc)
    -- Clear interior
    gc:setColorRGB(0xF3, 0xF7, 0xFA)
    gc:fillRect(self.x, self.y, self.w, self.h)

    -- Paint border
    gc:setColorRGB(0, 0, 0)
    gc:setPen("medium", "smooth")
    gc:drawRect(self.x, self.y, self.w, self.h)

    -- Paint close box
    local closeButton
    if platform.isDeviceModeRendering() then
        closeButton = Image.hhCloseButton
    else
        closeButton = Image.closeButton
    end
    gc:drawImage(closeButton, self.x + self.w - closeButton:width(),
                              self.y - closeButton:height()/2)
    
end


function ViewWithCloseButton:mouseDown(x, y)
    self:escapeKey()
end


function ViewWithCloseButton:escapeKey()
    closeTopView()
end





-------------------------------------------------------------------------- AboutView
AboutView = class(ViewWithCloseButton)


function AboutView:setup(gc)
    self:setFontInfo(gc)
    if platform.window:width() <= 320 then
        self.width = 280
    else
        self.width = 500
    end

    local height = self.titleHeight + 8*self.infoHeight + 10
    ViewWithCloseButton.setup(self, self.width, height)
end


function AboutView:paint(gc)
    self:setup(gc)
    local marginx = self.x + 10
    local marginy = self.y + 5
    local texty   = marginy

    ViewWithCloseButton.paint(self, gc)

    -- Title
    gc:setColorRGB(0, 0, 0)
    local title = "Riemann Integral"
    gc:setFont("sansserif", "b", self.titleFontSize)
    gc:drawString(title, marginx, texty, "top")
    texty = texty + self.titleHeight

    -- Version
    title     = "Version 1.0"
    gc:setFont("sansserif", "r", self.infoFontSize)
    gc:drawString(title, marginx, texty, "top")
    texty = texty + self.infoHeight

    -- Copyright
    title     = "Copyright " .. COPYRIGHT .. " 2010, Texas Instruments"
    gc:drawString(title, marginx, texty, "top")
    texty = texty + 1.5*self.infoHeight

    -- Help
    gc:drawString("Define function f1(x) in Graphs or in a Calculator", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Press '+' to increase the number of intervals", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Press '-' to decrease the number of intervals", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Drag the graph paper to view different portions of the graph", marginx, texty, "top")
    texty = texty + self.infoHeight
    gc:drawString("Drag the brackets on the x-axis to select domain of interest", marginx, texty, "top")
    texty = texty + self.infoHeight
end






---------------------------------------------------------------------- Graph

Graph = class(View)

function Graph:init()
    self.domain = 4
    self.factor = 2
    self.xaxis = HorizontalAxis(self)
    self.yaxis = VerticalAxis(self)
    self.paper = GraphPaper(self)
    self.func  = Function(self)
    self.lhandle = LeftHandle(self, 0)
    self.rhandle = RightHandle(self, 1)
    self.zoomInButton = CommandButton(Image.zoomInButton, function() self:zoomin() end)
    self.zoomOutButton = CommandButton(Image.zoomOutButton, function() self:zoomout() end)
    self.riemann = Riemann(self)
    MouseHit.register(self)
end


function Graph:setTracking(t)
end


function Graph:grab()
    theCursor = "hand closed"
    cursor.set(theCursor)
end


function Graph:setLeft()
    self.riemann:setLeft()
end


function Graph:setRight()
    self.riemann:setRight()
end


function Graph:setMid()
    self.riemann:setMid()
end


function Graph:setTrap()
    self.riemann:setTrap()
end


function Graph:incr()
    self.riemann:increaseDivs()
end


function Graph:decr()
    self.riemann:decreaseDivs()
end


function Graph:contains(px, py)
    return true
end


function Graph:getpx()
    return self.yaxis.origin
end


function Graph:getpy()
    return self.xaxis.origin
end


function Graph:setFactor(f)
    self.factor = f
end


function Graph:resize()
    local pwidth = platform.window:width()
    local divs = math.max(math.floor(pwidth/Theme.tickpixels), 10)
    self.inc, self.majortick = nicenum(self.domain/divs)
    self.dx = self.inc/Theme.tickpixels
    self.yaxis:setOrigin(self.yaxis.origin or 0.1*pwidth)
    self.xaxis:setOrigin(self.xaxis.origin or self.yaxis.origin)
end


function Graph:zoom(factor)
    self.domain = self.domain * factor
    document.markChanged()
    self:resize()
    platform.window:invalidate()
end


function Graph:zoomin()
    self:zoom(1/self.factor)
end


function Graph:zoomout()
    self:zoom(self.factor)
end



-- Returns the value of x at pixel coordinate px
function Graph:tox(px)
    return (px - self.yaxis.origin)*self.dx
end


-- Returns the pixel coordinate px at value of x
function Graph:topx(x)
    return math.floor(x/self.dx + 0.5) + self.yaxis.origin
end


-- Returns the pixel coordinate py at value of y
function Graph:topy(y)
    return platform.window:height() - math.floor(y/self.dx + 0.5) - self.xaxis.origin
end


function Graph:trackMouse(wx, wy)
    self.xaxis:setOrigin(platform.window:height() - wy)
    self.yaxis:setOrigin(wx)
    platform.window:invalidate()
end


function Graph:paint(gc)
    self.paper:paint(gc)
    self.xaxis:paint(gc)
    self.yaxis:paint(gc)
    self.riemann:paint(gc)
    self.func:paint(gc)
    self.lhandle:paint(gc)
    self.rhandle:paint(gc)

    local wx = platform.window:width()
    local wy = platform.window:height()
    self.zoomInButton:setxy(wx-64, wy-16)
    self.zoomInButton:paint(gc)
    self.zoomOutButton:setxy(wx-32, wy-16)
    self.zoomOutButton:paint(gc)
    selectRectButtonGroup:paint(gc)
end


function Graph:help()
    addView(AboutView())
end


function Graph:mouseDown(wx, wy)
    local obj = MouseHit.findHit(wx, wy)

    local ox = obj:getpx() - wx
    local oy = platform.window:height() - wy - obj:getpy()

    obj:setTracking(true)
    mouseTracking = function(wx, wy) obj:trackMouse(wx + ox, wy + oy) end

    on.mouseUp = function(wx, wy)
        obj:setTracking(false)
        mouseTracking = nil
        on.mouseUp    = nil
        theCursor = "pointer"
        cursor.set(theCursor)
        platform.window:invalidate()
    end
end


function Graph:grabDown(wx, wy)
    MouseHit.findHit(wx, wy):grab()
end


function Graph:charIn(ch)
    if ch == "i" then
        self:zoomin()
    elseif ch == "o" then
        self:zoomout()
    elseif ch == "l" then
        theLeftButton:select()
    elseif ch == "r" then
        theRightButton:select()
    elseif ch == "m" then
        theMidButton:select()
    elseif ch == "t" then
        theTrapButton:select()
    elseif ch == "-" then
        self:decr()
    elseif ch == "+" then
        self:incr()
    end
end



theGraph = Graph()
addView(theGraph)






---------------------------------------------------------------------- Tool palette

toolpalette.register {
    {"Rectangle",
        {"Left", function() theLeftButton:select() end},
        {"Right", function() theRightButton:select() end},
        {"Mid", function() theMidButton:select() end},
        {"Trapezoid", function() theTrapButton:select() end},
        "-",
        {"More rectangles", function() theGraph:incr() end},
        {"Fewer rectangles", function() theGraph:decr() end},
    },
    {"Zoom",
        {"In", function() theGraph:zoomin() end},
        {"Out", function() theGraph:zoomout() end},
        "-",
        {"Zoom factor " .. MULTIPLY .. "2", function() theGraph:setFactor(2) end},
        {"Zoom factor " .. MULTIPLY .. "3", function() theGraph:setFactor(3) end},
        {"Zoom factor " .. MULTIPLY .. "5", function() theGraph:setFactor(5) end},
        {"Zoom factor " .. MULTIPLY .. "10", function() theGraph:setFactor(10) end},
    }
}






---------------------------------------------------------------------- Event Handlers

function on.paint(gc)
    paintViews(gc)
end


function on.resize(w, h)
    closeTopView()
    currentView():resize()
end


function on.mouseDown(wx, wy)
    currentView():mouseDown(wx, wy)
end


function on.grabDown(wx, wy)
    currentView():grabDown(wx, wy)
end


function on.mouseMove(wx, wy)
    cursor.set(theCursor)
    cursor.show()
    if mouseTracking then
        mouseTracking(wx, wy)
    end
end


function on.charIn(ch)
    currentView():charIn(ch)
end


function on.help()
    currentView():help()
end


function on.escapeKey()
    currentView():escapeKey()
end


function on.create()
    if not var.recallstr("f1") then
        math.eval("define f1(x)=x^2")
    end
    var.monitor("f1")
    cursor.show()
end


function on.varChange(varlist)
    if not var.recallstr("f1") then
        return -3
    end
    for _, varname in ipairs(varlist) do
        if varname == "f1" then
            theGraph.func:flushCache()
            platform.window:invalidate()
            return 0
        end
    end
    return 0
end


function on.save()
    local g = theGraph
    return {
        domain  = g.domain,
        f1      = var.recallstr("f1"),
        divs    = g.riemann.divs,
        left    = g.lhandle.x,
        right   = g.rhandle.x,
        ox      = g.yaxis.origin,
        oy      = g.xaxis.origin,
    }
end


function on.restore(state)
    local g = theGraph
    g.domain        = state.domain
    g.riemann.divs  = state.divs
    g.lhandle.x     = state.left
    g.rhandle.x     = state.right
    g.yaxis.origin  = state.ox
    g.xaxis.origin  = state.oy
    math.eval("define f1(x)=" .. g.f1)
end
